package org.icroco.tablemodel.renderer;

import java.awt.Component;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;

/**
 * A {@link javax.swing.TableCellRenderer} that is constructed with an array of {@link IRendererStage}s that are consulted in sequence to
 * renderer a table cell. Can accept a delegate.
 * 
 * @author clalleme - 20 avr. 2009
 */
public final class PipelineTableCellRenderer
    implements TableCellRenderer
{
    private static final long         serialVersionUID = -5171942854128414154L;
    
    private IRendererStage[]          modifiers;
    protected final TableCellRenderer delegate;
    
    /**
     * Creates a {@link PipelineTableCellRenderer} that uses the modifiers as it's pipeline.
     * 
     * @todo Add a constructor that takes an array of RenderStages and a TableCellRenderer that is the source of the pipeline.
     * @param aModifiers
     *            The {@link IRendererStage} pipeline.
     */
    public PipelineTableCellRenderer(final TableCellRenderer aDelegate, final IRendererStage... aStages)
    {
        this.delegate = aDelegate == null ? new DefaultTableCellRenderer() : aDelegate;
        final List<IRendererStage> stages = new ArrayList<IRendererStage>(aStages.length);
        addModifiers(stages, aStages);
        this.modifiers = stages.toArray(new IRendererStage[0]);
        if (aDelegate instanceof Component)
        {
            for (int i = 0; i < this.modifiers.length; i++)
            {
                this.modifiers[i].initRenderer(null, (Component) this.delegate);
            }
        }
    }
    
    public PipelineTableCellRenderer(final IRendererStage... aStages)
    {
        this(null, aStages);
    }
    
    private final void addModifiers(final List<IRendererStage> aResult, final IRendererStage[] aStages)
    {
        for (int i = 0; i < aStages.length; i++)
            aResult.add(aStages[i]);
    }
    
    /**
     * Creates a {@link PipelineTableCellRenderer} that has only one {@link IRendererStage} in the pipeline.
     * 
     * @todo Add a constructor that takes a RenderStage and a TableCellRenderer that is the source of the pipeline.
     * @param stage
     *            The single stage in the pipeline.
     */
    public PipelineTableCellRenderer(final IRendererStage stage)
    {
        this(new IRendererStage[] { stage });
    }
    
    /**
     * addStage.
     * 
     * @param aStage
     */
     public final void addStage(final IRendererStage aStage)
    {
        if (aStage != null)
        {
            IRendererStage[] newModifiers = Arrays.copyOf(this.modifiers, this.modifiers.length+1);
            newModifiers[this.modifiers.length] = aStage;
            this.modifiers = newModifiers;
        }
    }
    /**
     * (non-Javadoc)
     * 
     * @see javax.swing.table.DefaultTableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean,
     *      int, int)
     */
    @Override
    public Component getTableCellRendererComponent(final JTable aTable, final Object aValue, final boolean aIsSelected,
                                                   final boolean aHasFocus, final int aRow, final int aColumn)
    {
        Object value = aValue;
        for (int i = 0; i < this.modifiers.length; i++)
        {
            value = this.modifiers[i].preRender(aTable, value, aIsSelected, aHasFocus, aRow, aColumn);
        }
        
        if (this.delegate instanceof DefaultTableCellRenderer)
        {
            ((DefaultTableCellRenderer) this.delegate).setBackground(null);
            ((DefaultTableCellRenderer) this.delegate).setForeground(null);
        }
        final JComponent rendered = (JComponent) this.delegate.getTableCellRendererComponent(aTable,
                                                                                             value,
                                                                                             aIsSelected,
                                                                                             aHasFocus,
                                                                                             aRow,
                                                                                             aColumn);
        
        for (int i = 0; i < this.modifiers.length; i++)
        {
            this.modifiers[i].render(rendered, aTable, value, aIsSelected, aHasFocus, aRow, aColumn);
        }
        
        return rendered;
    }
    
    /**
     * Returns the modifiers.
     * 
     * @return The modifiers to return.
     */
    public final IRendererStage[] getModifiers()
    {
        return this.modifiers;
    }
    
    /**
     * (non-Javadoc)
     * 
     * @see java.awt.Component#toString()
     */
    @Override
    public String toString()
    {
        StringBuilder buf = new StringBuilder(50);
        for (IRendererStage st : this.modifiers)
            buf.append(st.getClass().getSimpleName() + " -> ");
        
        return buf.toString();
    }
    
}