/**
 * Renderpack Pipeline
 * http://renderpack.dev.java.net/
 *
 * COPYRIGHT 2004 Edward Dale
 */
package org.icroco.tablemodel.renderer.plugin;

import java.awt.Color;
import java.awt.Component;
import java.util.Map;

import javax.swing.JTable;
import javax.swing.UIManager;

import org.icroco.tablemodel.renderer.IRendererStage;

/**
 * A {@link IRendererStage} that add stripes to alernating rows. This makes the rows look more like a ledger.
 * 
 * @author clalleme - 20 avr. 2009
 * 
 */
public final class StripedRenderStage
    implements IRendererStage
{
    public static final Color GENERIC_GRAY = new Color(229, 229, 229);
    
    // private static final double DARKER_COEF = 0.9;
    
    /** The {@link Color} for even rows. */
    private Color             evenRowsColor;
    
    /** The {@link Color} for odd rows. */
    private Color             oddRowsColor;
    
    public StripedRenderStage()
    {
		Color color = UIManager.getColor("Table.alternateRowColor"); // SwingX use:
		// "UIColorHighlighter.stripingBackground"
        if (color == null)
            color = GENERIC_GRAY;
        this.evenRowsColor = color;
        this.oddRowsColor = null;
    }
    
    /**
     * Creates a {@link StripedRenderStage} with the specified row colors.
     * 
     * @param aOddRowsColor
     *            The {@link Color} for odd rows.
     * @param aEvenRowsColor
     *            The {@link Color} for even rows.
     */
    public StripedRenderStage(final Color aOddRowsColor, final Color aEvenRowsColor)
    {
        this();
        this.oddRowsColor = aOddRowsColor;
        this.evenRowsColor = aEvenRowsColor;
    }
    
    /**
	 * aParameters can contains 2 kind of sub-string. Possible value for each parameters is:
	 * <ul>
	 * <li><b>oddBg</b>=<i>color_definition</i> is optional: in this case we try to get like this: <code>UIManager.getColor("Table.alternateRowColor");</code></li>
	 * <li><b>evenBg</b>=<i>color_definition</i> is optional</li>
	 * </ul>
	 * <b><i>color_definition</b></i> is define as:
     * <ul>
     * <li>Constant color as define into {@link Color}. For example: RED.</li>
     * <li>Or a web encoded value like #FF9999 (yellow light)</li>
     * </ul>
     * 
     * @throws NoSuchFieldException
     * @throws IllegalAccessException
     * @throws SecurityException
     * @throws IllegalArgumentException
     * @throws NumberFormatException
     * 
     * @see org.icroco.tablemodel.renderer.IRendererStage#init(java.lang.String)
     */
	public StripedRenderStage(String aParameters) throws NumberFormatException, IllegalArgumentException, SecurityException,
			IllegalAccessException, NoSuchFieldException
    {
        this();
		final Map<String, String> res = ParametersUtil.parseValue(aParameters);
		final String odd = res.get("oddBg");
		final String even = res.get("evenBg");

		if (odd != null)
			this.oddRowsColor = ParametersUtil.decodeColor(odd);
		if (even != null)
			this.evenRowsColor = ParametersUtil.decodeColor(even);
		else
			this.evenRowsColor = null;
    }
    
    @Override
    public void initRenderer(Class<?> aColumnClass, Component aRendered)
    {
        // NOTHING
    }
    
    @Override
    public Object preRender(JTable aTable, Object aValue, boolean aIsSelected, boolean aHasFocus, int aRow, int aColumn)
    {
        return aValue;
    }
    
    /**
     * (non-Javadoc)
     * 
	 * @see IRendererStage.ft.archi.common.swing.pipeline.RenderStage#render(javax.swing.JTable, java.awt.Component,
	 *      java.lang.Object, boolean, boolean, int, int)
     */
    public void render(Component aRendered, JTable aTable, Object aValue, boolean aIsSelected, boolean aHasFocus, int aRow, int aColumn)
    {
        if (aRow % 2 == 0)
        {
            aRendered.setBackground(aIsSelected ? darker(this.evenRowsColor) : this.evenRowsColor);
        }
        else if (this.oddRowsColor != null)
        {
            aRendered.setBackground(aIsSelected ? darker(this.oddRowsColor) : this.oddRowsColor);
        }
    }
    
    private final Color darker(final Color aColor)
    {
		return aColor == null ? null : aColor.darker();
        
		// return new Color(Math.max((int) (aColor.getRed() * DARKER_COEF), 0), Math.max((int) (aColor.getGreen() *
		// DARKER_COEF), 0), Math
        // .max((int) (aColor.getBlue() * DARKER_COEF), 0));
        
    }
}
