package org.icroco.tablemodel;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.swing.JTable;
import javax.swing.table.TableModel;

/**
 * IBeanTableModel.
 * 
 * @author ouaibsky - 22 avr. 2009
 * @param <M>
 * 
 */
public interface IBeanTableModel<M>
		extends TableModel
{

	/**
	 * Can be used to to add a new value into this model. Be careful we put
	 * aValue into a {@link Map} as key, so we use {@link Object#hashCode()} and
	 * {@link Object#equals(Object)}.<br/>
	 * This method should not be called from EDT.
	 * 
	 * @param aValue
	 * @return true if add with success, else return false.
	 */
	public boolean add(final M aValue);

	/**
	 * This method should not be called from EDT.
	 * 
	 * @param aValue
	 * @return the old value if existing.
	 */
	public M deleteRow(final M aValue);

	/**
	 * This method should not be called from EDT.
	 * 
	 * @param aIdx
	 * @return the old value if existing or null if not found or aIdx greater
	 *         than model size..
	 */
	public M deleteRowAt(final int aIdx);

	/**
	 * used to delete q non continuous range of row.
	 * 
	 * @param aIdx represents index in the model (not the view).
	 * @return
	 */
	public int[] deleteRows(final int... aIdx);

	/**
	 * Delete continuous range of index.
	 * 
	 * @param aStartIdx
	 *            inclusive.
	 * @param aEndIdx
	 *            inclusive.
	 * @return an array of effective removed index. Could be less than aEnnIdx -
	 *         aStartIdx.
	 */
	public int deleteRows(final int aStartIdx, final int aEndIdx);

	/**
	 * find the index og the given M key (use hashcode and equals).
	 * 
	 * @param aValue
	 * @return model index (not view).
	 */
	public int getIdxFor(final M aValue);

	/**
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	public int getRowCount();

	/**
	 * getValueAt return the M Object a the given index.
	 * 
	 * @param aRowIndex
	 * @return
	 */
	public M getValueAt(final int aRowIndex);

	/**
	 * getValueAt return the M Object a index which math the key paremeter (hashCode and equals require.
	 * 
	 * @param aKey
	 * @return
	 */
	public M getValueAt(final M aKey);

	/**
	 * getValues return a unmodifiable {@link List} of underlying model. <b>Be careful</b>, we just copy the container and not all objects. </br>
	 * It means if M is mutable or not thread safe, you could display wrong data into your {@link JTable} if you uppadte datas out of the model.
	 * 
	 * @see Collections#unmodifiableList(List)
	 * 
	 * @return
	 */
	public List<M> getValues();
	
	/**
	 * clear the model and refresh the table.
	 */
	public void reset();

	/**
	 * clear the model, insert the given collection and make a full refresh of
	 * the table.
	 * 
	 * @param collection
	 */
	public void reset(final Collection<M> collection);
    
    /**
     * setValueAtFromModel is the same thing as {@link #setValueAt(Object, int, int)} but should occurs only out of EDT. It's important
     * because in this case we make some stuff like detect if cell is visible in view.<br/>
     * Can throw a {@link IllegalStateException} if no setter was found. We assume we are on developer side and it know how the bean was
     * construct.
     * 
     * @param aValue
     * @param aRowIndex
     * @param aColumnIndex
     *            as define into your annotation order.
     * 
     * 
     */
	public M updateRow(final Object aValue, final int aRowIndex, final int aColumnIndex) throws IllegalStateException;

    /**
     * See {@link #updateRow(Object, int, int)}.
     * 
     * @param aValue represents the value to be updated (means a cell value).
     * @param aRowKey, can be original value or another key object.
     * @param aCol represents order of this column as define into your order key into {@link ABeanTableModelProperty}.
     */
    public void updateRow(final Object aValue, final M aRowKey, final int aCol) throws IllegalStateException;

	/**
	 * updateRow can be use to update (or add if not exist) aValue.<br/>
	 * This method should not be called from EDT.
	 * 
	 * @param aValue
	 * @return true if update with success, else return false.
	 */
	public boolean addOrUpdateRow(final M aValue);

}