package org.icroco.tablemodel.impl;

import java.lang.reflect.Method;

import org.icroco.tablemodel.BeanTableModelException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An helper to get getter/setter complaint with JabaBean naming convention.
 * 
 */
public final class BeanHelper
{
    /**
     * Logger for this class
     */
    private transient static Logger logger = LoggerFactory.getLogger(BeanHelper.class);
    
    private BeanHelper()
    {
        super();
    }
    
    /**
     * 
     */
    private static final String START_STR  = "^";
    /**
     * 
     */
    private static final String EMPTY_STR  = "";
    /**
     * 
     */
    private static final String SET_PREFIX = "set"; //$NON-NLS-1$
    /**
     * 
     */
    private static final String GET_PREFIX = "get"; //$NON-NLS-1$
    /**
     * 
     */
    private static final String IS_PREFIX  = "is"; //$NON-NLS-1$
                                                    
    public static boolean isGetter(final Method aMethod) throws BeanTableModelException
    {
        final String name = aMethod.getName();
        if (name.startsWith(GET_PREFIX) || name.startsWith(IS_PREFIX))
        {
            if (aMethod.isVarArgs() || aMethod.getParameterTypes().length > 0)
            {
                throw new BeanTableModelException("Getter based on bean naming convemtion must have 0 parameters, method: " + aMethod);
            }
            if (aMethod.getReturnType().isPrimitive())
            {
                throw new BeanTableModelException("Getter need to have non-primitive as return value" + aMethod);
            }
            return true;
        }
        else
        {
            throw new BeanTableModelException("You have to annotate getter based on bean naming convention: need to start by 'get' or 'is' prefix. method: "
                    + aMethod);
        }
    }
    
    public static boolean matchGetter(final Method aMethod)
    {
        final String name = aMethod.getName();
        if (name.startsWith(GET_PREFIX) || name.startsWith(IS_PREFIX))
        {
            if (name.equals("getClass") && aMethod.getParameterTypes().length == 0)
            {
                return false;
            }
            return true;
        }
        return false;
    }
    
    public static final Method checkSetter(final Class<?> aClazz, final Method aGetter) throws BeanTableModelException
    {
        String baseName = aGetter.getName();
        if (baseName.startsWith(IS_PREFIX))
        {
            baseName = baseName.replaceFirst(IS_PREFIX, "");
        }
        else if (baseName.startsWith(GET_PREFIX))
        {
            baseName = baseName.replaceFirst(GET_PREFIX, "");
        }
        
        final Class<?> propType = aGetter.getReturnType();
        Method setter = null;
        try
        {
            setter = aClazz.getMethod(SET_PREFIX + baseName, propType);
        }
        catch (final SecurityException e)
        {
            throw new BeanTableModelException("Failed to find setter: set" + baseName + "(" + propType.getName() + ")", e);
        }
        catch (final NoSuchMethodException e)
        {
            // OK, we throw exception later.
            // throw new BeanTableModelException("Failed to find setter: set" + baseName + "(" + propType.getName()
            // + ") , check if parameter type is not a primitive type", e);
            // TODO replace by a generic setter wich dump method call.
        }
        
        return setter;
    }
    
    public static final Method getSetter(final Class<?> aClazz, final Method aGetter, final Class<?> aParameters)
    {
        String baseName = aGetter.getName();
        if (baseName.startsWith(IS_PREFIX))
        {
            baseName = baseName.replaceFirst(IS_PREFIX, "");
        }
        else if (baseName.startsWith(GET_PREFIX))
        {
            baseName = baseName.replaceFirst(GET_PREFIX, "");
        }
        
        try
        {
            return aClazz.getMethod(SET_PREFIX + baseName, aParameters);
        }
        catch (final Exception aException)
        {
            logger.debug("failed to find setter for getter '" + aGetter + "' and args type '" + aParameters + "'");
            return null;
        }
    }
    
    /**
     * @param aMethod
     * @return
     */
    public static boolean isSetter(final Method aMethod)
    {
        boolean lReturn = false;
        
        if (aMethod.getName().startsWith(SET_PREFIX))
        {
            lReturn = true;
        }
        
        return lReturn;
        
    }
    
    /**
     * This method
     * 
     * @param aMethod
     * @return
     */
    public static String getBeanName(final Method aMethod)
    {
        String lReturn = null;
        
        if (aMethod.getName().startsWith(IS_PREFIX))
        {
            lReturn = aMethod.getName().replaceFirst(START_STR + IS_PREFIX, EMPTY_STR);
        }
        else if (aMethod.getName().startsWith(GET_PREFIX))
        {
            lReturn = aMethod.getName().replaceFirst(START_STR + GET_PREFIX, EMPTY_STR);
        }
        else if (aMethod.getName().startsWith(SET_PREFIX))
        {
            lReturn = aMethod.getName().replaceFirst(START_STR + SET_PREFIX, EMPTY_STR);
        }
        
        return lReturn;
    }
}
