package org.icroco.tablemodel.impl;

import java.awt.EventQueue;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;

import org.icroco.tablemodel.ABeanTableModel;
import org.icroco.tablemodel.ABeanTableModelProperty;
import org.icroco.tablemodel.BeanTableModelException;
import org.icroco.tablemodel.IBeanTableModel;
import org.icroco.tablemodel.renderer.ARendererStages;
import org.icroco.tablemodel.renderer.IRendererStage;
import org.icroco.tablemodel.renderer.PipelineTableCellRenderer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class wrap a benaobject into a JTable. Do not use directly this class, in place use the helper
 * {@link BeanTableModelHelper}
 * 
 * @see BeanTableModelHelper
 * @version $Revision: 1.8 $
 * 
 * @param <M>
 */
public class BeanTableModel<M>
		extends AbstractTableModel
		implements IBeanTableModel<M>
{
	/**
     * 
     */
	private static final int						BUF_SIZE			= 100;

	/**
	 * Logger for this class
	 */
	private transient static Logger					logger				= LoggerFactory.getLogger(BeanTableModel.class);

	/**
     * 
     */
	private static final int						DEFAULT_LIST_SIZE	= 50;
	/**
     * 
     */
	private static final long						serialVersionUID	= 1L;
	final private ArrayList<M>						array;
	final private HashMap<M, Integer>				arrayMap;
	private final ReadWriteLock						lock;
	private final Class<M>							clazz;
	protected TableModelProperty					properties[];
	private static final transient MessageFormat	ERROR_1				= new MessageFormat(Messages.getString("BeanTableModel.1"));	//$NON-NLS-1$
	private static final transient MessageFormat	ERROR_2				= new MessageFormat(Messages.getString("BeanTableModel.2"));	//$NON-NLS-1$
	private static final transient MessageFormat	ERROR_3				= new MessageFormat(Messages.getString("BeanTableModel.3"));	//$NON-NLS-1$
	//	private static final transient MessageFormat ERROR_4 = new MessageFormat(Messages.getString("BeanTableModel.4")); //$NON-NLS-1$
	//	private static final transient MessageFormat ERROR_5 = new MessageFormat(Messages.getString("BeanTableModel.5")); //$NON-NLS-1$
	//	private static final transient MessageFormat ERROR_6 = new MessageFormat(Messages.getString("BeanTableModel.6")); //$NON-NLS-1$
	//	private static final transient MessageFormat ERROR_7 = new MessageFormat(Messages.getString("BeanTableModel.7")); //$NON-NLS-1$
	//	private static final transient MessageFormat ERROR_8 = new MessageFormat(Messages.getString("BeanTableModel.8")); //$NON-NLS-1$
	private static final transient MessageFormat	ERROR_9				= new MessageFormat(Messages.getString("BeanTableModel.9"));	//$NON-NLS-1$
	private static final transient MessageFormat	ERROR_10			= new MessageFormat(Messages.getString("BeanTableModel.10"));	//$NON-NLS-1$
	private static final transient MessageFormat	ERROR_12			= new MessageFormat(Messages.getString("BeanTableModel.12"));	//$NON-NLS-1$
	private static final transient MessageFormat	ERROR_13			= new MessageFormat(Messages.getString("BeanTableModel.13"));	//$NON-NLS-1$
	private static final transient MessageFormat	ERROR_14			= new MessageFormat(Messages.getString("BeanTableModel.14"));	//$NON-NLS-1$

	// // Flashing
	// private transient JTable table;
	// private final transient Rectangle cacheRectangle = new Rectangle();
	private final List<IRendererStage>				globalRenderers		= new ArrayList<IRendererStage>();
	final ABeanTableModel							options;

	/**
	 * Constructor.
	 * 
	 * @param aClass
	 * @throws BeanTableModelException
	 */
	public BeanTableModel(final Class<M> aClass) throws BeanTableModelException
	{
		super();
		if (aClass == null)
		{
			throw new BeanTableModelException(Messages.getString("BeanTableModel.0")); //$NON-NLS-1$
		}
		this.clazz = aClass;
		this.array = new ArrayList<M>(DEFAULT_LIST_SIZE);
		this.arrayMap = new HashMap<M, Integer>();
		this.options = aClass.getAnnotation(ABeanTableModel.class);
		// we do that to get more performance ...
		this.lock = this.options != null && this.options.isTreadSafe() ? new ReentrantReadWriteLock() : new EmptyLock();
	}

	public void parseAnnotation() throws BeanTableModelException
	{
		try
		{
			readBTMAnnotationType(this.clazz);
			final List<TableModelProperty> lProperties = readBTMAnnotationMethod(this.clazz);
			Collections.sort(lProperties);
			checkSetter(lProperties);
			this.properties = lProperties.toArray(new TableModelProperty[lProperties.size() > 0 ? lProperties.size() - 1 : 0]);

			// dump info at end for debugging
			if (logger.isDebugEnabled())
			{
				dumpProperties(this, false);
			}
		}
		catch (final BeanTableModelException exception)
		{
			dumpProperties(this, true);
			throw exception;
		}
	}

	protected void readBTMAnnotationType(final Class<M> clazz) throws BeanTableModelException
	{
		final ABeanTableModel annotation = clazz.getAnnotation(ABeanTableModel.class);

		if (annotation != null)
		{
			if (annotation.excludeGetters().trim().length() > 0 && annotation.autoAddAllGetters())
				logger.warn("ABeanTableModel set on '" + clazz.getName() + "' but autoAddAllGetters is disable");
		}

		final ARendererStages stages = clazz.getAnnotation(ARendererStages.class);

		if (stages != null)
		{
			final String[] constructorArgs = stages.stagesParameters();
			final Class<? extends IRendererStage>[] clazzStg = stages.stages();
			int i = 0;
			try
			{

				for (; i < clazzStg.length; i++)
				{
					this.globalRenderers.add(clazzStg[i].getConstructor(String.class)
							.newInstance(i < constructorArgs.length ? constructorArgs[i] : ""));
				}
			}
			catch (final Exception exception)
			{
				throw new BeanTableModelException("Failed to install global Renderer: "
						+ ERROR_13.format(new Object[] { clazzStg[i].getName() }), exception);
			}
		}
	}

	protected TableModelProperty createNewProperty()
	{
		return new TableModelProperty();
	}

	protected List<TableModelProperty> readBTMAnnotationMethod(final Class<M> aClazz) throws BeanTableModelException
	{
		final Method methods[] = aClazz.getMethods();
		final List<TableModelProperty> lProperties = new ArrayList<TableModelProperty>(methods.length);
		final TableModelProperty search = createNewProperty();
		ABeanTableModelProperty aProp;
		String found;
		int idx;
		final String[] excludeGetters = this.options == null ? new String[0] : this.options.excludeGetters().split(",");
		for (int i = 0; i < excludeGetters.length; i++)
			excludeGetters[i] = excludeGetters[i].trim();
		Arrays.sort(excludeGetters);

		try
		{
			for (final Method method : methods)
			{
				aProp = method.getAnnotation(ABeanTableModelProperty.class);
				if (isAutoDiscoverGetter()) // first we treat auto discover (and can be override by special annotation
				{
					if (BeanHelper.matchGetter(method) && Arrays.binarySearch(excludeGetters, method.getName()) < 0)
					{
						found = BeanHelper.getBeanName(method);
						search.name = found;
						idx = lProperties.indexOf(search);
						final TableModelProperty prop = idx < 0 ? createNewProperty() : lProperties.get(idx);
						if (idx < 0 || equalsMethod(lProperties.get(idx).getter, method)) // not found or alreday
						// declared
						{
							prop.name = found;
							prop.label = found; // TODO: read from resource file
							prop.isEditable = false;
							prop.columnClass = method.getReturnType();
							prop.orderKey = found;
							prop.getter = method;
							if (idx < 0)
								lProperties.add(prop);
							readRSAnnotationMethod(prop, method);
						}
						else
						{
							throw new BeanTableModelException(ERROR_1.format(new Object[] { found }));
						}
					}
				}
				if (aProp != null)
				{
					if (BeanHelper.isGetter(method))
					{
						found = aProp.name() == null ? BeanHelper.getBeanName(method) : aProp.name();
						search.name = found;
						idx = lProperties.indexOf(search);
						final TableModelProperty prop = idx < 0 ? createNewProperty() : lProperties.get(idx);
						if (idx < 0 || equalsMethod(lProperties.get(idx).getter, method)) // not found
						{
							prop.name = found;
							prop.label = found; // TODO: read from resource file
							prop.isEditable = aProp.isEditable();
							prop.columnClass = method.getReturnType();
							prop.orderKey = aProp.orderKey();
							prop.getter = method;
							final Method setter = BeanHelper.checkSetter(aClazz, method);
							if (setter != null)
							{
								prop.setter = setter;
							}
							if (idx < 0)
								lProperties.add(prop);
							// read renderer
							readRSAnnotationMethod(prop, method);
						}
						else
						{
							throw new BeanTableModelException(ERROR_1.format(new Object[] { found }));
						}
					}
					else
					{
						throw new BeanTableModelException(ERROR_12.format(new Object[] {}));
					}
				}
			}
		}
		catch (final BeanTableModelException exception)
		{
			throw exception;
		}
		finally
		{
			Collections.sort(lProperties);
			this.properties = lProperties.toArray(new TableModelProperty[lProperties.size() > 0 ? lProperties.size() - 1 : 0]);
		}

		return lProperties;
	}

	private final static boolean equalsMethod(final Method aMethod1, final Method aMethod2)
	{
		if (!aMethod1.getName().equals(aMethod2.getName()))
			return false;
		return Arrays.equals(aMethod1.getParameterTypes(), aMethod2.getParameterTypes());
	}

	private boolean isAutoDiscoverGetter()
	{
		return this.options != null && this.options.autoAddAllGetters();
	}

	private final void readRSAnnotationMethod(final TableModelProperty aProp, final Method aMethod)
	{
		final ARendererStages stages = aMethod.getAnnotation(ARendererStages.class);
		final List<IRendererStage> res = new ArrayList<IRendererStage>(this.globalRenderers);
		if (stages != null)
		{
			try
			{
				if (!stages.tableCellRenderer().equals(TableCellRenderer.class))
				{
					aProp.tableRenderer = stages.tableCellRenderer().newInstance();
				}
			}
			catch (Exception aException)
			{
				throw new BeanTableModelException(ERROR_14.format(new Object[] { stages.tableCellRenderer() }), aException);
			}
			try
			{
				final String[] constructorArgs = stages.stagesParameters();
				final Class<? extends IRendererStage>[] clazzStg = stages.stages();

				for (int i = 0; i < clazzStg.length; i++)
				{
					res.add(clazzStg[i].getConstructor(String.class).newInstance(i < constructorArgs.length ? constructorArgs[i] : ""));
				}
			}
			catch (final Exception exception)
			{
				throw new BeanTableModelException(ERROR_13.format(new Object[] { aProp.toString() }), exception);
			}
		}
		aProp.renderers = res.toArray(new IRendererStage[0]);
	}

	public void installRenderer(final JTable aTable)
	{
		if (aTable == null)
		{
			logger.debug("installRenderer a null JTable parameter, we skip installation.");
			return;
		}
		int i = 0;
		for (final TableModelProperty prop : this.properties)
		{
			if (prop.tableRenderer != null)
			{
				aTable.getColumnModel().getColumn(i).setCellRenderer(prop.tableRenderer);
			}

			if (prop.renderers != null && prop.renderers.length > 0)
			{
				TableCellRenderer oldRenderer = null;
				oldRenderer = aTable.getCellRenderer(0, i);
				if (oldRenderer == null)
				{
					oldRenderer = aTable.getDefaultRenderer(this.getColumnClass(i));
				}
				aTable.getColumnModel().getColumn(i)
						.setCellRenderer(new PipelineTableCellRenderer(prop.tableRenderer != null ? prop.tableRenderer
								: cloneExisitingRenderer(oldRenderer), prop.renderers));
			}
			i++;
		}
	}

	/**
	 * cloneExisitingRenderer is used to clone render to prevent existing UI which share renderer between column. To do
	 * this we try in order to
	 * <ul>
	 * <li>create a new instance with a public empty constructor.</li>
	 * <li>If failed try to change visibility of this constructor with reflection</li>
	 * <li>If failed again, return this renderer (and possibility share)</li>
	 * </ul>
	 * 
	 * @param aOldRenderer
	 * @return
	 */
	protected final TableCellRenderer cloneExisitingRenderer(final TableCellRenderer aOldRenderer)
	{
		TableCellRenderer value = aOldRenderer;
		if (aOldRenderer == null)
			return new DefaultTableCellRenderer();
		try
		{
			value = aOldRenderer.getClass().newInstance();
		}
		catch (final InstantiationException aException)
		{
			logger.warn("Technical error: Failed to instanciate " + aOldRenderer.getClass().getName() + " we return current instance: "
					+ aOldRenderer.hashCode(), aException);

		}
		catch (final IllegalAccessException aException)
		{
			// aException.printStackTrace();
			try
			{
				final Constructor<?> constructor = aOldRenderer.getClass().getConstructor();
				constructor.setAccessible(true);
				value = (TableCellRenderer) constructor.newInstance();
			}
			catch (final Throwable aException1)
			{
				logger.warn("Technical error: Failed to change constructor visibilty " + aOldRenderer.getClass().getName()
						+ " we return current instance: " + aOldRenderer.hashCode(), aException);
				value = aOldRenderer;
			}
		}

		return value;
	}

	/**
	 * @param aClazz
	 * @param aProperties
	 * @throws BeanTableModelException
	 */
	private void checkSetter(final List<TableModelProperty> aProperties) throws BeanTableModelException
	{
		for (final TableModelProperty prop : aProperties)
		{
			if (prop.getter == null)
			{
				throw new BeanTableModelException(ERROR_3.format(new Object[] { prop.name }));
			}
			if (prop.isEditable && prop.setter == null)
			{
				throw new BeanTableModelException(ERROR_2.format(new Object[] { prop.name, prop.getter.getName() }));
			}
		}
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	@Override
	public int getColumnCount()
	{
		return this.properties.length;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	@Override
	public int getRowCount()
	{
		return this.array.size();
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	@Override
	public Object getValueAt(final int rowIndex, final int columnIndex)
	{
		Object lReturn = null;
		try
		{
			this.lock.readLock().lock();
			lReturn = this.properties[columnIndex].getter.invoke(this.array.get(rowIndex), (Object[]) null);
		}
		catch (final IllegalArgumentException aException)
		{
			logger.error(ERROR_9.format(new Object[] {
					this.properties[columnIndex].name,
															Integer.valueOf(rowIndex),
															Integer.valueOf(columnIndex) }), aException);
		}
		catch (final IllegalAccessException aException)
		{
			logger.error(ERROR_9.format(new Object[] {
					this.properties[columnIndex].name,
															Integer.valueOf(rowIndex),
															Integer.valueOf(columnIndex) }), aException);
		}
		catch (final InvocationTargetException aException)
		{
			logger.error(ERROR_9.format(new Object[] {
					this.properties[columnIndex].name,
															Integer.valueOf(rowIndex),
															Integer.valueOf(columnIndex) }), aException);
		}
		finally
		{
			this.lock.readLock().unlock();
		}
		return lReturn;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
	 */
	@Override
	public Class<?> getColumnClass(final int aColumnIndex)
	{
		return this.properties[aColumnIndex].getter.getReturnType();
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.table.AbstractTableModel#getColumnName(int)
	 */
	@Override
	public String getColumnName(final int aColumn)
	{
		return this.properties[aColumn].label;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
	 */
	@Override
	public boolean isCellEditable(final int aRowIndex, final int aColumnIndex)
	{
		return this.properties[aColumnIndex].isEditable;
	}

	/**
	 * This method is called from EDT !!! no not call him from another thread.
	 * 
	 * @see javax.swing.table.AbstractTableModel#setValueAt(java.lang.Object, int, int)
	 */
	@SuppressWarnings("boxing")
	@Override
	public void setValueAt(final Object aValue, final int aRowIndex, final int aColumnIndex)
	{
		try
		{
			if (!EventQueue.isDispatchThread())
			{
				throw new IllegalAccessException("Can not call setValueAt from another thread that EDT. Calling thread is: "
						+ Thread.currentThread().getName());
			}

			this.lock.writeLock().lock();
			final int modelColIdx = aColumnIndex;// this.table.convertColumnIndexToModel(aColumnIndex);
			final int modelRowIdx = aRowIndex; // this.table.convertColumnIndexToModel(aRowIndex);
			final M toto = this.array.get(modelRowIdx);
			this.properties[modelColIdx].setter.invoke(toto, aValue);
			this.fireBeanTableCellUpdated(aRowIndex, aColumnIndex);
		}
		catch (final IllegalArgumentException aException)
		{
			logger.error(ERROR_10.format(new Object[] {
					this.properties[aRowIndex].name,
															aRowIndex,
															aColumnIndex,
															aValue,
															this.properties[aColumnIndex].columnClass.getSimpleName(),
															aValue.getClass().getSimpleName() }), aException);
		}
		catch (final IllegalAccessException aException)
		{
			logger.error(ERROR_10.format(new Object[] {
					this.properties[aRowIndex].name,
															aRowIndex,
															aColumnIndex,
															aValue,
															this.properties[aColumnIndex].columnClass.getSimpleName(),
															aValue.getClass().getSimpleName() }), aException);
		}
		catch (final InvocationTargetException aException)
		{
			logger.error(ERROR_10.format(new Object[] {
					this.properties[aRowIndex].name,
															aRowIndex,
															aColumnIndex,
															aValue,
															this.properties[aColumnIndex].columnClass.getSimpleName(),
															aValue.getClass().getSimpleName() }), aException);
		}
		catch (final NullPointerException aException)
		{
			logger.error(ERROR_10.format(new Object[] {
					this.properties[aRowIndex].name,
															aRowIndex,
															aColumnIndex,
															aValue,
															this.properties[aColumnIndex].columnClass.getSimpleName(),
															aValue.getClass().getSimpleName() }), aException);
		}
		finally
		{
			this.lock.writeLock().unlock();
		}
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#getValueAt(int)
	 */
	@Override
	public final M getValueAt(final int rowIndex)
	{
		try
		{
			this.lock.readLock().lock();
			if (rowIndex >= 0 && rowIndex < this.array.size())
			{
				return this.array.get(rowIndex);
			}
		}
		finally
		{
			this.lock.readLock().unlock();
		}

		return null;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#getIdxFor(java.lang.Object)
	 */
	@Override
	public final int getIdxFor(final M aValue)
	{
		try
		{
			this.lock.readLock().lock();
			return this.arrayMap.get(aValue).intValue();
		}
		finally
		{
			this.lock.readLock().unlock();
		}
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see tempo.impl.IBeanTableModel#getOld(M)
	 */
	public final M getValueAt(final M aRow)
	{
		final Integer value = Integer.valueOf(getIdxFor(aRow));
		return value == null ? null : this.getValueAt(value.intValue());
	}

	@Override
	public boolean add(final M aValue)
	{
		try
		{
			this.lock.writeLock().lock();
			if (this.arrayMap.containsKey(aValue))
			{
				return false;
			}
			if (this.array.add(aValue))
			{
				this.arrayMap.put(aValue, Integer.valueOf(this.array.size() - 1));
				fireBeanTableRowsInserted(this.array.size() - 1, this.array.size() - 1);
				return true;
			}
			else
			{
				logger.error("Failed to insert data at index:'" + this.array.size() + "', value: " + aValue);
				return false;
			}
		}
		finally
		{
			this.lock.writeLock().unlock();
		}
	}

	protected void fireBeanTableRowsInserted(final int aFirstRow, final int aLastRow)
	{
		fireTableRowsInserted(aFirstRow, aLastRow);
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#addOrUpdateRow(java.lang.Object)
	 */
	@Override
	public boolean addOrUpdateRow(final M aValue)
	{
		try
		{
			this.lock.writeLock().lock();
			final Integer idx = this.arrayMap.get(aValue);

			if (idx == null)
			{
				return this.add(aValue);
			}
			this.array.set(idx.intValue(), aValue);
			fireBeanTableRowsUpdated(idx.intValue(), idx.intValue());
			return true;
		}
		finally
		{
			this.lock.writeLock().unlock();
		}
	}

	/**
	 * fireBeanTableRowsUpdated.
	 * 
	 * @param aFirstRow
	 * @param aLastRow
	 */
	protected void fireBeanTableRowsUpdated(final int aFirstRow, final int aLastRow)
	{
		fireTableRowsUpdated(aFirstRow, aLastRow);
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#updateRow(java.lang.Object, java.lang.Object, int)
	 */
	@Override
	public void updateRow(final Object aValue, final M aRowKey, final int aCol)
	{
		try
		{
			this.lock.writeLock().lock();
			final Integer row = this.arrayMap.get(aRowKey);
			if (row != null)
			{
				this.updateRow(aValue, row.intValue(), aCol);
			}
			else
			{
				if (logger.isDebugEnabled())
				{
					logger.debug("Row not found: " + aRowKey);
				}
			}
		}
		finally
		{
			this.lock.writeLock().unlock();
		}
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#updateRow(java.lang.Object, int, int)
	 */
	@SuppressWarnings("boxing")
	@Override
	public final M updateRow(final Object aValue, final int aRowIndex, final int aColumnIndex)
	{
		try
		{
			this.lock.writeLock().lock();
			final M record = this.array.get(aRowIndex);
			if (this.properties[aColumnIndex].setter == null)
			{
				// we try to find another setter with right type.
				final Method setter = BeanHelper.getSetter(this.clazz, this.properties[aColumnIndex].getter, aValue.getClass());
				if (setter == null)
					throw new IllegalAccessException("Failed to find setter corresponding to getter "
							+ this.properties[aColumnIndex].getter.getName() + " , check if parameter type is not a primitive type");
				else
				{
					setter.invoke(record, aValue);
					this.fireBeanTableCellUpdated(aRowIndex, aColumnIndex);
				}
			}
			else
			{
				this.properties[aColumnIndex].setter.invoke(record, aValue);
				this.fireBeanTableCellUpdated(aRowIndex, aColumnIndex);
			}
			return record;
		}
		catch (final IllegalArgumentException aException)
		{
			logger.error(ERROR_10.format(new Object[] {
					this.properties[aColumnIndex].name,
															aRowIndex,
															aColumnIndex,
															aValue,
															this.properties[aColumnIndex].columnClass.getSimpleName(),
															aValue.getClass().getSimpleName() }), aException);
		}
		catch (final IllegalAccessException aException)
		{
			logger.error(ERROR_10.format(new Object[] {
					this.properties[aColumnIndex].name,
															aRowIndex,
															aColumnIndex,
															aValue,
															this.properties[aColumnIndex].columnClass.getSimpleName(),
															aValue.getClass().getSimpleName() }), aException);
		}
		catch (final InvocationTargetException aException)
		{
			logger.error(ERROR_10.format(new Object[] {
					this.properties[aColumnIndex].name,
															aRowIndex,
															aColumnIndex,
															aValue,
															this.properties[aColumnIndex].columnClass.getSimpleName(),
															aValue.getClass().getSimpleName() }), aException);
		}
		catch (final NullPointerException aException)
		{
			logger.error(ERROR_10.format(new Object[] {
					this.properties[aColumnIndex].name,
															aRowIndex,
															aColumnIndex,
															aValue,
															this.properties[aColumnIndex].columnClass.getSimpleName(),
															aValue.getClass().getSimpleName() }), aException);
		}
		finally
		{
			this.lock.writeLock().unlock();
		}
		return null;
	}

	protected void fireBeanTableCellUpdated(final int aRow, final int aColumn)
	{
		fireTableCellUpdated(aRow, aColumn);
	}

	/**
	 * @param model
	 * @param error
	 */
	protected static void dumpProperties(final BeanTableModel<?> model, final boolean error)
	{
		if (model != null)
		{
			final StringBuilder buffer = new StringBuilder(BUF_SIZE);
			buffer.append("Current Status: " + (model.clazz == null ? "null" : model.clazz.getName()) + "\n");
			int index = 0;
			if (model.properties != null)
			{
				for (final TableModelProperty prop : model.properties)
				{
					buffer.append("  * col=" + (index++) + " " + prop.toString() + "\n");
				}
			}
			if (error)
			{
				logger.error(buffer.toString());
			}
			else
			{
				logger.debug(buffer.toString());
			}
		}
	}

	@Override
	public void reset()
	{
		try
		{
			this.lock.writeLock().lock();
			this.array.clear();
		}
		finally
		{
			this.lock.writeLock().unlock();
		}
		fireTableDataChanged();
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#reset(java.util.Collection)
	 */
	@Override
	public void reset(final Collection<M> collection)
	{
		try
		{
			this.lock.writeLock().lock();
			this.array.clear();
			this.array.addAll(collection);
		}
		finally
		{
			this.lock.writeLock().unlock();
		}
		fireTableDataChanged();
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#deleteRow(java.lang.Object)
	 */
	@Override
	public M deleteRow(final M aValue)
	{
		try
		{
			this.lock.writeLock().lock();
			final Integer idx = this.arrayMap.get(aValue);
			if (idx != null)
			{
				return deleteRowAt(idx.intValue());
			}
		}
		finally
		{
			this.lock.writeLock().unlock();
		}

		return null;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#deleteRowAt(int)
	 */
	@Override
	public M deleteRowAt(final int aIdx)
	{
		M remove = null;
		try
		{
			this.lock.writeLock().lock();
			if (aIdx >= this.array.size())
			{
				return null;
			}

			remove = this.array.get(aIdx);
			if (remove != null)
			{
				this.array.remove(aIdx);
				this.arrayMap.remove(remove);
			}
		}
		finally
		{
			this.lock.writeLock().unlock();
		}
		fireTableRowsDeleted(aIdx, aIdx);
		return remove;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#deleteRows(int, int)
	 */
	@Override
	public int deleteRows(final int aStartIdx, final int aEndIdx)
	{
		if (aStartIdx >= 0)
		{
			final int end = Math.min(this.array.size() - 1, aEndIdx);
			int i = aStartIdx;
			try
			{
				this.lock.writeLock().lock();
				for (; i <= end; i++)
				{
					this.arrayMap.remove(this.array.get(i));
				}
				for (int j = i - 1; j >= aStartIdx; j--)
				{
					this.array.remove(j);
				}
			}
			finally
			{
				this.lock.writeLock().unlock();
			}
			fireTableRowsDeleted(aStartIdx, end);
			return (i - aStartIdx);
		}
		return 0;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.icroco.tablemodel.IBeanTableModel#deleteRows(int[])
	 */
	@Override
	public int[] deleteRows(final int... aIdx)
	{
		final List<Integer> removed = new ArrayList<Integer>();
		final int[] newIdx = Arrays.copyOf(aIdx, aIdx.length);
		Arrays.sort(newIdx);
		try
		{
			this.lock.writeLock().lock();
			for (int i = newIdx.length - 1; i >= 0; i--)
			{
				if (newIdx[i] < this.array.size())
				{
					final M value = this.array.remove(newIdx[i]);
					if (value != null)
					{
						this.arrayMap.remove(value);
						removed.add(Integer.valueOf(newIdx[i]));
						// TODO is it right ?
						fireTableRowsDeleted(newIdx[i], newIdx[i]);
					}
				}
			}
		}
		finally
		{
			this.lock.writeLock().unlock();
		}

		Collections.sort(removed);
		final int[] lReturn = new int[removed.size()];
		for (int i = 0; i < removed.size(); i++)
		{
			lReturn[i] = removed.get(i).intValue();
		}
		return lReturn;
	}

    @Override
    public List<M> getValues()
    {
        return Collections.unmodifiableList(this.array);
    }
}
