/**
 * 
 */
package org.icroco.tablemodel.impl.blinking;

import static org.icroco.tablemodel.impl.blinking.TableModelBlinkProperty.DEFAULT_BG_COLOR;

import java.awt.Color;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.TableCellRenderer;

import org.icroco.tablemodel.BeanTableModelException;
import org.icroco.tablemodel.blinking.ABlinking;
import org.icroco.tablemodel.blinking.EBlinkingMode;
import org.icroco.tablemodel.blinking.IBlinkTableModel;
import org.icroco.tablemodel.impl.BeanTableModel;
import org.icroco.tablemodel.impl.TableModelProperty;
import org.icroco.tablemodel.renderer.IRendererStage;
import org.icroco.tablemodel.renderer.PipelineTableCellRenderer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * BeanBlinkTableModel.
 * 
 * @author ouaibsky - 22 avr. 2009
 * @param <M>
 * 
 */
class BeanBlinkTableModel<M>
    extends BeanTableModel<M>
    implements IBlinkTableModel<M>
{
    private EBlinkingMode           defaultMode        = EBlinkingMode.ANIMATED_BG_COLOR;
    private Color                   defaultFgColor     = null;
    private Color                   defaultBgColor     = Color.decode(DEFAULT_BG_COLOR);
    private int                     defaultDuration    = 500;
    private int                     defaultRepeat      = 1;
    private boolean                 autoBlinkingForAll = false;
    
    /**
     * serialVersionUID
     */
    private static final long       serialVersionUID   = 1L;
    
    /**
     * Logger for this class
     */
    private transient static Logger logger             = LoggerFactory.getLogger(BeanBlinkTableModel.class);
    
    final TableFlasher              flasher;
    
    /**
     * @param aClass
     * @throws BeanTableModelException
     */
    public BeanBlinkTableModel(final JTable aTable, final Class<M> aClass)
        throws BeanTableModelException
    {
        super(aClass);
        this.flasher = new TableFlasher(aTable);
    }
    
    @Override
    protected TableModelProperty createNewProperty()
    {
        return new TableModelBlinkProperty();
    }
    
    @Override
    protected void readBTMAnnotationType(final Class<M> aClazz) throws BeanTableModelException
    {
        super.readBTMAnnotationType(aClazz); // read other annotation from super
        // class.
        readBBTMAnnotationType(aClazz);
        
    }
    
    protected void readBBTMAnnotationType(final Class<M> aClazz)
    {
        final ABlinking annotation = aClazz.getAnnotation(ABlinking.class);
        
        if (annotation != null)
        {
            if (annotation.mode() != EBlinkingMode.DEFAULT)
            {
                this.defaultMode = annotation.mode();
            }
            if (!annotation.bgColor().equalsIgnoreCase(ABlinking.DEFAULT_COLOR))
            {
                this.defaultBgColor = Color.decode(annotation.bgColor());
            }
            if (!annotation.fgColor().equalsIgnoreCase(ABlinking.DEFAULT_COLOR))
            {
                this.defaultFgColor = Color.decode(annotation.fgColor());
            }
            if (annotation.duration() > 0)
            {
                this.defaultDuration = annotation.duration();
            }
            if (annotation.repeat() > 0)
            {
                this.defaultRepeat = annotation.repeat();
            }
            this.autoBlinkingForAll = annotation.autoAddAllSetters();
        }
    }
    
    /**
     * (non-Javadoc)
     * 
     * @see org.icroco.tablemodel.impl.BeanTableModel#readBTMAnnotationMethod(java.lang.Class)
     */
    @Override
    protected List<TableModelProperty> readBTMAnnotationMethod(final Class<M> aClazz) throws BeanTableModelException
    {
        final List<TableModelProperty> props = super.readBTMAnnotationMethod(aClazz);
        readBBTMAnnotationMethod(props);
        
        return props;
    }
    
    private final void readBBTMAnnotationMethod(final List<TableModelProperty> aProps)
    {
        int i = -1;
        for (final TableModelProperty prop : aProps)
        {
            i++;
            final TableModelBlinkProperty bProp = (TableModelBlinkProperty) prop;
            final Method setter = bProp.getSetter();
            if (setter != null)
            {
                final ABlinking blinkAnnotation = setter.getAnnotation(ABlinking.class);
                if (this.autoBlinkingForAll)
                {
                    // TODO
                }
                if (blinkAnnotation != null)
                {
                    bProp.setBlink(true);
                    bProp.setMode(blinkAnnotation.mode() == EBlinkingMode.DEFAULT ? this.defaultMode : blinkAnnotation.mode());
                    bProp.setDuration(blinkAnnotation.duration() <= 0 ? this.defaultDuration : blinkAnnotation.duration());
                    bProp.setRepeat(blinkAnnotation.repeat() <= 0 ? this.defaultRepeat : blinkAnnotation.repeat());
                    bProp.setBlinkBgColor(blinkAnnotation.bgColor().equalsIgnoreCase(ABlinking.DEFAULT_COLOR) ? this.defaultBgColor : Color
                            .decode(blinkAnnotation.bgColor()));
                    bProp.setBlinkFgColor(blinkAnnotation.fgColor().equalsIgnoreCase(ABlinking.DEFAULT_COLOR) ? this.defaultFgColor : Color
                            .decode(blinkAnnotation.fgColor()));
                }
            }
        }
    }
    
    @Override
    public void setGlobalDuration(final int aDurationInMs)
    {
        try
        {
            if (SwingUtilities.isEventDispatchThread())
            {
                for (final TableModelProperty prop : this.properties)
                {
                    final TableModelBlinkProperty bProp = (TableModelBlinkProperty) prop;
                    bProp.setDuration(aDurationInMs);
                }
            }
            else
            {
                SwingUtilities.invokeAndWait(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        for (final TableModelProperty prop : BeanBlinkTableModel.this.properties)
                        {
                            final TableModelBlinkProperty bProp = (TableModelBlinkProperty) prop;
                            bProp.setDuration(aDurationInMs);
                        }
                    }
                    
                });
            }
        }
        catch (final InterruptedException aException)
        {
            // TODO Auto-generated catch block
            aException.printStackTrace();
        }
        catch (final InvocationTargetException aException)
        {
            // TODO Auto-generated catch block
            aException.printStackTrace();
        }
    }
    
    /**
     * package protected method, only called by
     */
    final void installRenderer()
    {
        super.installRenderer(this.flasher.getTable());
        int i = -1;
        for (final TableModelProperty prop : this.properties)
        {
            i++;
            final TableModelBlinkProperty bProp = (TableModelBlinkProperty) prop;
            final JTable table = this.flasher.getTable();
            
            if (!bProp.isBlink())
            {
                continue;
            }
            TableCellRenderer oldRenderer = null;
            // TODO check annotation.
            oldRenderer = table.getCellRenderer(0, i);
            if (oldRenderer == null)
            {
                oldRenderer = table.getDefaultRenderer(getColumnClass(i));
            }
            
            // Wrap the color renderer around the default renderer
            // TableCellRenderer colorRenderer = new
            // CellColorRenderer(defaultRenderer, provider);
            // Wrap the flash renderer around the colour renderer
            // // Register our flash renderer with the table
            // table.setDefaultRenderer(klass, flashRenderer);
            IRendererStage blinkStage = null;
            
            switch (bProp.getMode())
            {
            case ANIMATED_BG_COLOR:
                blinkStage = new BlinkAnimatedColorRendererStage(this.flasher, bProp.getBlinkBgColor(), bProp.getBlinkFgColor());
                break;
            case COLOR:
                blinkStage = new BlinkColorRendererStage(this.flasher, bProp.getBlinkBgColor(), bProp.getBlinkFgColor());
                break;
            case BORDER:
                blinkStage = new BlinkBorderRendererStage(this.flasher, bProp.getBlinkBgColor());
                break;
            case FONT:
                blinkStage = new BlinkFontRendererStage(this.flasher, bProp.getBlinkFgColor());
                break;
            
            default:
                logger.error("unknown blinking mode: " + bProp.getMode());
                break;
            }
            
            if (blinkStage != null)
            {
                TableCellRenderer newRenderer = null;
                if (oldRenderer instanceof PipelineTableCellRenderer)
                {
                    ((PipelineTableCellRenderer) oldRenderer).addStage(blinkStage);
                }
                else
                {
                    newRenderer = new PipelineTableCellRenderer(oldRenderer, blinkStage);
                    table.getColumnModel().getColumn(i).setCellRenderer(newRenderer);
                }
            }
            
        }
        
    }
    
    @Override
    protected void fireBeanTableCellUpdated(final int aRow, final int aColumn)
    {
        final TableModelBlinkProperty prop = (TableModelBlinkProperty) this.properties[aColumn];
        if (prop.isBlink())
        {
            this.flasher.flashCell(aRow, aColumn, prop.getRepeat(), prop.getDuration(), prop.getMode());
        }
        else
        {
            fireTableCellUpdated(aRow, aColumn);
        }
    }
    
    // @Override
    // protected void fireBeanTableRowsInserted(int aFirstRow, int aLastRow)
    // {
    // // TODO Auto-generated method stub
    // super.fireBeanTableRowsInserted(aFirstRow, aLastRow);
    // }
    //	
    // @Override
    // protected void fireBeanTableRowsUpdated(int aFirstRow, int aLastRow)
    // {
    // // TODO Auto-generated method stub
    // super.fireBeanTableRowsUpdated(aFirstRow, aLastRow);
    // }
}
