package org.icroco.tablemodel.impl.blinking;

import java.awt.Color;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.Timer;
import javax.swing.event.TableModelEvent;

import org.icroco.tablemodel.blinking.CellFlashProvider;
import org.icroco.tablemodel.blinking.EBlinkingMode;

/**
 * TableFlasher.
 * 
 * @author ouaibsky - 22 avr. 2009
 * 
 */
public final class TableFlasher
    implements CellFlashProvider, IFlashListener
{
    
    // Statics
    public final static int          MAX_FLASHES        = 1;
    public final static int          STD_FLASH_INTERVAL = 1000;
    //private final static int         MAX_FLASHER_POOL   = 1000;
    
    // Member attributes
    private final ReentrantLock      lock;
    private final Map<Long, Flasher> flashers;
    private final JTable             table;
    
    // private final Deque<Flasher> flasherPool;
    
    /**
     * Construct a Table flasher for a table
     * 
     * @param table
     *            , the table to flash
     */
    public TableFlasher(final JTable table)
    {
        this.lock = new ReentrantLock();
        this.table = table;
        this.flashers = new HashMap<Long, Flasher>();
        // this.flasherPool = new ArrayDeque<Flasher>(MAX_FLASHER_POOL);
    }
    
    JTable getTable()
    {
        return this.table;
    }
    
    /**
     * Determines the flash state of the cell Assumes the column model co-ordinate system not the view model
     * 
     * @param row
     *            , the row identifier
     * @param column
     *            , the column identifier
     * @return true if the cell flash is on, false otherwise
     */
    public boolean isFlashOn(final int row, final int column)
    {
        try
        {
            lock.lock();
            // Preempt checks
            if (this.flashers.isEmpty())
            {
                return false;
            }
            // check for any cell or row flashers
            return isFlasherOn(this.getCellFlasher(row, column));// ||
            // isFlasherOn(this.getRowFlasher(row));
        }
        finally
        {
            lock.unlock();
        }
    }
    
    public Object getFlashOn(final int row, final int column)
    {
        try
        {
            lock.lock();
            // Preempt checks
            if (this.flashers.isEmpty())
            {
                return null;
            }
            // check for any cell or row flashers
            final CellFlasher cell = this.getCellFlasher(row, column);
            return isFlasherOn(cell) ? cell : null;// ||
            // isFlasherOn(this.getRowFlasher(row));
        }
        finally
        {
            lock.unlock();
        }
    }
    
    /**
     * Flash a Cell in the Table using the default flash settings
     */
    public final void flashCell(final int row, final int col)
    {
        this.flashCell(row, col, MAX_FLASHES, STD_FLASH_INTERVAL, EBlinkingMode.ANIMATED_BG_COLOR);
    }
    
    /**
     * Flash a Cell in the Table
     * 
     * @param aBlinkingMode
     */
    public final void flashCell(final int row, final int col, final int maxNumberOfFlashes, final int flashInterval,
                                final EBlinkingMode aBlinkingMode)
    {
        if (isCellVisible(this.table, row, col))
        {
            final Long key = constructLongKey(row, col);
            // before creation, check if already exist for this pair row/col
            // final Flasher f = new CellFlasher(this.table, maxNumberOfFlashes,
            // flashInterval, new FlashListener() {
            // public void onFlashDone(Flasher flasher)
            // {
            // TableFlasher.this.removeFlasher(key);
            // }
            // }, row, col);
            // final CellFlasher f = new CellFlasher(this.table,
            // maxNumberOfFlashes, flashInterval, this, row, col, key);
            final CellFlasher f = aBlinkingMode == EBlinkingMode.ANIMATED_BG_COLOR ? new AnimatedCellFlasher(this.table,
                                                                                                             maxNumberOfFlashes,
                                                                                                             flashInterval,
                                                                                                             this,
                                                                                                             row,
                                                                                                             col,
                                                                                                             key)
                    : new CellFlasher(this.table, maxNumberOfFlashes, flashInterval, this, row, col, key);
            // this.addFlasher(key, f);
            // final CellFlasher f = borrowFlasher();// new
            // CellFlasher(this.table,
            // f.init(this.table, maxNumberOfFlashes, flashInterval, row, col,
            // key);
        }
    }
    
    /**
     * @param aTable
     * @param aRowIndex
     * @param aColIndex
     * @return
     */
    private final static boolean isCellVisible(final JTable aTable, final int aRowIndex, final int aColIndex)
    {
        if (aTable == null || !(aTable.getParent() instanceof JViewport))
        {
            return false;
        }
        final JViewport viewport = (JViewport) aTable.getParent();
        
        final int aViewRow = aTable.convertRowIndexToView(aRowIndex);
        final int aViewCol = aTable.convertColumnIndexToView(aColIndex);
        
        final Rectangle rect = aTable.getCellRect(aViewRow, aViewCol, true);
        // This rectangle is relative to the table where the
        // northwest corner of cell (0,0) is always (0,0)
        // if (aTable.getModel() != this || aTable.getModel() instanceof
        // TableModelAddon)
        // rect = aTable.getCellRect(((TableModelAddon)
        // aTable.getModel()).getModelToView(rowIndex), vColIndex, true);
        // else
        // rect = aTable.getCellRect(rowIndex, vColIndex, true);
        
        // The location of the viewport relative to the table
        final Point pt = viewport.getViewPosition();
        
        // Translate the cell location so that it is relative
        // to the view, assuming the northwest corner of the
        // view is (0,0)
        rect.setLocation(rect.x - pt.x, rect.y - pt.y);
        
        // Check if view completely contains cell
        return new Rectangle(viewport.getExtentSize()).intersects(rect);
    }
    
    /**
     * Flash a Row in the Table using the default flash settings
     */
    public void flashRow(final int row)
    {
        this.flashRow(row, MAX_FLASHES, STD_FLASH_INTERVAL);
    }
    
    /**
     * Flash a Row in the Table
     */
    public void flashRow(final int row, final int maxNumberOfFlashes, final int flashInterval)
    {
        final Long key = constructLongKey(row);
        
        final Flasher f = new RowFlasher(this.table, maxNumberOfFlashes, flashInterval, this, row, key);
        // final Flasher f = new RowFlasher(this.table, maxNumberOfFlashes,
        // flashInterval, new FlashListener()
        // {
        // public void onFlashDone(Flasher flasher)
        // {
        // TableFlasher.this.removeFlasher(key);
        // }
        // }, row);
        this.addFlasher(key, f);
    }
    
    /**
     * Clear flash for cell
     */
    public void clearCellFlash(final int row, final int col)
    {
        final Flasher f = this.getCellFlasher(row, col);
        if (f != null)
        {
            f.stopFlash();
            System.out.println("clearCellFlash: " + Thread.currentThread().getName());
        }
    }
    
    /**
     * Clear flash for a row
     */
    public void clearRowFlash(final int row)
    {
        final Flasher f = this.getRowFlasher(row);
        if (f != null)
        {
            f.stopFlash();
        }
    }
    
    /**
     * Clear all flashers
     */
    public void clearAllFlashers()
    {
        try
        {
            this.lock.lock();
            // Stop all flashers
            for (final Flasher oneFlasher : this.flashers.values())
            {
                // Stop the flasher
                oneFlasher.stopFlash();
            }
            // Clear the map for good measure
            this.flashers.clear();
        }
        finally
        {
            this.lock.unlock();
        }
    }
    
    /**
     * Add a flasher to the flash list
     */
    protected void addFlasher(final Long key, final Flasher f)
    {
        try
        {
            
            this.lock.lock();
            final Flasher old = (Flasher) this.flashers.get(key);
            if (old != null)
            {
                old.stopFlash();
            }
            this.flashers.put(key, f);
        }
        finally
        {
            this.lock.unlock();
        }
        f.startFlash();
    }
    
    /**
     * Return the CellFlasher for a cell if it exists
     */
    private final CellFlasher getCellFlasher(final int row, final int col)
    {
        final Object key = constructLongKey(row, col);
        try
        {
            this.lock.lock();
            final CellFlasher f = (CellFlasher) this.flashers.get(key);
            return f;
        }
        finally
        {
            this.lock.unlock();
        }
    }
    
    /**
     * Return the RowFlasher for a row if it exists
     */
    protected RowFlasher getRowFlasher(final int row)
    {
        final Object key = constructLongKey(row);
        try
        {
            this.lock.lock();
            final RowFlasher f = (RowFlasher) this.flashers.get(key);
            return f;
        }
        finally
        {
            this.lock.unlock();
        }
    }
    
    /**
     * Remove a flasher
     */
    protected void removeFlasher(final Object key)
    {
        try
        {
            this.lock.lock();
            // releaseFlasher(this.flashers.remove(key));
            this.flashers.remove(key);
        }
        finally
        {
            this.lock.unlock();
        }
    }
    
    // private final CellFlasher borrowFlasher()
    // {
    // final CellFlasher flasher = (CellFlasher) this.flasherPool.pollLast();
    // if (flasher == null)
    // {
    // return new AnimatedCellFlasher(this);
    // }
    //
    // return flasher;
    // }
    
    // private final void releaseFlasher(final Flasher aFlasher)
    // {
    // if (this.flasherPool.size() < MAX_FLASHER_POOL)
    // this.flasherPool.addLast(aFlasher);
    // // else let GC work !
    // }
    
    /**
     * Logic to determine whether a flasher is on or not
     */
    protected static boolean isFlasherOn(final Flasher f)
    {
        return ((f != null) && !f.isDoneFlashing() && f.isFlashOn());
    }
    
    // /**
    // * Utility to construct a flasher key
    // */
    // protected static Object constructKey(final int row, final int col)
    // {
    // return "" + row + "." + col;
    // }
    //
    // /**
    // * Utility to construct a flasher key
    // */
    // protected static Object constructKey(final int row)
    // {
    // return "" + row + ".-1";
    // }
    
    /**
     * constructLongKey is package protected.
     * 
     * @param row
     * @param col
     * @return
     */
    final static Long constructLongKey(final int row, final int col)
    {
        // final static long mixFullQuick(final int aFullLen, final int
        // aQuickLen)
        // {
        // // Warning, i don't know why (feel implicit int conversion)
        // // you need to affect variable into a long for shifting datas on long
        // 64b.
        // long lReturn = aFullLen;
        // lReturn <<= INT_LEN;
        // lReturn |= aQuickLen;
        //            
        // return lReturn;
        // }
        
        long lReturn = col;
        lReturn <<= 32;
        lReturn |= row;
        
        return Long.valueOf(lReturn);
        // return "" + row + "." + col;
    }
    
    /**
     * Utility to construct a flasher key
     */
    public final static Long constructLongKey(final int row)
    {
        return constructLongKey(row, -1);
    }
    
    public final static int getRow(final Long aValue)
    {
        return (int) aValue.longValue();
    }
    
    public final static int getCol(final Long aValue)
    {
        return (int) (aValue.longValue() >> 32);
    }
    
    public static void main(String[] args)
    {
        Long value = constructLongKey(50, 12);
        System.out.println("row:" + getRow(value) + " col:" + getCol(value));
        value = constructLongKey(Integer.MAX_VALUE, 12);
        System.out.println("row:" + getRow(value) + "=" + Integer.MAX_VALUE + " col:" + getCol(value));
        value = constructLongKey(Integer.MAX_VALUE, Integer.MAX_VALUE);
        System.out.println("row:" + getRow(value) + "=" + Integer.MAX_VALUE + " col:" + getCol(value) + "=" + Integer.MAX_VALUE);
        value = constructLongKey(60);
        System.out.println("row:" + getRow(value) + " col:" + getCol(value));
        value = constructLongKey(Integer.MAX_VALUE);
        System.out.println("row:" + getRow(value) + "=" + Integer.MAX_VALUE + " col:" + getCol(value));
    }
    
    @Override
    public void onFlashDone(Flasher flasher)
    {
        removeFlasher(flasher.key);
    }
    
    @Override
    public void onFlash(Flasher aFlasher)
    {
        addFlasher(aFlasher.getKey(), aFlasher);
    }
}

/**
 * Basic Flasher class for managing the Flash timer.
 */
abstract class Flasher
    implements ActionListener
{
    protected int            numberOfFlashes;
    // protected int maxFlashes;
    protected boolean        isFlashOn;
    protected Timer          timer;
    protected IFlashListener listener;
    protected JTable         table;
    protected Long           key;
    
    public Flasher(final JTable aTable, final int aMaxFlashes, final int aFlashInterval, final IFlashListener aListener, final Long aKey)
    {
        this.table = aTable;
        // this.maxFlashes = maxFlashes;
        this.numberOfFlashes = aMaxFlashes + 1; // +1 for un-flash
        this.isFlashOn = false;
        this.listener = aListener;
        this.key = aKey;
        
        this.timer = new Timer(aFlashInterval, this);
        this.timer.setInitialDelay(0);
        this.timer.setRepeats(true);
        this.listener.onFlash(this);
    }
    
    public Flasher(final IFlashListener aListener)
    {
        this.isFlashOn = true;
        this.listener = aListener;
        
        this.timer = new Timer(1000, this);
        this.timer.setInitialDelay(0);
        this.timer.setRepeats(true);
        this.listener.onFlash(this);
    }
    
    protected void init(final JTable aTable, final int aMaxFlashes, final int aFlashInterval, final Long aKey)
    {
        this.table = aTable;
        this.numberOfFlashes = aMaxFlashes + 1; // +1 for un-flash
        this.isFlashOn = false;
        this.key = aKey;
        this.timer.setInitialDelay(0);
        this.timer.setDelay(aFlashInterval);
        this.listener.onFlash(this);
    }
    
    /**
     * (non-Javadoc)
     * 
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    @Override
    public void actionPerformed(ActionEvent aE)
    {
        doFlash();
    }
    
    public boolean isDoneFlashing()
    {
        return (this.numberOfFlashes == 0);
    }
    
    public void startFlash()
    {
        this.timer.start();
    }
    
    public void stopFlash()
    {
        this.timer.stop();
        this.listener.onFlashDone(this);
    }
    
    public boolean isFlashOn()
    {
        return this.isFlashOn;
    }
    
    protected void flash(final TableModelEvent event)
    {
        // reverse flash state
        this.isFlashOn = !this.isFlashOn;
        // inc the number of flashes
        if (this.isFlashOn)
        {
            --this.numberOfFlashes;
        }
        // Notify table to refresh
        this.table.tableChanged(event);
        // Check to see if we are done and if so stop.
        if (this.isDoneFlashing())
        {
            this.stopFlash();
        }
    }
    
    public abstract void doFlash();
    
    public Long getKey()
    {
        return this.key;
    }
    
};

/**
 * Class for flashing a row
 */
final class RowFlasher
    extends Flasher
{
    
    int row;
    
    public RowFlasher(final JTable table, final int maxFlashes, final int flashInterval, final IFlashListener listener, final int row,
                      final Long aKey)
    {
        super(table, maxFlashes, flashInterval, listener, aKey);
        this.row = row;
    }
    
    @Override
    public void doFlash()
    {
        // Notify table that the row to be flashed should be redrawn
        final TableModelEvent event = new TableModelEvent(this.table.getModel(),
                                                          this.row,
                                                          this.row,
                                                          TableModelEvent.ALL_COLUMNS,
                                                          TableModelEvent.UPDATE);
        this.flash(event);
    }
};

/**
 * Class for flashing a cell
 */
class CellFlasher
    extends Flasher
{
    
    int row;
    int col;
    
    public CellFlasher(final JTable table, final int maxFlashes, final int flashInterval, final IFlashListener listener, final int row,
                       final int col, final Long aKey)
    {
        super(table, maxFlashes, flashInterval, listener, aKey);
        this.row = row;
        this.col = col;
    }
    
    public CellFlasher(IFlashListener listener)
    {
        super(listener);
        // TODO Auto-generated constructor stub
    }
    
    protected void init(JTable table, int maxFlashes, int flashInterval, final int row, final int col, Long key)
    {
        super.init(table, maxFlashes, flashInterval, key);
        this.row = row;
        this.col = col;
    }
    
    @Override
    public void doFlash()
    {
        // Notify table that the cell to be flashed should be redrawn
        final TableModelEvent event = new TableModelEvent(this.table.getModel(), this.row, this.row, this.col);
        this.flash(event);
    }
}

final class AnimatedCellFlasher
    extends CellFlasher
{
    final static int animatedStep = 10 * 2;
    final static int ANIM_UP      = 0;
    final static int ANIM_DOWN    = 1;
    int              maxAge;
    int              curAge;
    int              animWay;
    int              upMax;
    
    public AnimatedCellFlasher(final JTable table, final int maxFlashes, final int flashInterval, final IFlashListener listener,
                               final int row, final int col, final Long aKey)
    {
        super(table, animatedStep, flashInterval / animatedStep, listener, row, col, aKey);
        this.maxAge = animatedStep;
        this.upMax = 8;
        this.curAge = 0;
        this.animWay = ANIM_UP;
        this.isFlashOn = true;
    }
    
    public AnimatedCellFlasher(IFlashListener listener)
    {
        super(listener);
    }
    
    protected void init(JTable table, int maxFlashes, int flashInterval, final int row, final int col, Long key)
    {
        super.init(table, maxFlashes, flashInterval, key);
        this.maxAge = animatedStep;
        this.curAge = 0;
    }
    
    @Override
    public void doFlash()
    {
//        if (this.animWay == ANIM_UP)
//        {
//            this.curAge++;
//            if (this.curAge == this.maxAge)
//                this.animWay = ANIM_DOWN;
//        }
//        else
//            this.curAge--;
        this.curAge++;
        final TableModelEvent event = new TableModelEvent(this.table.getModel(), this.row, this.row, this.col);
        flash(event);
    }
    
    protected void flash(final TableModelEvent event)
    {
        --this.numberOfFlashes;
        // reverse flash state
        if (this.numberOfFlashes <= 0)
            this.isFlashOn = false;
        // Notify table to refresh
        this.table.tableChanged(event);
        // Check to see if we are done and if so stop.
        if (this.isDoneFlashing())
        {
            this.stopFlash();
        }
    }
    
    public Color getGradient(final Color aNormal, final Color aFlash)
    {
       return aNormal == null ? aFlash : mix(aFlash, aNormal, getCoef((float) this.curAge / (float) this.maxAge));
        // (x)*e^(-x*3.6+2.25)
        // (x)*e^(-x*7+3)
        //return aNormal == null ? aFlash : mix(aFlash, aNormal, );
    }
    
//    final static float getCoef(float x)
//    {
//        //return (float) (x*Math.exp(-x*7+3));
//        return (float) (Math.sqrt(x)*Math.cos(x)+1/2);
//    }

    final float getCoef(float x)
    {
        return -x+1;
     //   return this.curAge <= this.upMax ? getCoefUp(x, this.upMax) : getCoefDown(x, this.upMax);
    }

    
    final static float getCoefUp(float x, final int ajust)
    {
        return (float) (ajust*x);
    }
    
    final static float getCoefDown(float x, final int ajust)
    {
        return (float) -(ajust/(ajust-1))*x+1+(1/ajust);
    }

    
    private final static Color mix(final Color normal, final Color color, float pos)
    {
        
        pos = Math.min(1.0F, pos);
        final float npos = 1.0F - pos;
        int r = (int) ((normal.getRed() * pos) + (npos * color.getRed()));
        int g = (int) ((normal.getGreen() * pos) + (npos * color.getGreen()));
        int b = (int) ((normal.getBlue() * pos) + (npos * color.getBlue()));
        r = Math.max(0, Math.min(255, r));
        g = Math.max(0, Math.min(255, g));
        b = Math.max(0, Math.min(255, b));
        return new Color(r, g, b);
    }
    
}
