package org.icroco.tablemodel.renderer.plugin;

import java.awt.Component;
import java.util.Map;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.SwingConstants;

import org.icroco.tablemodel.renderer.IHistoricalValue;
import org.icroco.tablemodel.renderer.IRendererStage;

/**
 * CellColorRenderStage.
 * 
 * @author clalleme - 26 mai 08
 * 
 */
public final class UpDownIconRenderStage
    implements IRendererStage
{
    private Icon upIcon;
    private Icon downIcon;
    private Icon equalIcon;
    
    /**
     * Constructor.
     * 
     */
    public UpDownIconRenderStage(final Icon aUpIcon, final Icon aDownIcon, final Icon aEqualIcon)
    {
        super();
        this.upIcon = aUpIcon;
        this.downIcon = aDownIcon;
        this.equalIcon = aEqualIcon;
    }
    
    /**
     * Constructor.
     * 
     */
    public UpDownIconRenderStage()
    {
        super();
        this.upIcon = new ImageIcon(getClass().getResource("arrow_up.png"));
        this.downIcon = new ImageIcon(getClass().getResource("arrow_down.png"));
        this.equalIcon = new ImageIcon(getClass().getResource("arrow_empty.png"));
    }
    
    /**
     * 
     * @param aParameters
     *            must contains 3 sub string separate by comma, possible value are<br/>
     *            <ul>
     *            <li><b>up</b>=<i>icon_resource_path</i></li>
     *            <li><b>down</b>=<i>icon_resource_path</i></li>
     *            <li><b>equal</b>=icon_resource_path</i></li>
     *            </ul>
     *            <b><i>icon_resource_path</i></b> represent a resource URL as taken by {@link Class#getResource(String)}. you can passe
     *            resource relative to this class or 'absolute' resource (start with '/').<br/>
     * 
     *            All args are optional (in this case we use default icon) or value can be <b>none</b> to indicate you don't want an icon
     *            (for equal by example).<br/>
     */
    public UpDownIconRenderStage(final String aParameters)
    {
        this();
        
        final Map<String, String> res = ParametersUtil.parseValue(aParameters);
        final String upIconStr = res.get("up");
        final String downIconStr = res.get("down");
        final String equalIconStr = res.get("equal");
        
        if (upIconStr != null)
            this.upIcon = upIconStr.trim().equalsIgnoreCase("none") ? null : new ImageIcon(getClass().getResource(upIconStr));
        if (downIconStr != null)
            this.downIcon = downIconStr.trim().equalsIgnoreCase("none") ? null : new ImageIcon(getClass().getResource(downIconStr));
        if (equalIconStr != null)
            this.equalIcon = equalIconStr.trim().equalsIgnoreCase("none") ? null : new ImageIcon(getClass().getResource(equalIconStr));
    }
    
    /**
     * (non-Javadoc)
     * 
     * @see org.icroco.tablemodel.renderer.IRendererStage#initRenderer(Class, java.awt.Component)
     */
    @Override
    public void initRenderer(Class<?> aColumnClass, final Component aRendered)
    {
        if (aRendered instanceof JLabel)
        {
            ((JLabel) aRendered).setHorizontalTextPosition(SwingConstants.LEFT);
            ((JLabel) aRendered).setHorizontalAlignment(JLabel.RIGHT);
        }
    }
    
    @Override
    public Object preRender(JTable aTable, Object aValue, boolean aIsSelected, boolean aHasFocus, int aRow, int aColumn)
    {
        return aValue;
    }
    
    @Override
    public void render(Component aRendered, JTable aTable, Object aValue, boolean aIsSelected, boolean aHasFocus, int aRow, int aColumn)
    {
        final IHistoricalValue<?> hValue = (IHistoricalValue<?>) aValue;
        
        final double newValue = hValue.getLast() != null ? ((Number) hValue.getLast()).doubleValue() :0;
        final double oldValue = hValue.getOld() != null ? ((Number) hValue.getOld()).doubleValue() : newValue;
        
        final JLabel label = (JLabel) aRendered;
        if (oldValue < newValue)
            label.setIcon(this.upIcon);
        else if (oldValue > newValue)
            label.setIcon(this.downIcon);
        else
            label.setIcon(this.equalIcon);
    }
}