package org.immutables.value.processor.meta;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Booleans;
import com.google.common.primitives.Ints;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;

/**
 * StyleMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.value.Value.Style}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.ValueMirrors.Style} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class StyleMirror implements ValueMirrors.Style {
  public static final String QUALIFIED_NAME = "org.immutables.value.Value.Style";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.ValueMirrors.Style";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Style";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code StyleMirror}, present if this annotation found
   */
  public static Optional<StyleMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code StyleMirror}, present if this annotation found
   */
  public static Optional<StyleMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new StyleMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code StyleMirror}s
   */
  public static ImmutableList<StyleMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<StyleMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new StyleMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code StyleMirror}
   */
  public static StyleMirror from(TypeElement element) {
    return new StyleMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code StyleMirror}, present if mirror matched this annotation type
   */
  public static Optional<StyleMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String[] get;
  private final String init;
  private final String with;
  private final String withUnaryOperator;
  private final String add;
  private final String addAll;
  private final String put;
  private final String putAll;
  private final String copyOf;
  private final String of;
  private final String instance;
  private final String builder;
  private final String newBuilder;
  private final String from;
  private final String build;
  private final String buildOrThrow;
  private final String canBuild;
  private final String toBuilder;
  private final String underrideHashCode;
  private final String underrideEquals;
  private final String underrideToString;
  private final String delegateToString;
  private final String includeHashCode;
  private final String isInitialized;
  private final String isSet;
  private final String set;
  private final String unset;
  private final String clear;
  private final String create;
  private final String toImmutable;
  private final String typeBuilder;
  private final String typeInnerBuilder;
  private final String[] typeAbstract;
  private final String typeImmutable;
  private final String typeImmutableEnclosing;
  private final String typeImmutableNested;
  private final String typeModifiable;
  private final String typeInnerModifiable;
  private final String typeWith;
  private final String packageGenerated;
  private final AnnotationMirror defaults;
  private final ImmutableMirror defaultsMirror;
  private final boolean strictBuilder;
  private final boolean strictModifiable;
  private final ValueMirrors.Style.ValidationMethod validationMethod;
  private final boolean allParameters;
  private final boolean defaultAsDefault;
  private final boolean headerComments;
  private final boolean jdkOnly;
  private final boolean jdk9Collections;
  private final TypeMirror[] passAnnotations;
  private final String[] passAnnotationsName;
  private final TypeMirror[] additionalJsonAnnotations;
  private final String[] additionalJsonAnnotationsName;
  private final ValueMirrors.Style.ImplementationVisibility visibility;
  private final String visibilityString;
  private final boolean optionalAcceptNullable;
  private final boolean generateSuppressAllWarnings;
  private final boolean privateNoargConstructor;
  private final boolean protectedNoargConstructor;
  private final boolean attributelessSingleton;
  private final boolean unsafeDefaultAndDerived;
  private final boolean clearBuilder;
  private final boolean deferCollectionAllocation;
  private final boolean deepImmutablesDetection;
  private final boolean overshadowImplementation;
  private final boolean implementationNestedInBuilder;
  private final boolean forceJacksonPropertyNames;
  private final boolean forceJacksonIgnoreFields;
  private final boolean forceEqualsInWithers;
  private final boolean jacksonIntegration;
  private final boolean weakInterning;
  private final boolean alwaysPublicInitializers;
  private final ValueMirrors.Style.BuilderVisibility builderVisibility;
  private final String builderVisibilityString;
  private final TypeMirror throwForInvalidImmutableState;
  private final String throwForInvalidImmutableStateName;
  private final TypeMirror throwForNullPointer;
  private final String throwForNullPointerName;
  private final boolean depluralize;
  private final String[] depluralizeDictionary;
  private final TypeMirror[] immutableCopyOfRoutines;
  private final String[] immutableCopyOfRoutinesName;
  private final boolean stagedBuilder;
  private final boolean builtinContainerAttributes;
  private final boolean beanFriendlyModifiables;
  private final boolean allMandatoryParameters;
  private final String redactedMask;
  private final boolean attributeBuilderDetection;
  private final boolean transientDerivedFields;
  private final boolean finalInstanceFields;
  private final String[] attributeBuilder;
  private final String getBuilder;
  private final String setBuilder;
  private final String addBuilder;
  private final String addAllBuilder;
  private final String getBuilders;
  private final String nullableAnnotation;
  private final int limitStringLengthInToString;
  private final TypeMirror[] allowedClasspathAnnotations;
  private final String[] allowedClasspathAnnotationsName;
  private final TypeMirror fallbackNullableAnnotation;
  private final String fallbackNullableAnnotationName;
  private final boolean jakarta;

  private StyleMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String[] get = null;
    String init = null;
    String with = null;
    String withUnaryOperator = null;
    String add = null;
    String addAll = null;
    String put = null;
    String putAll = null;
    String copyOf = null;
    String of = null;
    String instance = null;
    String builder = null;
    String newBuilder = null;
    String from = null;
    String build = null;
    String buildOrThrow = null;
    String canBuild = null;
    String toBuilder = null;
    String underrideHashCode = null;
    String underrideEquals = null;
    String underrideToString = null;
    String delegateToString = null;
    String includeHashCode = null;
    String isInitialized = null;
    String isSet = null;
    String set = null;
    String unset = null;
    String clear = null;
    String create = null;
    String toImmutable = null;
    String typeBuilder = null;
    String typeInnerBuilder = null;
    String[] typeAbstract = null;
    String typeImmutable = null;
    String typeImmutableEnclosing = null;
    String typeImmutableNested = null;
    String typeModifiable = null;
    String typeInnerModifiable = null;
    String typeWith = null;
    String packageGenerated = null;
    AnnotationMirror defaults = null;
    ImmutableMirror defaultsMirror = null;
    boolean strictBuilder = false;
    boolean strictModifiable = false;
    ValueMirrors.Style.ValidationMethod validationMethod = null;
    boolean allParameters = false;
    boolean defaultAsDefault = false;
    boolean headerComments = false;
    boolean jdkOnly = false;
    boolean jdk9Collections = false;
    TypeMirror[] passAnnotations = null;
    String[] passAnnotationsName = null;
    TypeMirror[] additionalJsonAnnotations = null;
    String[] additionalJsonAnnotationsName = null;
    ValueMirrors.Style.ImplementationVisibility visibility = null;
    String visibilityString = null;
    boolean optionalAcceptNullable = false;
    boolean generateSuppressAllWarnings = false;
    boolean privateNoargConstructor = false;
    boolean protectedNoargConstructor = false;
    boolean attributelessSingleton = false;
    boolean unsafeDefaultAndDerived = false;
    boolean clearBuilder = false;
    boolean deferCollectionAllocation = false;
    boolean deepImmutablesDetection = false;
    boolean overshadowImplementation = false;
    boolean implementationNestedInBuilder = false;
    boolean forceJacksonPropertyNames = false;
    boolean forceJacksonIgnoreFields = false;
    boolean forceEqualsInWithers = false;
    boolean jacksonIntegration = false;
    boolean weakInterning = false;
    boolean alwaysPublicInitializers = false;
    ValueMirrors.Style.BuilderVisibility builderVisibility = null;
    String builderVisibilityString = null;
    TypeMirror throwForInvalidImmutableState = null;
    String throwForInvalidImmutableStateName = null;
    TypeMirror throwForNullPointer = null;
    String throwForNullPointerName = null;
    boolean depluralize = false;
    String[] depluralizeDictionary = null;
    TypeMirror[] immutableCopyOfRoutines = null;
    String[] immutableCopyOfRoutinesName = null;
    boolean stagedBuilder = false;
    boolean builtinContainerAttributes = false;
    boolean beanFriendlyModifiables = false;
    boolean allMandatoryParameters = false;
    String redactedMask = null;
    boolean attributeBuilderDetection = false;
    boolean transientDerivedFields = false;
    boolean finalInstanceFields = false;
    String[] attributeBuilder = null;
    String getBuilder = null;
    String setBuilder = null;
    String addBuilder = null;
    String addAllBuilder = null;
    String getBuilders = null;
    String nullableAnnotation = null;
    int limitStringLengthInToString = 0;
    TypeMirror[] allowedClasspathAnnotations = null;
    String[] allowedClasspathAnnotationsName = null;
    TypeMirror fallbackNullableAnnotation = null;
    String fallbackNullableAnnotationName = null;
    boolean jakarta = false;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("get".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        GetExtractor getExtractor$ = new GetExtractor();
        annotationValue$.accept(getExtractor$, null);

        get = getExtractor$.get();
        continue;
      }
      if ("init".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        InitExtractor initExtractor$ = new InitExtractor();
        annotationValue$.accept(initExtractor$, null);

        init = initExtractor$.get();
        continue;
      }
      if ("with".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        WithExtractor withExtractor$ = new WithExtractor();
        annotationValue$.accept(withExtractor$, null);

        with = withExtractor$.get();
        continue;
      }
      if ("withUnaryOperator".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        WithUnaryOperatorExtractor withUnaryOperatorExtractor$ = new WithUnaryOperatorExtractor();
        annotationValue$.accept(withUnaryOperatorExtractor$, null);

        withUnaryOperator = withUnaryOperatorExtractor$.get();
        continue;
      }
      if ("add".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AddExtractor addExtractor$ = new AddExtractor();
        annotationValue$.accept(addExtractor$, null);

        add = addExtractor$.get();
        continue;
      }
      if ("addAll".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AddAllExtractor addAllExtractor$ = new AddAllExtractor();
        annotationValue$.accept(addAllExtractor$, null);

        addAll = addAllExtractor$.get();
        continue;
      }
      if ("put".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        PutExtractor putExtractor$ = new PutExtractor();
        annotationValue$.accept(putExtractor$, null);

        put = putExtractor$.get();
        continue;
      }
      if ("putAll".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        PutAllExtractor putAllExtractor$ = new PutAllExtractor();
        annotationValue$.accept(putAllExtractor$, null);

        putAll = putAllExtractor$.get();
        continue;
      }
      if ("copyOf".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        CopyOfExtractor copyOfExtractor$ = new CopyOfExtractor();
        annotationValue$.accept(copyOfExtractor$, null);

        copyOf = copyOfExtractor$.get();
        continue;
      }
      if ("of".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        OfExtractor ofExtractor$ = new OfExtractor();
        annotationValue$.accept(ofExtractor$, null);

        of = ofExtractor$.get();
        continue;
      }
      if ("instance".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        InstanceExtractor instanceExtractor$ = new InstanceExtractor();
        annotationValue$.accept(instanceExtractor$, null);

        instance = instanceExtractor$.get();
        continue;
      }
      if ("builder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        BuilderExtractor builderExtractor$ = new BuilderExtractor();
        annotationValue$.accept(builderExtractor$, null);

        builder = builderExtractor$.get();
        continue;
      }
      if ("newBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        NewBuilderExtractor newBuilderExtractor$ = new NewBuilderExtractor();
        annotationValue$.accept(newBuilderExtractor$, null);

        newBuilder = newBuilderExtractor$.get();
        continue;
      }
      if ("from".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        FromExtractor fromExtractor$ = new FromExtractor();
        annotationValue$.accept(fromExtractor$, null);

        from = fromExtractor$.get();
        continue;
      }
      if ("build".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        BuildExtractor buildExtractor$ = new BuildExtractor();
        annotationValue$.accept(buildExtractor$, null);

        build = buildExtractor$.get();
        continue;
      }
      if ("buildOrThrow".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        BuildOrThrowExtractor buildOrThrowExtractor$ = new BuildOrThrowExtractor();
        annotationValue$.accept(buildOrThrowExtractor$, null);

        buildOrThrow = buildOrThrowExtractor$.get();
        continue;
      }
      if ("canBuild".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        CanBuildExtractor canBuildExtractor$ = new CanBuildExtractor();
        annotationValue$.accept(canBuildExtractor$, null);

        canBuild = canBuildExtractor$.get();
        continue;
      }
      if ("toBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ToBuilderExtractor toBuilderExtractor$ = new ToBuilderExtractor();
        annotationValue$.accept(toBuilderExtractor$, null);

        toBuilder = toBuilderExtractor$.get();
        continue;
      }
      if ("underrideHashCode".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        UnderrideHashCodeExtractor underrideHashCodeExtractor$ = new UnderrideHashCodeExtractor();
        annotationValue$.accept(underrideHashCodeExtractor$, null);

        underrideHashCode = underrideHashCodeExtractor$.get();
        continue;
      }
      if ("underrideEquals".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        UnderrideEqualsExtractor underrideEqualsExtractor$ = new UnderrideEqualsExtractor();
        annotationValue$.accept(underrideEqualsExtractor$, null);

        underrideEquals = underrideEqualsExtractor$.get();
        continue;
      }
      if ("underrideToString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        UnderrideToStringExtractor underrideToStringExtractor$ = new UnderrideToStringExtractor();
        annotationValue$.accept(underrideToStringExtractor$, null);

        underrideToString = underrideToStringExtractor$.get();
        continue;
      }
      if ("delegateToString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        DelegateToStringExtractor delegateToStringExtractor$ = new DelegateToStringExtractor();
        annotationValue$.accept(delegateToStringExtractor$, null);

        delegateToString = delegateToStringExtractor$.get();
        continue;
      }
      if ("includeHashCode".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        IncludeHashCodeExtractor includeHashCodeExtractor$ = new IncludeHashCodeExtractor();
        annotationValue$.accept(includeHashCodeExtractor$, null);

        includeHashCode = includeHashCodeExtractor$.get();
        continue;
      }
      if ("isInitialized".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        IsInitializedExtractor isInitializedExtractor$ = new IsInitializedExtractor();
        annotationValue$.accept(isInitializedExtractor$, null);

        isInitialized = isInitializedExtractor$.get();
        continue;
      }
      if ("isSet".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        IsSetExtractor isSetExtractor$ = new IsSetExtractor();
        annotationValue$.accept(isSetExtractor$, null);

        isSet = isSetExtractor$.get();
        continue;
      }
      if ("set".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        SetExtractor setExtractor$ = new SetExtractor();
        annotationValue$.accept(setExtractor$, null);

        set = setExtractor$.get();
        continue;
      }
      if ("unset".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        UnsetExtractor unsetExtractor$ = new UnsetExtractor();
        annotationValue$.accept(unsetExtractor$, null);

        unset = unsetExtractor$.get();
        continue;
      }
      if ("clear".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ClearExtractor clearExtractor$ = new ClearExtractor();
        annotationValue$.accept(clearExtractor$, null);

        clear = clearExtractor$.get();
        continue;
      }
      if ("create".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        CreateExtractor createExtractor$ = new CreateExtractor();
        annotationValue$.accept(createExtractor$, null);

        create = createExtractor$.get();
        continue;
      }
      if ("toImmutable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ToImmutableExtractor toImmutableExtractor$ = new ToImmutableExtractor();
        annotationValue$.accept(toImmutableExtractor$, null);

        toImmutable = toImmutableExtractor$.get();
        continue;
      }
      if ("typeBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeBuilderExtractor typeBuilderExtractor$ = new TypeBuilderExtractor();
        annotationValue$.accept(typeBuilderExtractor$, null);

        typeBuilder = typeBuilderExtractor$.get();
        continue;
      }
      if ("typeInnerBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeInnerBuilderExtractor typeInnerBuilderExtractor$ = new TypeInnerBuilderExtractor();
        annotationValue$.accept(typeInnerBuilderExtractor$, null);

        typeInnerBuilder = typeInnerBuilderExtractor$.get();
        continue;
      }
      if ("typeAbstract".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeAbstractExtractor typeAbstractExtractor$ = new TypeAbstractExtractor();
        annotationValue$.accept(typeAbstractExtractor$, null);

        typeAbstract = typeAbstractExtractor$.get();
        continue;
      }
      if ("typeImmutable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeImmutableExtractor typeImmutableExtractor$ = new TypeImmutableExtractor();
        annotationValue$.accept(typeImmutableExtractor$, null);

        typeImmutable = typeImmutableExtractor$.get();
        continue;
      }
      if ("typeImmutableEnclosing".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeImmutableEnclosingExtractor typeImmutableEnclosingExtractor$ = new TypeImmutableEnclosingExtractor();
        annotationValue$.accept(typeImmutableEnclosingExtractor$, null);

        typeImmutableEnclosing = typeImmutableEnclosingExtractor$.get();
        continue;
      }
      if ("typeImmutableNested".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeImmutableNestedExtractor typeImmutableNestedExtractor$ = new TypeImmutableNestedExtractor();
        annotationValue$.accept(typeImmutableNestedExtractor$, null);

        typeImmutableNested = typeImmutableNestedExtractor$.get();
        continue;
      }
      if ("typeModifiable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeModifiableExtractor typeModifiableExtractor$ = new TypeModifiableExtractor();
        annotationValue$.accept(typeModifiableExtractor$, null);

        typeModifiable = typeModifiableExtractor$.get();
        continue;
      }
      if ("typeInnerModifiable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeInnerModifiableExtractor typeInnerModifiableExtractor$ = new TypeInnerModifiableExtractor();
        annotationValue$.accept(typeInnerModifiableExtractor$, null);

        typeInnerModifiable = typeInnerModifiableExtractor$.get();
        continue;
      }
      if ("typeWith".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TypeWithExtractor typeWithExtractor$ = new TypeWithExtractor();
        annotationValue$.accept(typeWithExtractor$, null);

        typeWith = typeWithExtractor$.get();
        continue;
      }
      if ("packageGenerated".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        PackageGeneratedExtractor packageGeneratedExtractor$ = new PackageGeneratedExtractor();
        annotationValue$.accept(packageGeneratedExtractor$, null);

        packageGenerated = packageGeneratedExtractor$.get();
        continue;
      }
      if ("defaults".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        DefaultsExtractor defaultsExtractor$ = new DefaultsExtractor();
        annotationValue$.accept(defaultsExtractor$, null);

        defaults = defaultsExtractor$.get();
        defaultsMirror = defaultsExtractor$.mirror();
        continue;
      }
      if ("strictBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        StrictBuilderExtractor strictBuilderExtractor$ = new StrictBuilderExtractor();
        annotationValue$.accept(strictBuilderExtractor$, null);

        strictBuilder = strictBuilderExtractor$.get();
        continue;
      }
      if ("strictModifiable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        StrictModifiableExtractor strictModifiableExtractor$ = new StrictModifiableExtractor();
        annotationValue$.accept(strictModifiableExtractor$, null);

        strictModifiable = strictModifiableExtractor$.get();
        continue;
      }
      if ("validationMethod".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ValidationMethodExtractor validationMethodExtractor$ = new ValidationMethodExtractor();
        annotationValue$.accept(validationMethodExtractor$, null);

        validationMethod = validationMethodExtractor$.get();
        continue;
      }
      if ("allParameters".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AllParametersExtractor allParametersExtractor$ = new AllParametersExtractor();
        annotationValue$.accept(allParametersExtractor$, null);

        allParameters = allParametersExtractor$.get();
        continue;
      }
      if ("defaultAsDefault".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        DefaultAsDefaultExtractor defaultAsDefaultExtractor$ = new DefaultAsDefaultExtractor();
        annotationValue$.accept(defaultAsDefaultExtractor$, null);

        defaultAsDefault = defaultAsDefaultExtractor$.get();
        continue;
      }
      if ("headerComments".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        HeaderCommentsExtractor headerCommentsExtractor$ = new HeaderCommentsExtractor();
        annotationValue$.accept(headerCommentsExtractor$, null);

        headerComments = headerCommentsExtractor$.get();
        continue;
      }
      if ("jdkOnly".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        JdkOnlyExtractor jdkOnlyExtractor$ = new JdkOnlyExtractor();
        annotationValue$.accept(jdkOnlyExtractor$, null);

        jdkOnly = jdkOnlyExtractor$.get();
        continue;
      }
      if ("jdk9Collections".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        Jdk9CollectionsExtractor jdk9CollectionsExtractor$ = new Jdk9CollectionsExtractor();
        annotationValue$.accept(jdk9CollectionsExtractor$, null);

        jdk9Collections = jdk9CollectionsExtractor$.get();
        continue;
      }
      if ("passAnnotations".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        PassAnnotationsExtractor passAnnotationsExtractor$ = new PassAnnotationsExtractor();
        annotationValue$.accept(passAnnotationsExtractor$, null);

        passAnnotations = passAnnotationsExtractor$.get();
        passAnnotationsName = passAnnotationsExtractor$.name();
        continue;
      }
      if ("additionalJsonAnnotations".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AdditionalJsonAnnotationsExtractor additionalJsonAnnotationsExtractor$ = new AdditionalJsonAnnotationsExtractor();
        annotationValue$.accept(additionalJsonAnnotationsExtractor$, null);

        additionalJsonAnnotations = additionalJsonAnnotationsExtractor$.get();
        additionalJsonAnnotationsName = additionalJsonAnnotationsExtractor$.name();
        continue;
      }
      if ("visibility".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        VisibilityExtractor visibilityExtractor$ = new VisibilityExtractor();
        annotationValue$.accept(visibilityExtractor$, null);

        visibility = visibilityExtractor$.get();
        continue;
      }
      if ("visibilityString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        VisibilityStringExtractor visibilityStringExtractor$ = new VisibilityStringExtractor();
        annotationValue$.accept(visibilityStringExtractor$, null);

        visibilityString = visibilityStringExtractor$.get();
        continue;
      }
      if ("optionalAcceptNullable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        OptionalAcceptNullableExtractor optionalAcceptNullableExtractor$ = new OptionalAcceptNullableExtractor();
        annotationValue$.accept(optionalAcceptNullableExtractor$, null);

        optionalAcceptNullable = optionalAcceptNullableExtractor$.get();
        continue;
      }
      if ("generateSuppressAllWarnings".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        GenerateSuppressAllWarningsExtractor generateSuppressAllWarningsExtractor$ = new GenerateSuppressAllWarningsExtractor();
        annotationValue$.accept(generateSuppressAllWarningsExtractor$, null);

        generateSuppressAllWarnings = generateSuppressAllWarningsExtractor$.get();
        continue;
      }
      if ("privateNoargConstructor".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        PrivateNoargConstructorExtractor privateNoargConstructorExtractor$ = new PrivateNoargConstructorExtractor();
        annotationValue$.accept(privateNoargConstructorExtractor$, null);

        privateNoargConstructor = privateNoargConstructorExtractor$.get();
        continue;
      }
      if ("protectedNoargConstructor".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ProtectedNoargConstructorExtractor protectedNoargConstructorExtractor$ = new ProtectedNoargConstructorExtractor();
        annotationValue$.accept(protectedNoargConstructorExtractor$, null);

        protectedNoargConstructor = protectedNoargConstructorExtractor$.get();
        continue;
      }
      if ("attributelessSingleton".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AttributelessSingletonExtractor attributelessSingletonExtractor$ = new AttributelessSingletonExtractor();
        annotationValue$.accept(attributelessSingletonExtractor$, null);

        attributelessSingleton = attributelessSingletonExtractor$.get();
        continue;
      }
      if ("unsafeDefaultAndDerived".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        UnsafeDefaultAndDerivedExtractor unsafeDefaultAndDerivedExtractor$ = new UnsafeDefaultAndDerivedExtractor();
        annotationValue$.accept(unsafeDefaultAndDerivedExtractor$, null);

        unsafeDefaultAndDerived = unsafeDefaultAndDerivedExtractor$.get();
        continue;
      }
      if ("clearBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ClearBuilderExtractor clearBuilderExtractor$ = new ClearBuilderExtractor();
        annotationValue$.accept(clearBuilderExtractor$, null);

        clearBuilder = clearBuilderExtractor$.get();
        continue;
      }
      if ("deferCollectionAllocation".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        DeferCollectionAllocationExtractor deferCollectionAllocationExtractor$ = new DeferCollectionAllocationExtractor();
        annotationValue$.accept(deferCollectionAllocationExtractor$, null);

        deferCollectionAllocation = deferCollectionAllocationExtractor$.get();
        continue;
      }
      if ("deepImmutablesDetection".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        DeepImmutablesDetectionExtractor deepImmutablesDetectionExtractor$ = new DeepImmutablesDetectionExtractor();
        annotationValue$.accept(deepImmutablesDetectionExtractor$, null);

        deepImmutablesDetection = deepImmutablesDetectionExtractor$.get();
        continue;
      }
      if ("overshadowImplementation".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        OvershadowImplementationExtractor overshadowImplementationExtractor$ = new OvershadowImplementationExtractor();
        annotationValue$.accept(overshadowImplementationExtractor$, null);

        overshadowImplementation = overshadowImplementationExtractor$.get();
        continue;
      }
      if ("implementationNestedInBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ImplementationNestedInBuilderExtractor implementationNestedInBuilderExtractor$ = new ImplementationNestedInBuilderExtractor();
        annotationValue$.accept(implementationNestedInBuilderExtractor$, null);

        implementationNestedInBuilder = implementationNestedInBuilderExtractor$.get();
        continue;
      }
      if ("forceJacksonPropertyNames".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ForceJacksonPropertyNamesExtractor forceJacksonPropertyNamesExtractor$ = new ForceJacksonPropertyNamesExtractor();
        annotationValue$.accept(forceJacksonPropertyNamesExtractor$, null);

        forceJacksonPropertyNames = forceJacksonPropertyNamesExtractor$.get();
        continue;
      }
      if ("forceJacksonIgnoreFields".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ForceJacksonIgnoreFieldsExtractor forceJacksonIgnoreFieldsExtractor$ = new ForceJacksonIgnoreFieldsExtractor();
        annotationValue$.accept(forceJacksonIgnoreFieldsExtractor$, null);

        forceJacksonIgnoreFields = forceJacksonIgnoreFieldsExtractor$.get();
        continue;
      }
      if ("forceEqualsInWithers".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ForceEqualsInWithersExtractor forceEqualsInWithersExtractor$ = new ForceEqualsInWithersExtractor();
        annotationValue$.accept(forceEqualsInWithersExtractor$, null);

        forceEqualsInWithers = forceEqualsInWithersExtractor$.get();
        continue;
      }
      if ("jacksonIntegration".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        JacksonIntegrationExtractor jacksonIntegrationExtractor$ = new JacksonIntegrationExtractor();
        annotationValue$.accept(jacksonIntegrationExtractor$, null);

        jacksonIntegration = jacksonIntegrationExtractor$.get();
        continue;
      }
      if ("weakInterning".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        WeakInterningExtractor weakInterningExtractor$ = new WeakInterningExtractor();
        annotationValue$.accept(weakInterningExtractor$, null);

        weakInterning = weakInterningExtractor$.get();
        continue;
      }
      if ("alwaysPublicInitializers".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AlwaysPublicInitializersExtractor alwaysPublicInitializersExtractor$ = new AlwaysPublicInitializersExtractor();
        annotationValue$.accept(alwaysPublicInitializersExtractor$, null);

        alwaysPublicInitializers = alwaysPublicInitializersExtractor$.get();
        continue;
      }
      if ("builderVisibility".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        BuilderVisibilityExtractor builderVisibilityExtractor$ = new BuilderVisibilityExtractor();
        annotationValue$.accept(builderVisibilityExtractor$, null);

        builderVisibility = builderVisibilityExtractor$.get();
        continue;
      }
      if ("builderVisibilityString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        BuilderVisibilityStringExtractor builderVisibilityStringExtractor$ = new BuilderVisibilityStringExtractor();
        annotationValue$.accept(builderVisibilityStringExtractor$, null);

        builderVisibilityString = builderVisibilityStringExtractor$.get();
        continue;
      }
      if ("throwForInvalidImmutableState".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ThrowForInvalidImmutableStateExtractor throwForInvalidImmutableStateExtractor$ = new ThrowForInvalidImmutableStateExtractor();
        annotationValue$.accept(throwForInvalidImmutableStateExtractor$, null);

        throwForInvalidImmutableState = throwForInvalidImmutableStateExtractor$.get();
        throwForInvalidImmutableStateName = throwForInvalidImmutableStateExtractor$.name();
        continue;
      }
      if ("throwForNullPointer".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ThrowForNullPointerExtractor throwForNullPointerExtractor$ = new ThrowForNullPointerExtractor();
        annotationValue$.accept(throwForNullPointerExtractor$, null);

        throwForNullPointer = throwForNullPointerExtractor$.get();
        throwForNullPointerName = throwForNullPointerExtractor$.name();
        continue;
      }
      if ("depluralize".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        DepluralizeExtractor depluralizeExtractor$ = new DepluralizeExtractor();
        annotationValue$.accept(depluralizeExtractor$, null);

        depluralize = depluralizeExtractor$.get();
        continue;
      }
      if ("depluralizeDictionary".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        DepluralizeDictionaryExtractor depluralizeDictionaryExtractor$ = new DepluralizeDictionaryExtractor();
        annotationValue$.accept(depluralizeDictionaryExtractor$, null);

        depluralizeDictionary = depluralizeDictionaryExtractor$.get();
        continue;
      }
      if ("immutableCopyOfRoutines".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        ImmutableCopyOfRoutinesExtractor immutableCopyOfRoutinesExtractor$ = new ImmutableCopyOfRoutinesExtractor();
        annotationValue$.accept(immutableCopyOfRoutinesExtractor$, null);

        immutableCopyOfRoutines = immutableCopyOfRoutinesExtractor$.get();
        immutableCopyOfRoutinesName = immutableCopyOfRoutinesExtractor$.name();
        continue;
      }
      if ("stagedBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        StagedBuilderExtractor stagedBuilderExtractor$ = new StagedBuilderExtractor();
        annotationValue$.accept(stagedBuilderExtractor$, null);

        stagedBuilder = stagedBuilderExtractor$.get();
        continue;
      }
      if ("builtinContainerAttributes".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        BuiltinContainerAttributesExtractor builtinContainerAttributesExtractor$ = new BuiltinContainerAttributesExtractor();
        annotationValue$.accept(builtinContainerAttributesExtractor$, null);

        builtinContainerAttributes = builtinContainerAttributesExtractor$.get();
        continue;
      }
      if ("beanFriendlyModifiables".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        BeanFriendlyModifiablesExtractor beanFriendlyModifiablesExtractor$ = new BeanFriendlyModifiablesExtractor();
        annotationValue$.accept(beanFriendlyModifiablesExtractor$, null);

        beanFriendlyModifiables = beanFriendlyModifiablesExtractor$.get();
        continue;
      }
      if ("allMandatoryParameters".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AllMandatoryParametersExtractor allMandatoryParametersExtractor$ = new AllMandatoryParametersExtractor();
        annotationValue$.accept(allMandatoryParametersExtractor$, null);

        allMandatoryParameters = allMandatoryParametersExtractor$.get();
        continue;
      }
      if ("redactedMask".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        RedactedMaskExtractor redactedMaskExtractor$ = new RedactedMaskExtractor();
        annotationValue$.accept(redactedMaskExtractor$, null);

        redactedMask = redactedMaskExtractor$.get();
        continue;
      }
      if ("attributeBuilderDetection".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AttributeBuilderDetectionExtractor attributeBuilderDetectionExtractor$ = new AttributeBuilderDetectionExtractor();
        annotationValue$.accept(attributeBuilderDetectionExtractor$, null);

        attributeBuilderDetection = attributeBuilderDetectionExtractor$.get();
        continue;
      }
      if ("transientDerivedFields".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        TransientDerivedFieldsExtractor transientDerivedFieldsExtractor$ = new TransientDerivedFieldsExtractor();
        annotationValue$.accept(transientDerivedFieldsExtractor$, null);

        transientDerivedFields = transientDerivedFieldsExtractor$.get();
        continue;
      }
      if ("finalInstanceFields".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        FinalInstanceFieldsExtractor finalInstanceFieldsExtractor$ = new FinalInstanceFieldsExtractor();
        annotationValue$.accept(finalInstanceFieldsExtractor$, null);

        finalInstanceFields = finalInstanceFieldsExtractor$.get();
        continue;
      }
      if ("attributeBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AttributeBuilderExtractor attributeBuilderExtractor$ = new AttributeBuilderExtractor();
        annotationValue$.accept(attributeBuilderExtractor$, null);

        attributeBuilder = attributeBuilderExtractor$.get();
        continue;
      }
      if ("getBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        GetBuilderExtractor getBuilderExtractor$ = new GetBuilderExtractor();
        annotationValue$.accept(getBuilderExtractor$, null);

        getBuilder = getBuilderExtractor$.get();
        continue;
      }
      if ("setBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        SetBuilderExtractor setBuilderExtractor$ = new SetBuilderExtractor();
        annotationValue$.accept(setBuilderExtractor$, null);

        setBuilder = setBuilderExtractor$.get();
        continue;
      }
      if ("addBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AddBuilderExtractor addBuilderExtractor$ = new AddBuilderExtractor();
        annotationValue$.accept(addBuilderExtractor$, null);

        addBuilder = addBuilderExtractor$.get();
        continue;
      }
      if ("addAllBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AddAllBuilderExtractor addAllBuilderExtractor$ = new AddAllBuilderExtractor();
        annotationValue$.accept(addAllBuilderExtractor$, null);

        addAllBuilder = addAllBuilderExtractor$.get();
        continue;
      }
      if ("getBuilders".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        GetBuildersExtractor getBuildersExtractor$ = new GetBuildersExtractor();
        annotationValue$.accept(getBuildersExtractor$, null);

        getBuilders = getBuildersExtractor$.get();
        continue;
      }
      if ("nullableAnnotation".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        NullableAnnotationExtractor nullableAnnotationExtractor$ = new NullableAnnotationExtractor();
        annotationValue$.accept(nullableAnnotationExtractor$, null);

        nullableAnnotation = nullableAnnotationExtractor$.get();
        continue;
      }
      if ("limitStringLengthInToString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        LimitStringLengthInToStringExtractor limitStringLengthInToStringExtractor$ = new LimitStringLengthInToStringExtractor();
        annotationValue$.accept(limitStringLengthInToStringExtractor$, null);

        limitStringLengthInToString = limitStringLengthInToStringExtractor$.get();
        continue;
      }
      if ("allowedClasspathAnnotations".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        AllowedClasspathAnnotationsExtractor allowedClasspathAnnotationsExtractor$ = new AllowedClasspathAnnotationsExtractor();
        annotationValue$.accept(allowedClasspathAnnotationsExtractor$, null);

        allowedClasspathAnnotations = allowedClasspathAnnotationsExtractor$.get();
        allowedClasspathAnnotationsName = allowedClasspathAnnotationsExtractor$.name();
        continue;
      }
      if ("fallbackNullableAnnotation".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        FallbackNullableAnnotationExtractor fallbackNullableAnnotationExtractor$ = new FallbackNullableAnnotationExtractor();
        annotationValue$.accept(fallbackNullableAnnotationExtractor$, null);

        fallbackNullableAnnotation = fallbackNullableAnnotationExtractor$.get();
        fallbackNullableAnnotationName = fallbackNullableAnnotationExtractor$.name();
        continue;
      }
      if ("jakarta".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Style");
        }
        JakartaExtractor jakartaExtractor$ = new JakartaExtractor();
        annotationValue$.accept(jakartaExtractor$, null);

        jakarta = jakartaExtractor$.get();
        continue;
      }
    }
    this.get = Preconditions.checkNotNull(get, "default attribute 'get'");
    this.init = Preconditions.checkNotNull(init, "default attribute 'init'");
    this.with = Preconditions.checkNotNull(with, "default attribute 'with'");
    this.withUnaryOperator = Preconditions.checkNotNull(withUnaryOperator, "default attribute 'withUnaryOperator'");
    this.add = Preconditions.checkNotNull(add, "default attribute 'add'");
    this.addAll = Preconditions.checkNotNull(addAll, "default attribute 'addAll'");
    this.put = Preconditions.checkNotNull(put, "default attribute 'put'");
    this.putAll = Preconditions.checkNotNull(putAll, "default attribute 'putAll'");
    this.copyOf = Preconditions.checkNotNull(copyOf, "default attribute 'copyOf'");
    this.of = Preconditions.checkNotNull(of, "default attribute 'of'");
    this.instance = Preconditions.checkNotNull(instance, "default attribute 'instance'");
    this.builder = Preconditions.checkNotNull(builder, "default attribute 'builder'");
    this.newBuilder = Preconditions.checkNotNull(newBuilder, "default attribute 'newBuilder'");
    this.from = Preconditions.checkNotNull(from, "default attribute 'from'");
    this.build = Preconditions.checkNotNull(build, "default attribute 'build'");
    this.buildOrThrow = Preconditions.checkNotNull(buildOrThrow, "default attribute 'buildOrThrow'");
    this.canBuild = Preconditions.checkNotNull(canBuild, "default attribute 'canBuild'");
    this.toBuilder = Preconditions.checkNotNull(toBuilder, "default attribute 'toBuilder'");
    this.underrideHashCode = Preconditions.checkNotNull(underrideHashCode, "default attribute 'underrideHashCode'");
    this.underrideEquals = Preconditions.checkNotNull(underrideEquals, "default attribute 'underrideEquals'");
    this.underrideToString = Preconditions.checkNotNull(underrideToString, "default attribute 'underrideToString'");
    this.delegateToString = Preconditions.checkNotNull(delegateToString, "default attribute 'delegateToString'");
    this.includeHashCode = Preconditions.checkNotNull(includeHashCode, "default attribute 'includeHashCode'");
    this.isInitialized = Preconditions.checkNotNull(isInitialized, "default attribute 'isInitialized'");
    this.isSet = Preconditions.checkNotNull(isSet, "default attribute 'isSet'");
    this.set = Preconditions.checkNotNull(set, "default attribute 'set'");
    this.unset = Preconditions.checkNotNull(unset, "default attribute 'unset'");
    this.clear = Preconditions.checkNotNull(clear, "default attribute 'clear'");
    this.create = Preconditions.checkNotNull(create, "default attribute 'create'");
    this.toImmutable = Preconditions.checkNotNull(toImmutable, "default attribute 'toImmutable'");
    this.typeBuilder = Preconditions.checkNotNull(typeBuilder, "default attribute 'typeBuilder'");
    this.typeInnerBuilder = Preconditions.checkNotNull(typeInnerBuilder, "default attribute 'typeInnerBuilder'");
    this.typeAbstract = Preconditions.checkNotNull(typeAbstract, "default attribute 'typeAbstract'");
    this.typeImmutable = Preconditions.checkNotNull(typeImmutable, "default attribute 'typeImmutable'");
    this.typeImmutableEnclosing = Preconditions.checkNotNull(typeImmutableEnclosing, "default attribute 'typeImmutableEnclosing'");
    this.typeImmutableNested = Preconditions.checkNotNull(typeImmutableNested, "default attribute 'typeImmutableNested'");
    this.typeModifiable = Preconditions.checkNotNull(typeModifiable, "default attribute 'typeModifiable'");
    this.typeInnerModifiable = Preconditions.checkNotNull(typeInnerModifiable, "default attribute 'typeInnerModifiable'");
    this.typeWith = Preconditions.checkNotNull(typeWith, "default attribute 'typeWith'");
    this.packageGenerated = Preconditions.checkNotNull(packageGenerated, "default attribute 'packageGenerated'");
    this.defaults = Preconditions.checkNotNull(defaults, "default attribute 'defaults'");
    this.defaultsMirror = Preconditions.checkNotNull(defaultsMirror, "default attribute 'defaults'");
    this.strictBuilder = strictBuilder;
    this.strictModifiable = strictModifiable;
    this.validationMethod = Preconditions.checkNotNull(validationMethod, "default attribute 'validationMethod'");
    this.allParameters = allParameters;
    this.defaultAsDefault = defaultAsDefault;
    this.headerComments = headerComments;
    this.jdkOnly = jdkOnly;
    this.jdk9Collections = jdk9Collections;
    this.passAnnotations = Preconditions.checkNotNull(passAnnotations, "default attribute 'passAnnotations'");
    this.passAnnotationsName = Preconditions.checkNotNull(passAnnotationsName, "default attribute 'passAnnotations'");
    this.additionalJsonAnnotations = Preconditions.checkNotNull(additionalJsonAnnotations, "default attribute 'additionalJsonAnnotations'");
    this.additionalJsonAnnotationsName = Preconditions.checkNotNull(additionalJsonAnnotationsName, "default attribute 'additionalJsonAnnotations'");
    this.visibility = Preconditions.checkNotNull(visibility, "default attribute 'visibility'");
    this.visibilityString = Preconditions.checkNotNull(visibilityString, "default attribute 'visibilityString'");
    this.optionalAcceptNullable = optionalAcceptNullable;
    this.generateSuppressAllWarnings = generateSuppressAllWarnings;
    this.privateNoargConstructor = privateNoargConstructor;
    this.protectedNoargConstructor = protectedNoargConstructor;
    this.attributelessSingleton = attributelessSingleton;
    this.unsafeDefaultAndDerived = unsafeDefaultAndDerived;
    this.clearBuilder = clearBuilder;
    this.deferCollectionAllocation = deferCollectionAllocation;
    this.deepImmutablesDetection = deepImmutablesDetection;
    this.overshadowImplementation = overshadowImplementation;
    this.implementationNestedInBuilder = implementationNestedInBuilder;
    this.forceJacksonPropertyNames = forceJacksonPropertyNames;
    this.forceJacksonIgnoreFields = forceJacksonIgnoreFields;
    this.forceEqualsInWithers = forceEqualsInWithers;
    this.jacksonIntegration = jacksonIntegration;
    this.weakInterning = weakInterning;
    this.alwaysPublicInitializers = alwaysPublicInitializers;
    this.builderVisibility = Preconditions.checkNotNull(builderVisibility, "default attribute 'builderVisibility'");
    this.builderVisibilityString = Preconditions.checkNotNull(builderVisibilityString, "default attribute 'builderVisibilityString'");
    this.throwForInvalidImmutableState = Preconditions.checkNotNull(throwForInvalidImmutableState, "default attribute 'throwForInvalidImmutableState'");
    this.throwForInvalidImmutableStateName = Preconditions.checkNotNull(throwForInvalidImmutableStateName, "default attribute 'throwForInvalidImmutableState'");
    this.throwForNullPointer = Preconditions.checkNotNull(throwForNullPointer, "default attribute 'throwForNullPointer'");
    this.throwForNullPointerName = Preconditions.checkNotNull(throwForNullPointerName, "default attribute 'throwForNullPointer'");
    this.depluralize = depluralize;
    this.depluralizeDictionary = Preconditions.checkNotNull(depluralizeDictionary, "default attribute 'depluralizeDictionary'");
    this.immutableCopyOfRoutines = Preconditions.checkNotNull(immutableCopyOfRoutines, "default attribute 'immutableCopyOfRoutines'");
    this.immutableCopyOfRoutinesName = Preconditions.checkNotNull(immutableCopyOfRoutinesName, "default attribute 'immutableCopyOfRoutines'");
    this.stagedBuilder = stagedBuilder;
    this.builtinContainerAttributes = builtinContainerAttributes;
    this.beanFriendlyModifiables = beanFriendlyModifiables;
    this.allMandatoryParameters = allMandatoryParameters;
    this.redactedMask = Preconditions.checkNotNull(redactedMask, "default attribute 'redactedMask'");
    this.attributeBuilderDetection = attributeBuilderDetection;
    this.transientDerivedFields = transientDerivedFields;
    this.finalInstanceFields = finalInstanceFields;
    this.attributeBuilder = Preconditions.checkNotNull(attributeBuilder, "default attribute 'attributeBuilder'");
    this.getBuilder = Preconditions.checkNotNull(getBuilder, "default attribute 'getBuilder'");
    this.setBuilder = Preconditions.checkNotNull(setBuilder, "default attribute 'setBuilder'");
    this.addBuilder = Preconditions.checkNotNull(addBuilder, "default attribute 'addBuilder'");
    this.addAllBuilder = Preconditions.checkNotNull(addAllBuilder, "default attribute 'addAllBuilder'");
    this.getBuilders = Preconditions.checkNotNull(getBuilders, "default attribute 'getBuilders'");
    this.nullableAnnotation = Preconditions.checkNotNull(nullableAnnotation, "default attribute 'nullableAnnotation'");
    this.limitStringLengthInToString = limitStringLengthInToString;
    this.allowedClasspathAnnotations = Preconditions.checkNotNull(allowedClasspathAnnotations, "default attribute 'allowedClasspathAnnotations'");
    this.allowedClasspathAnnotationsName = Preconditions.checkNotNull(allowedClasspathAnnotationsName, "default attribute 'allowedClasspathAnnotations'");
    this.fallbackNullableAnnotation = Preconditions.checkNotNull(fallbackNullableAnnotation, "default attribute 'fallbackNullableAnnotation'");
    this.fallbackNullableAnnotationName = Preconditions.checkNotNull(fallbackNullableAnnotationName, "default attribute 'fallbackNullableAnnotation'");
    this.jakarta = jakarta;
  }

  private StyleMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String[] get = null;
    String init = null;
    String with = null;
    String withUnaryOperator = null;
    String add = null;
    String addAll = null;
    String put = null;
    String putAll = null;
    String copyOf = null;
    String of = null;
    String instance = null;
    String builder = null;
    String newBuilder = null;
    String from = null;
    String build = null;
    String buildOrThrow = null;
    String canBuild = null;
    String toBuilder = null;
    String underrideHashCode = null;
    String underrideEquals = null;
    String underrideToString = null;
    String delegateToString = null;
    String includeHashCode = null;
    String isInitialized = null;
    String isSet = null;
    String set = null;
    String unset = null;
    String clear = null;
    String create = null;
    String toImmutable = null;
    String typeBuilder = null;
    String typeInnerBuilder = null;
    String[] typeAbstract = null;
    String typeImmutable = null;
    String typeImmutableEnclosing = null;
    String typeImmutableNested = null;
    String typeModifiable = null;
    String typeInnerModifiable = null;
    String typeWith = null;
    String packageGenerated = null;
    AnnotationMirror defaults = null;
    ImmutableMirror defaultsMirror = null;
    boolean strictBuilder = false;
    boolean strictModifiable = false;
    ValueMirrors.Style.ValidationMethod validationMethod = null;
    boolean allParameters = false;
    boolean defaultAsDefault = false;
    boolean headerComments = false;
    boolean jdkOnly = false;
    boolean jdk9Collections = false;
    TypeMirror[] passAnnotations = null;
    String[] passAnnotationsName = null;
    TypeMirror[] additionalJsonAnnotations = null;
    String[] additionalJsonAnnotationsName = null;
    ValueMirrors.Style.ImplementationVisibility visibility = null;
    String visibilityString = null;
    boolean optionalAcceptNullable = false;
    boolean generateSuppressAllWarnings = false;
    boolean privateNoargConstructor = false;
    boolean protectedNoargConstructor = false;
    boolean attributelessSingleton = false;
    boolean unsafeDefaultAndDerived = false;
    boolean clearBuilder = false;
    boolean deferCollectionAllocation = false;
    boolean deepImmutablesDetection = false;
    boolean overshadowImplementation = false;
    boolean implementationNestedInBuilder = false;
    boolean forceJacksonPropertyNames = false;
    boolean forceJacksonIgnoreFields = false;
    boolean forceEqualsInWithers = false;
    boolean jacksonIntegration = false;
    boolean weakInterning = false;
    boolean alwaysPublicInitializers = false;
    ValueMirrors.Style.BuilderVisibility builderVisibility = null;
    String builderVisibilityString = null;
    TypeMirror throwForInvalidImmutableState = null;
    String throwForInvalidImmutableStateName = null;
    TypeMirror throwForNullPointer = null;
    String throwForNullPointerName = null;
    boolean depluralize = false;
    String[] depluralizeDictionary = null;
    TypeMirror[] immutableCopyOfRoutines = null;
    String[] immutableCopyOfRoutinesName = null;
    boolean stagedBuilder = false;
    boolean builtinContainerAttributes = false;
    boolean beanFriendlyModifiables = false;
    boolean allMandatoryParameters = false;
    String redactedMask = null;
    boolean attributeBuilderDetection = false;
    boolean transientDerivedFields = false;
    boolean finalInstanceFields = false;
    String[] attributeBuilder = null;
    String getBuilder = null;
    String setBuilder = null;
    String addBuilder = null;
    String addAllBuilder = null;
    String getBuilders = null;
    String nullableAnnotation = null;
    int limitStringLengthInToString = 0;
    TypeMirror[] allowedClasspathAnnotations = null;
    String[] allowedClasspathAnnotationsName = null;
    TypeMirror fallbackNullableAnnotation = null;
    String fallbackNullableAnnotationName = null;
    boolean jakarta = false;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("get".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'get' attribute of @Style");
        }
        GetExtractor getExtractor$ = new GetExtractor();
        annotationValue$.accept(getExtractor$, null);

        get = getExtractor$.get();
        continue;
      }
      if ("init".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'init' attribute of @Style");
        }
        InitExtractor initExtractor$ = new InitExtractor();
        annotationValue$.accept(initExtractor$, null);

        init = initExtractor$.get();
        continue;
      }
      if ("with".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'with' attribute of @Style");
        }
        WithExtractor withExtractor$ = new WithExtractor();
        annotationValue$.accept(withExtractor$, null);

        with = withExtractor$.get();
        continue;
      }
      if ("withUnaryOperator".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'withUnaryOperator' attribute of @Style");
        }
        WithUnaryOperatorExtractor withUnaryOperatorExtractor$ = new WithUnaryOperatorExtractor();
        annotationValue$.accept(withUnaryOperatorExtractor$, null);

        withUnaryOperator = withUnaryOperatorExtractor$.get();
        continue;
      }
      if ("add".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'add' attribute of @Style");
        }
        AddExtractor addExtractor$ = new AddExtractor();
        annotationValue$.accept(addExtractor$, null);

        add = addExtractor$.get();
        continue;
      }
      if ("addAll".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'addAll' attribute of @Style");
        }
        AddAllExtractor addAllExtractor$ = new AddAllExtractor();
        annotationValue$.accept(addAllExtractor$, null);

        addAll = addAllExtractor$.get();
        continue;
      }
      if ("put".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'put' attribute of @Style");
        }
        PutExtractor putExtractor$ = new PutExtractor();
        annotationValue$.accept(putExtractor$, null);

        put = putExtractor$.get();
        continue;
      }
      if ("putAll".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'putAll' attribute of @Style");
        }
        PutAllExtractor putAllExtractor$ = new PutAllExtractor();
        annotationValue$.accept(putAllExtractor$, null);

        putAll = putAllExtractor$.get();
        continue;
      }
      if ("copyOf".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'copyOf' attribute of @Style");
        }
        CopyOfExtractor copyOfExtractor$ = new CopyOfExtractor();
        annotationValue$.accept(copyOfExtractor$, null);

        copyOf = copyOfExtractor$.get();
        continue;
      }
      if ("of".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'of' attribute of @Style");
        }
        OfExtractor ofExtractor$ = new OfExtractor();
        annotationValue$.accept(ofExtractor$, null);

        of = ofExtractor$.get();
        continue;
      }
      if ("instance".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'instance' attribute of @Style");
        }
        InstanceExtractor instanceExtractor$ = new InstanceExtractor();
        annotationValue$.accept(instanceExtractor$, null);

        instance = instanceExtractor$.get();
        continue;
      }
      if ("builder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'builder' attribute of @Style");
        }
        BuilderExtractor builderExtractor$ = new BuilderExtractor();
        annotationValue$.accept(builderExtractor$, null);

        builder = builderExtractor$.get();
        continue;
      }
      if ("newBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'newBuilder' attribute of @Style");
        }
        NewBuilderExtractor newBuilderExtractor$ = new NewBuilderExtractor();
        annotationValue$.accept(newBuilderExtractor$, null);

        newBuilder = newBuilderExtractor$.get();
        continue;
      }
      if ("from".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'from' attribute of @Style");
        }
        FromExtractor fromExtractor$ = new FromExtractor();
        annotationValue$.accept(fromExtractor$, null);

        from = fromExtractor$.get();
        continue;
      }
      if ("build".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'build' attribute of @Style");
        }
        BuildExtractor buildExtractor$ = new BuildExtractor();
        annotationValue$.accept(buildExtractor$, null);

        build = buildExtractor$.get();
        continue;
      }
      if ("buildOrThrow".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'buildOrThrow' attribute of @Style");
        }
        BuildOrThrowExtractor buildOrThrowExtractor$ = new BuildOrThrowExtractor();
        annotationValue$.accept(buildOrThrowExtractor$, null);

        buildOrThrow = buildOrThrowExtractor$.get();
        continue;
      }
      if ("canBuild".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'canBuild' attribute of @Style");
        }
        CanBuildExtractor canBuildExtractor$ = new CanBuildExtractor();
        annotationValue$.accept(canBuildExtractor$, null);

        canBuild = canBuildExtractor$.get();
        continue;
      }
      if ("toBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'toBuilder' attribute of @Style");
        }
        ToBuilderExtractor toBuilderExtractor$ = new ToBuilderExtractor();
        annotationValue$.accept(toBuilderExtractor$, null);

        toBuilder = toBuilderExtractor$.get();
        continue;
      }
      if ("underrideHashCode".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'underrideHashCode' attribute of @Style");
        }
        UnderrideHashCodeExtractor underrideHashCodeExtractor$ = new UnderrideHashCodeExtractor();
        annotationValue$.accept(underrideHashCodeExtractor$, null);

        underrideHashCode = underrideHashCodeExtractor$.get();
        continue;
      }
      if ("underrideEquals".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'underrideEquals' attribute of @Style");
        }
        UnderrideEqualsExtractor underrideEqualsExtractor$ = new UnderrideEqualsExtractor();
        annotationValue$.accept(underrideEqualsExtractor$, null);

        underrideEquals = underrideEqualsExtractor$.get();
        continue;
      }
      if ("underrideToString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'underrideToString' attribute of @Style");
        }
        UnderrideToStringExtractor underrideToStringExtractor$ = new UnderrideToStringExtractor();
        annotationValue$.accept(underrideToStringExtractor$, null);

        underrideToString = underrideToStringExtractor$.get();
        continue;
      }
      if ("delegateToString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'delegateToString' attribute of @Style");
        }
        DelegateToStringExtractor delegateToStringExtractor$ = new DelegateToStringExtractor();
        annotationValue$.accept(delegateToStringExtractor$, null);

        delegateToString = delegateToStringExtractor$.get();
        continue;
      }
      if ("includeHashCode".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'includeHashCode' attribute of @Style");
        }
        IncludeHashCodeExtractor includeHashCodeExtractor$ = new IncludeHashCodeExtractor();
        annotationValue$.accept(includeHashCodeExtractor$, null);

        includeHashCode = includeHashCodeExtractor$.get();
        continue;
      }
      if ("isInitialized".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'isInitialized' attribute of @Style");
        }
        IsInitializedExtractor isInitializedExtractor$ = new IsInitializedExtractor();
        annotationValue$.accept(isInitializedExtractor$, null);

        isInitialized = isInitializedExtractor$.get();
        continue;
      }
      if ("isSet".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'isSet' attribute of @Style");
        }
        IsSetExtractor isSetExtractor$ = new IsSetExtractor();
        annotationValue$.accept(isSetExtractor$, null);

        isSet = isSetExtractor$.get();
        continue;
      }
      if ("set".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'set' attribute of @Style");
        }
        SetExtractor setExtractor$ = new SetExtractor();
        annotationValue$.accept(setExtractor$, null);

        set = setExtractor$.get();
        continue;
      }
      if ("unset".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'unset' attribute of @Style");
        }
        UnsetExtractor unsetExtractor$ = new UnsetExtractor();
        annotationValue$.accept(unsetExtractor$, null);

        unset = unsetExtractor$.get();
        continue;
      }
      if ("clear".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'clear' attribute of @Style");
        }
        ClearExtractor clearExtractor$ = new ClearExtractor();
        annotationValue$.accept(clearExtractor$, null);

        clear = clearExtractor$.get();
        continue;
      }
      if ("create".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'create' attribute of @Style");
        }
        CreateExtractor createExtractor$ = new CreateExtractor();
        annotationValue$.accept(createExtractor$, null);

        create = createExtractor$.get();
        continue;
      }
      if ("toImmutable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'toImmutable' attribute of @Style");
        }
        ToImmutableExtractor toImmutableExtractor$ = new ToImmutableExtractor();
        annotationValue$.accept(toImmutableExtractor$, null);

        toImmutable = toImmutableExtractor$.get();
        continue;
      }
      if ("typeBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeBuilder' attribute of @Style");
        }
        TypeBuilderExtractor typeBuilderExtractor$ = new TypeBuilderExtractor();
        annotationValue$.accept(typeBuilderExtractor$, null);

        typeBuilder = typeBuilderExtractor$.get();
        continue;
      }
      if ("typeInnerBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeInnerBuilder' attribute of @Style");
        }
        TypeInnerBuilderExtractor typeInnerBuilderExtractor$ = new TypeInnerBuilderExtractor();
        annotationValue$.accept(typeInnerBuilderExtractor$, null);

        typeInnerBuilder = typeInnerBuilderExtractor$.get();
        continue;
      }
      if ("typeAbstract".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeAbstract' attribute of @Style");
        }
        TypeAbstractExtractor typeAbstractExtractor$ = new TypeAbstractExtractor();
        annotationValue$.accept(typeAbstractExtractor$, null);

        typeAbstract = typeAbstractExtractor$.get();
        continue;
      }
      if ("typeImmutable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeImmutable' attribute of @Style");
        }
        TypeImmutableExtractor typeImmutableExtractor$ = new TypeImmutableExtractor();
        annotationValue$.accept(typeImmutableExtractor$, null);

        typeImmutable = typeImmutableExtractor$.get();
        continue;
      }
      if ("typeImmutableEnclosing".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeImmutableEnclosing' attribute of @Style");
        }
        TypeImmutableEnclosingExtractor typeImmutableEnclosingExtractor$ = new TypeImmutableEnclosingExtractor();
        annotationValue$.accept(typeImmutableEnclosingExtractor$, null);

        typeImmutableEnclosing = typeImmutableEnclosingExtractor$.get();
        continue;
      }
      if ("typeImmutableNested".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeImmutableNested' attribute of @Style");
        }
        TypeImmutableNestedExtractor typeImmutableNestedExtractor$ = new TypeImmutableNestedExtractor();
        annotationValue$.accept(typeImmutableNestedExtractor$, null);

        typeImmutableNested = typeImmutableNestedExtractor$.get();
        continue;
      }
      if ("typeModifiable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeModifiable' attribute of @Style");
        }
        TypeModifiableExtractor typeModifiableExtractor$ = new TypeModifiableExtractor();
        annotationValue$.accept(typeModifiableExtractor$, null);

        typeModifiable = typeModifiableExtractor$.get();
        continue;
      }
      if ("typeInnerModifiable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeInnerModifiable' attribute of @Style");
        }
        TypeInnerModifiableExtractor typeInnerModifiableExtractor$ = new TypeInnerModifiableExtractor();
        annotationValue$.accept(typeInnerModifiableExtractor$, null);

        typeInnerModifiable = typeInnerModifiableExtractor$.get();
        continue;
      }
      if ("typeWith".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeWith' attribute of @Style");
        }
        TypeWithExtractor typeWithExtractor$ = new TypeWithExtractor();
        annotationValue$.accept(typeWithExtractor$, null);

        typeWith = typeWithExtractor$.get();
        continue;
      }
      if ("packageGenerated".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'packageGenerated' attribute of @Style");
        }
        PackageGeneratedExtractor packageGeneratedExtractor$ = new PackageGeneratedExtractor();
        annotationValue$.accept(packageGeneratedExtractor$, null);

        packageGenerated = packageGeneratedExtractor$.get();
        continue;
      }
      if ("defaults".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'defaults' attribute of @Style");
        }
        DefaultsExtractor defaultsExtractor$ = new DefaultsExtractor();
        annotationValue$.accept(defaultsExtractor$, null);

        defaults = defaultsExtractor$.get();
        defaultsMirror = defaultsExtractor$.mirror();
        continue;
      }
      if ("strictBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'strictBuilder' attribute of @Style");
        }
        StrictBuilderExtractor strictBuilderExtractor$ = new StrictBuilderExtractor();
        annotationValue$.accept(strictBuilderExtractor$, null);

        strictBuilder = strictBuilderExtractor$.get();
        continue;
      }
      if ("strictModifiable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'strictModifiable' attribute of @Style");
        }
        StrictModifiableExtractor strictModifiableExtractor$ = new StrictModifiableExtractor();
        annotationValue$.accept(strictModifiableExtractor$, null);

        strictModifiable = strictModifiableExtractor$.get();
        continue;
      }
      if ("validationMethod".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'validationMethod' attribute of @Style");
        }
        ValidationMethodExtractor validationMethodExtractor$ = new ValidationMethodExtractor();
        annotationValue$.accept(validationMethodExtractor$, null);

        validationMethod = validationMethodExtractor$.get();
        continue;
      }
      if ("allParameters".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'allParameters' attribute of @Style");
        }
        AllParametersExtractor allParametersExtractor$ = new AllParametersExtractor();
        annotationValue$.accept(allParametersExtractor$, null);

        allParameters = allParametersExtractor$.get();
        continue;
      }
      if ("defaultAsDefault".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'defaultAsDefault' attribute of @Style");
        }
        DefaultAsDefaultExtractor defaultAsDefaultExtractor$ = new DefaultAsDefaultExtractor();
        annotationValue$.accept(defaultAsDefaultExtractor$, null);

        defaultAsDefault = defaultAsDefaultExtractor$.get();
        continue;
      }
      if ("headerComments".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'headerComments' attribute of @Style");
        }
        HeaderCommentsExtractor headerCommentsExtractor$ = new HeaderCommentsExtractor();
        annotationValue$.accept(headerCommentsExtractor$, null);

        headerComments = headerCommentsExtractor$.get();
        continue;
      }
      if ("jdkOnly".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'jdkOnly' attribute of @Style");
        }
        JdkOnlyExtractor jdkOnlyExtractor$ = new JdkOnlyExtractor();
        annotationValue$.accept(jdkOnlyExtractor$, null);

        jdkOnly = jdkOnlyExtractor$.get();
        continue;
      }
      if ("jdk9Collections".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'jdk9Collections' attribute of @Style");
        }
        Jdk9CollectionsExtractor jdk9CollectionsExtractor$ = new Jdk9CollectionsExtractor();
        annotationValue$.accept(jdk9CollectionsExtractor$, null);

        jdk9Collections = jdk9CollectionsExtractor$.get();
        continue;
      }
      if ("passAnnotations".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'passAnnotations' attribute of @Style");
        }
        PassAnnotationsExtractor passAnnotationsExtractor$ = new PassAnnotationsExtractor();
        annotationValue$.accept(passAnnotationsExtractor$, null);

        passAnnotations = passAnnotationsExtractor$.get();
        passAnnotationsName = passAnnotationsExtractor$.name();
        continue;
      }
      if ("additionalJsonAnnotations".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'additionalJsonAnnotations' attribute of @Style");
        }
        AdditionalJsonAnnotationsExtractor additionalJsonAnnotationsExtractor$ = new AdditionalJsonAnnotationsExtractor();
        annotationValue$.accept(additionalJsonAnnotationsExtractor$, null);

        additionalJsonAnnotations = additionalJsonAnnotationsExtractor$.get();
        additionalJsonAnnotationsName = additionalJsonAnnotationsExtractor$.name();
        continue;
      }
      if ("visibility".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'visibility' attribute of @Style");
        }
        VisibilityExtractor visibilityExtractor$ = new VisibilityExtractor();
        annotationValue$.accept(visibilityExtractor$, null);

        visibility = visibilityExtractor$.get();
        continue;
      }
      if ("visibilityString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'visibilityString' attribute of @Style");
        }
        VisibilityStringExtractor visibilityStringExtractor$ = new VisibilityStringExtractor();
        annotationValue$.accept(visibilityStringExtractor$, null);

        visibilityString = visibilityStringExtractor$.get();
        continue;
      }
      if ("optionalAcceptNullable".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'optionalAcceptNullable' attribute of @Style");
        }
        OptionalAcceptNullableExtractor optionalAcceptNullableExtractor$ = new OptionalAcceptNullableExtractor();
        annotationValue$.accept(optionalAcceptNullableExtractor$, null);

        optionalAcceptNullable = optionalAcceptNullableExtractor$.get();
        continue;
      }
      if ("generateSuppressAllWarnings".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'generateSuppressAllWarnings' attribute of @Style");
        }
        GenerateSuppressAllWarningsExtractor generateSuppressAllWarningsExtractor$ = new GenerateSuppressAllWarningsExtractor();
        annotationValue$.accept(generateSuppressAllWarningsExtractor$, null);

        generateSuppressAllWarnings = generateSuppressAllWarningsExtractor$.get();
        continue;
      }
      if ("privateNoargConstructor".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'privateNoargConstructor' attribute of @Style");
        }
        PrivateNoargConstructorExtractor privateNoargConstructorExtractor$ = new PrivateNoargConstructorExtractor();
        annotationValue$.accept(privateNoargConstructorExtractor$, null);

        privateNoargConstructor = privateNoargConstructorExtractor$.get();
        continue;
      }
      if ("protectedNoargConstructor".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'protectedNoargConstructor' attribute of @Style");
        }
        ProtectedNoargConstructorExtractor protectedNoargConstructorExtractor$ = new ProtectedNoargConstructorExtractor();
        annotationValue$.accept(protectedNoargConstructorExtractor$, null);

        protectedNoargConstructor = protectedNoargConstructorExtractor$.get();
        continue;
      }
      if ("attributelessSingleton".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'attributelessSingleton' attribute of @Style");
        }
        AttributelessSingletonExtractor attributelessSingletonExtractor$ = new AttributelessSingletonExtractor();
        annotationValue$.accept(attributelessSingletonExtractor$, null);

        attributelessSingleton = attributelessSingletonExtractor$.get();
        continue;
      }
      if ("unsafeDefaultAndDerived".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'unsafeDefaultAndDerived' attribute of @Style");
        }
        UnsafeDefaultAndDerivedExtractor unsafeDefaultAndDerivedExtractor$ = new UnsafeDefaultAndDerivedExtractor();
        annotationValue$.accept(unsafeDefaultAndDerivedExtractor$, null);

        unsafeDefaultAndDerived = unsafeDefaultAndDerivedExtractor$.get();
        continue;
      }
      if ("clearBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'clearBuilder' attribute of @Style");
        }
        ClearBuilderExtractor clearBuilderExtractor$ = new ClearBuilderExtractor();
        annotationValue$.accept(clearBuilderExtractor$, null);

        clearBuilder = clearBuilderExtractor$.get();
        continue;
      }
      if ("deferCollectionAllocation".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'deferCollectionAllocation' attribute of @Style");
        }
        DeferCollectionAllocationExtractor deferCollectionAllocationExtractor$ = new DeferCollectionAllocationExtractor();
        annotationValue$.accept(deferCollectionAllocationExtractor$, null);

        deferCollectionAllocation = deferCollectionAllocationExtractor$.get();
        continue;
      }
      if ("deepImmutablesDetection".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'deepImmutablesDetection' attribute of @Style");
        }
        DeepImmutablesDetectionExtractor deepImmutablesDetectionExtractor$ = new DeepImmutablesDetectionExtractor();
        annotationValue$.accept(deepImmutablesDetectionExtractor$, null);

        deepImmutablesDetection = deepImmutablesDetectionExtractor$.get();
        continue;
      }
      if ("overshadowImplementation".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'overshadowImplementation' attribute of @Style");
        }
        OvershadowImplementationExtractor overshadowImplementationExtractor$ = new OvershadowImplementationExtractor();
        annotationValue$.accept(overshadowImplementationExtractor$, null);

        overshadowImplementation = overshadowImplementationExtractor$.get();
        continue;
      }
      if ("implementationNestedInBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'implementationNestedInBuilder' attribute of @Style");
        }
        ImplementationNestedInBuilderExtractor implementationNestedInBuilderExtractor$ = new ImplementationNestedInBuilderExtractor();
        annotationValue$.accept(implementationNestedInBuilderExtractor$, null);

        implementationNestedInBuilder = implementationNestedInBuilderExtractor$.get();
        continue;
      }
      if ("forceJacksonPropertyNames".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'forceJacksonPropertyNames' attribute of @Style");
        }
        ForceJacksonPropertyNamesExtractor forceJacksonPropertyNamesExtractor$ = new ForceJacksonPropertyNamesExtractor();
        annotationValue$.accept(forceJacksonPropertyNamesExtractor$, null);

        forceJacksonPropertyNames = forceJacksonPropertyNamesExtractor$.get();
        continue;
      }
      if ("forceJacksonIgnoreFields".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'forceJacksonIgnoreFields' attribute of @Style");
        }
        ForceJacksonIgnoreFieldsExtractor forceJacksonIgnoreFieldsExtractor$ = new ForceJacksonIgnoreFieldsExtractor();
        annotationValue$.accept(forceJacksonIgnoreFieldsExtractor$, null);

        forceJacksonIgnoreFields = forceJacksonIgnoreFieldsExtractor$.get();
        continue;
      }
      if ("forceEqualsInWithers".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'forceEqualsInWithers' attribute of @Style");
        }
        ForceEqualsInWithersExtractor forceEqualsInWithersExtractor$ = new ForceEqualsInWithersExtractor();
        annotationValue$.accept(forceEqualsInWithersExtractor$, null);

        forceEqualsInWithers = forceEqualsInWithersExtractor$.get();
        continue;
      }
      if ("jacksonIntegration".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'jacksonIntegration' attribute of @Style");
        }
        JacksonIntegrationExtractor jacksonIntegrationExtractor$ = new JacksonIntegrationExtractor();
        annotationValue$.accept(jacksonIntegrationExtractor$, null);

        jacksonIntegration = jacksonIntegrationExtractor$.get();
        continue;
      }
      if ("weakInterning".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'weakInterning' attribute of @Style");
        }
        WeakInterningExtractor weakInterningExtractor$ = new WeakInterningExtractor();
        annotationValue$.accept(weakInterningExtractor$, null);

        weakInterning = weakInterningExtractor$.get();
        continue;
      }
      if ("alwaysPublicInitializers".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'alwaysPublicInitializers' attribute of @Style");
        }
        AlwaysPublicInitializersExtractor alwaysPublicInitializersExtractor$ = new AlwaysPublicInitializersExtractor();
        annotationValue$.accept(alwaysPublicInitializersExtractor$, null);

        alwaysPublicInitializers = alwaysPublicInitializersExtractor$.get();
        continue;
      }
      if ("builderVisibility".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'builderVisibility' attribute of @Style");
        }
        BuilderVisibilityExtractor builderVisibilityExtractor$ = new BuilderVisibilityExtractor();
        annotationValue$.accept(builderVisibilityExtractor$, null);

        builderVisibility = builderVisibilityExtractor$.get();
        continue;
      }
      if ("builderVisibilityString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'builderVisibilityString' attribute of @Style");
        }
        BuilderVisibilityStringExtractor builderVisibilityStringExtractor$ = new BuilderVisibilityStringExtractor();
        annotationValue$.accept(builderVisibilityStringExtractor$, null);

        builderVisibilityString = builderVisibilityStringExtractor$.get();
        continue;
      }
      if ("throwForInvalidImmutableState".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'throwForInvalidImmutableState' attribute of @Style");
        }
        ThrowForInvalidImmutableStateExtractor throwForInvalidImmutableStateExtractor$ = new ThrowForInvalidImmutableStateExtractor();
        annotationValue$.accept(throwForInvalidImmutableStateExtractor$, null);

        throwForInvalidImmutableState = throwForInvalidImmutableStateExtractor$.get();
        throwForInvalidImmutableStateName = throwForInvalidImmutableStateExtractor$.name();
        continue;
      }
      if ("throwForNullPointer".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'throwForNullPointer' attribute of @Style");
        }
        ThrowForNullPointerExtractor throwForNullPointerExtractor$ = new ThrowForNullPointerExtractor();
        annotationValue$.accept(throwForNullPointerExtractor$, null);

        throwForNullPointer = throwForNullPointerExtractor$.get();
        throwForNullPointerName = throwForNullPointerExtractor$.name();
        continue;
      }
      if ("depluralize".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'depluralize' attribute of @Style");
        }
        DepluralizeExtractor depluralizeExtractor$ = new DepluralizeExtractor();
        annotationValue$.accept(depluralizeExtractor$, null);

        depluralize = depluralizeExtractor$.get();
        continue;
      }
      if ("depluralizeDictionary".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'depluralizeDictionary' attribute of @Style");
        }
        DepluralizeDictionaryExtractor depluralizeDictionaryExtractor$ = new DepluralizeDictionaryExtractor();
        annotationValue$.accept(depluralizeDictionaryExtractor$, null);

        depluralizeDictionary = depluralizeDictionaryExtractor$.get();
        continue;
      }
      if ("immutableCopyOfRoutines".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'immutableCopyOfRoutines' attribute of @Style");
        }
        ImmutableCopyOfRoutinesExtractor immutableCopyOfRoutinesExtractor$ = new ImmutableCopyOfRoutinesExtractor();
        annotationValue$.accept(immutableCopyOfRoutinesExtractor$, null);

        immutableCopyOfRoutines = immutableCopyOfRoutinesExtractor$.get();
        immutableCopyOfRoutinesName = immutableCopyOfRoutinesExtractor$.name();
        continue;
      }
      if ("stagedBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'stagedBuilder' attribute of @Style");
        }
        StagedBuilderExtractor stagedBuilderExtractor$ = new StagedBuilderExtractor();
        annotationValue$.accept(stagedBuilderExtractor$, null);

        stagedBuilder = stagedBuilderExtractor$.get();
        continue;
      }
      if ("builtinContainerAttributes".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'builtinContainerAttributes' attribute of @Style");
        }
        BuiltinContainerAttributesExtractor builtinContainerAttributesExtractor$ = new BuiltinContainerAttributesExtractor();
        annotationValue$.accept(builtinContainerAttributesExtractor$, null);

        builtinContainerAttributes = builtinContainerAttributesExtractor$.get();
        continue;
      }
      if ("beanFriendlyModifiables".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'beanFriendlyModifiables' attribute of @Style");
        }
        BeanFriendlyModifiablesExtractor beanFriendlyModifiablesExtractor$ = new BeanFriendlyModifiablesExtractor();
        annotationValue$.accept(beanFriendlyModifiablesExtractor$, null);

        beanFriendlyModifiables = beanFriendlyModifiablesExtractor$.get();
        continue;
      }
      if ("allMandatoryParameters".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'allMandatoryParameters' attribute of @Style");
        }
        AllMandatoryParametersExtractor allMandatoryParametersExtractor$ = new AllMandatoryParametersExtractor();
        annotationValue$.accept(allMandatoryParametersExtractor$, null);

        allMandatoryParameters = allMandatoryParametersExtractor$.get();
        continue;
      }
      if ("redactedMask".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'redactedMask' attribute of @Style");
        }
        RedactedMaskExtractor redactedMaskExtractor$ = new RedactedMaskExtractor();
        annotationValue$.accept(redactedMaskExtractor$, null);

        redactedMask = redactedMaskExtractor$.get();
        continue;
      }
      if ("attributeBuilderDetection".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'attributeBuilderDetection' attribute of @Style");
        }
        AttributeBuilderDetectionExtractor attributeBuilderDetectionExtractor$ = new AttributeBuilderDetectionExtractor();
        annotationValue$.accept(attributeBuilderDetectionExtractor$, null);

        attributeBuilderDetection = attributeBuilderDetectionExtractor$.get();
        continue;
      }
      if ("transientDerivedFields".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'transientDerivedFields' attribute of @Style");
        }
        TransientDerivedFieldsExtractor transientDerivedFieldsExtractor$ = new TransientDerivedFieldsExtractor();
        annotationValue$.accept(transientDerivedFieldsExtractor$, null);

        transientDerivedFields = transientDerivedFieldsExtractor$.get();
        continue;
      }
      if ("finalInstanceFields".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'finalInstanceFields' attribute of @Style");
        }
        FinalInstanceFieldsExtractor finalInstanceFieldsExtractor$ = new FinalInstanceFieldsExtractor();
        annotationValue$.accept(finalInstanceFieldsExtractor$, null);

        finalInstanceFields = finalInstanceFieldsExtractor$.get();
        continue;
      }
      if ("attributeBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'attributeBuilder' attribute of @Style");
        }
        AttributeBuilderExtractor attributeBuilderExtractor$ = new AttributeBuilderExtractor();
        annotationValue$.accept(attributeBuilderExtractor$, null);

        attributeBuilder = attributeBuilderExtractor$.get();
        continue;
      }
      if ("getBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'getBuilder' attribute of @Style");
        }
        GetBuilderExtractor getBuilderExtractor$ = new GetBuilderExtractor();
        annotationValue$.accept(getBuilderExtractor$, null);

        getBuilder = getBuilderExtractor$.get();
        continue;
      }
      if ("setBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'setBuilder' attribute of @Style");
        }
        SetBuilderExtractor setBuilderExtractor$ = new SetBuilderExtractor();
        annotationValue$.accept(setBuilderExtractor$, null);

        setBuilder = setBuilderExtractor$.get();
        continue;
      }
      if ("addBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'addBuilder' attribute of @Style");
        }
        AddBuilderExtractor addBuilderExtractor$ = new AddBuilderExtractor();
        annotationValue$.accept(addBuilderExtractor$, null);

        addBuilder = addBuilderExtractor$.get();
        continue;
      }
      if ("addAllBuilder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'addAllBuilder' attribute of @Style");
        }
        AddAllBuilderExtractor addAllBuilderExtractor$ = new AddAllBuilderExtractor();
        annotationValue$.accept(addAllBuilderExtractor$, null);

        addAllBuilder = addAllBuilderExtractor$.get();
        continue;
      }
      if ("getBuilders".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'getBuilders' attribute of @Style");
        }
        GetBuildersExtractor getBuildersExtractor$ = new GetBuildersExtractor();
        annotationValue$.accept(getBuildersExtractor$, null);

        getBuilders = getBuildersExtractor$.get();
        continue;
      }
      if ("nullableAnnotation".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'nullableAnnotation' attribute of @Style");
        }
        NullableAnnotationExtractor nullableAnnotationExtractor$ = new NullableAnnotationExtractor();
        annotationValue$.accept(nullableAnnotationExtractor$, null);

        nullableAnnotation = nullableAnnotationExtractor$.get();
        continue;
      }
      if ("limitStringLengthInToString".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'limitStringLengthInToString' attribute of @Style");
        }
        LimitStringLengthInToStringExtractor limitStringLengthInToStringExtractor$ = new LimitStringLengthInToStringExtractor();
        annotationValue$.accept(limitStringLengthInToStringExtractor$, null);

        limitStringLengthInToString = limitStringLengthInToStringExtractor$.get();
        continue;
      }
      if ("allowedClasspathAnnotations".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'allowedClasspathAnnotations' attribute of @Style");
        }
        AllowedClasspathAnnotationsExtractor allowedClasspathAnnotationsExtractor$ = new AllowedClasspathAnnotationsExtractor();
        annotationValue$.accept(allowedClasspathAnnotationsExtractor$, null);

        allowedClasspathAnnotations = allowedClasspathAnnotationsExtractor$.get();
        allowedClasspathAnnotationsName = allowedClasspathAnnotationsExtractor$.name();
        continue;
      }
      if ("fallbackNullableAnnotation".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'fallbackNullableAnnotation' attribute of @Style");
        }
        FallbackNullableAnnotationExtractor fallbackNullableAnnotationExtractor$ = new FallbackNullableAnnotationExtractor();
        annotationValue$.accept(fallbackNullableAnnotationExtractor$, null);

        fallbackNullableAnnotation = fallbackNullableAnnotationExtractor$.get();
        fallbackNullableAnnotationName = fallbackNullableAnnotationExtractor$.name();
        continue;
      }
      if ("jakarta".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'jakarta' attribute of @Style");
        }
        JakartaExtractor jakartaExtractor$ = new JakartaExtractor();
        annotationValue$.accept(jakartaExtractor$, null);

        jakarta = jakartaExtractor$.get();
        continue;
      }
    }
    this.get = Preconditions.checkNotNull(get, "value for 'get'");
    this.init = Preconditions.checkNotNull(init, "value for 'init'");
    this.with = Preconditions.checkNotNull(with, "value for 'with'");
    this.withUnaryOperator = Preconditions.checkNotNull(withUnaryOperator, "value for 'withUnaryOperator'");
    this.add = Preconditions.checkNotNull(add, "value for 'add'");
    this.addAll = Preconditions.checkNotNull(addAll, "value for 'addAll'");
    this.put = Preconditions.checkNotNull(put, "value for 'put'");
    this.putAll = Preconditions.checkNotNull(putAll, "value for 'putAll'");
    this.copyOf = Preconditions.checkNotNull(copyOf, "value for 'copyOf'");
    this.of = Preconditions.checkNotNull(of, "value for 'of'");
    this.instance = Preconditions.checkNotNull(instance, "value for 'instance'");
    this.builder = Preconditions.checkNotNull(builder, "value for 'builder'");
    this.newBuilder = Preconditions.checkNotNull(newBuilder, "value for 'newBuilder'");
    this.from = Preconditions.checkNotNull(from, "value for 'from'");
    this.build = Preconditions.checkNotNull(build, "value for 'build'");
    this.buildOrThrow = Preconditions.checkNotNull(buildOrThrow, "value for 'buildOrThrow'");
    this.canBuild = Preconditions.checkNotNull(canBuild, "value for 'canBuild'");
    this.toBuilder = Preconditions.checkNotNull(toBuilder, "value for 'toBuilder'");
    this.underrideHashCode = Preconditions.checkNotNull(underrideHashCode, "value for 'underrideHashCode'");
    this.underrideEquals = Preconditions.checkNotNull(underrideEquals, "value for 'underrideEquals'");
    this.underrideToString = Preconditions.checkNotNull(underrideToString, "value for 'underrideToString'");
    this.delegateToString = Preconditions.checkNotNull(delegateToString, "value for 'delegateToString'");
    this.includeHashCode = Preconditions.checkNotNull(includeHashCode, "value for 'includeHashCode'");
    this.isInitialized = Preconditions.checkNotNull(isInitialized, "value for 'isInitialized'");
    this.isSet = Preconditions.checkNotNull(isSet, "value for 'isSet'");
    this.set = Preconditions.checkNotNull(set, "value for 'set'");
    this.unset = Preconditions.checkNotNull(unset, "value for 'unset'");
    this.clear = Preconditions.checkNotNull(clear, "value for 'clear'");
    this.create = Preconditions.checkNotNull(create, "value for 'create'");
    this.toImmutable = Preconditions.checkNotNull(toImmutable, "value for 'toImmutable'");
    this.typeBuilder = Preconditions.checkNotNull(typeBuilder, "value for 'typeBuilder'");
    this.typeInnerBuilder = Preconditions.checkNotNull(typeInnerBuilder, "value for 'typeInnerBuilder'");
    this.typeAbstract = Preconditions.checkNotNull(typeAbstract, "value for 'typeAbstract'");
    this.typeImmutable = Preconditions.checkNotNull(typeImmutable, "value for 'typeImmutable'");
    this.typeImmutableEnclosing = Preconditions.checkNotNull(typeImmutableEnclosing, "value for 'typeImmutableEnclosing'");
    this.typeImmutableNested = Preconditions.checkNotNull(typeImmutableNested, "value for 'typeImmutableNested'");
    this.typeModifiable = Preconditions.checkNotNull(typeModifiable, "value for 'typeModifiable'");
    this.typeInnerModifiable = Preconditions.checkNotNull(typeInnerModifiable, "value for 'typeInnerModifiable'");
    this.typeWith = Preconditions.checkNotNull(typeWith, "value for 'typeWith'");
    this.packageGenerated = Preconditions.checkNotNull(packageGenerated, "value for 'packageGenerated'");
    this.defaults = Preconditions.checkNotNull(defaults, "value for 'defaults'");
    this.defaultsMirror = Preconditions.checkNotNull(defaultsMirror, "value for 'defaults'");
    this.strictBuilder = strictBuilder;
    this.strictModifiable = strictModifiable;
    this.validationMethod = Preconditions.checkNotNull(validationMethod, "value for 'validationMethod'");
    this.allParameters = allParameters;
    this.defaultAsDefault = defaultAsDefault;
    this.headerComments = headerComments;
    this.jdkOnly = jdkOnly;
    this.jdk9Collections = jdk9Collections;
    this.passAnnotations = Preconditions.checkNotNull(passAnnotations, "value for 'passAnnotations'");
    this.passAnnotationsName = Preconditions.checkNotNull(passAnnotationsName, "Value for 'passAnnotations'");
    this.additionalJsonAnnotations = Preconditions.checkNotNull(additionalJsonAnnotations, "value for 'additionalJsonAnnotations'");
    this.additionalJsonAnnotationsName = Preconditions.checkNotNull(additionalJsonAnnotationsName, "Value for 'additionalJsonAnnotations'");
    this.visibility = Preconditions.checkNotNull(visibility, "value for 'visibility'");
    this.visibilityString = Preconditions.checkNotNull(visibilityString, "value for 'visibilityString'");
    this.optionalAcceptNullable = optionalAcceptNullable;
    this.generateSuppressAllWarnings = generateSuppressAllWarnings;
    this.privateNoargConstructor = privateNoargConstructor;
    this.protectedNoargConstructor = protectedNoargConstructor;
    this.attributelessSingleton = attributelessSingleton;
    this.unsafeDefaultAndDerived = unsafeDefaultAndDerived;
    this.clearBuilder = clearBuilder;
    this.deferCollectionAllocation = deferCollectionAllocation;
    this.deepImmutablesDetection = deepImmutablesDetection;
    this.overshadowImplementation = overshadowImplementation;
    this.implementationNestedInBuilder = implementationNestedInBuilder;
    this.forceJacksonPropertyNames = forceJacksonPropertyNames;
    this.forceJacksonIgnoreFields = forceJacksonIgnoreFields;
    this.forceEqualsInWithers = forceEqualsInWithers;
    this.jacksonIntegration = jacksonIntegration;
    this.weakInterning = weakInterning;
    this.alwaysPublicInitializers = alwaysPublicInitializers;
    this.builderVisibility = Preconditions.checkNotNull(builderVisibility, "value for 'builderVisibility'");
    this.builderVisibilityString = Preconditions.checkNotNull(builderVisibilityString, "value for 'builderVisibilityString'");
    this.throwForInvalidImmutableState = Preconditions.checkNotNull(throwForInvalidImmutableState, "value for 'throwForInvalidImmutableState'");
    this.throwForInvalidImmutableStateName = Preconditions.checkNotNull(throwForInvalidImmutableStateName, "Value for 'throwForInvalidImmutableState'");
    this.throwForNullPointer = Preconditions.checkNotNull(throwForNullPointer, "value for 'throwForNullPointer'");
    this.throwForNullPointerName = Preconditions.checkNotNull(throwForNullPointerName, "Value for 'throwForNullPointer'");
    this.depluralize = depluralize;
    this.depluralizeDictionary = Preconditions.checkNotNull(depluralizeDictionary, "value for 'depluralizeDictionary'");
    this.immutableCopyOfRoutines = Preconditions.checkNotNull(immutableCopyOfRoutines, "value for 'immutableCopyOfRoutines'");
    this.immutableCopyOfRoutinesName = Preconditions.checkNotNull(immutableCopyOfRoutinesName, "Value for 'immutableCopyOfRoutines'");
    this.stagedBuilder = stagedBuilder;
    this.builtinContainerAttributes = builtinContainerAttributes;
    this.beanFriendlyModifiables = beanFriendlyModifiables;
    this.allMandatoryParameters = allMandatoryParameters;
    this.redactedMask = Preconditions.checkNotNull(redactedMask, "value for 'redactedMask'");
    this.attributeBuilderDetection = attributeBuilderDetection;
    this.transientDerivedFields = transientDerivedFields;
    this.finalInstanceFields = finalInstanceFields;
    this.attributeBuilder = Preconditions.checkNotNull(attributeBuilder, "value for 'attributeBuilder'");
    this.getBuilder = Preconditions.checkNotNull(getBuilder, "value for 'getBuilder'");
    this.setBuilder = Preconditions.checkNotNull(setBuilder, "value for 'setBuilder'");
    this.addBuilder = Preconditions.checkNotNull(addBuilder, "value for 'addBuilder'");
    this.addAllBuilder = Preconditions.checkNotNull(addAllBuilder, "value for 'addAllBuilder'");
    this.getBuilders = Preconditions.checkNotNull(getBuilders, "value for 'getBuilders'");
    this.nullableAnnotation = Preconditions.checkNotNull(nullableAnnotation, "value for 'nullableAnnotation'");
    this.limitStringLengthInToString = limitStringLengthInToString;
    this.allowedClasspathAnnotations = Preconditions.checkNotNull(allowedClasspathAnnotations, "value for 'allowedClasspathAnnotations'");
    this.allowedClasspathAnnotationsName = Preconditions.checkNotNull(allowedClasspathAnnotationsName, "Value for 'allowedClasspathAnnotations'");
    this.fallbackNullableAnnotation = Preconditions.checkNotNull(fallbackNullableAnnotation, "value for 'fallbackNullableAnnotation'");
    this.fallbackNullableAnnotationName = Preconditions.checkNotNull(fallbackNullableAnnotationName, "Value for 'fallbackNullableAnnotation'");
    this.jakarta = jakarta;
  }

  /**
   * @return value of attribute {@code get}
   */
  @Override
  public String[] get() {
    return get.clone();
  }

  /**
   * @return value of attribute {@code init}
   */
  @Override
  public String init() {
    return init;
  }

  /**
   * @return value of attribute {@code with}
   */
  @Override
  public String with() {
    return with;
  }

  /**
   * @return value of attribute {@code withUnaryOperator}
   */
  @Override
  public String withUnaryOperator() {
    return withUnaryOperator;
  }

  /**
   * @return value of attribute {@code add}
   */
  @Override
  public String add() {
    return add;
  }

  /**
   * @return value of attribute {@code addAll}
   */
  @Override
  public String addAll() {
    return addAll;
  }

  /**
   * @return value of attribute {@code put}
   */
  @Override
  public String put() {
    return put;
  }

  /**
   * @return value of attribute {@code putAll}
   */
  @Override
  public String putAll() {
    return putAll;
  }

  /**
   * @return value of attribute {@code copyOf}
   */
  @Override
  public String copyOf() {
    return copyOf;
  }

  /**
   * @return value of attribute {@code of}
   */
  @Override
  public String of() {
    return of;
  }

  /**
   * @return value of attribute {@code instance}
   */
  @Override
  public String instance() {
    return instance;
  }

  /**
   * @return value of attribute {@code builder}
   */
  @Override
  public String builder() {
    return builder;
  }

  /**
   * @return value of attribute {@code newBuilder}
   */
  @Override
  public String newBuilder() {
    return newBuilder;
  }

  /**
   * @return value of attribute {@code from}
   */
  @Override
  public String from() {
    return from;
  }

  /**
   * @return value of attribute {@code build}
   */
  @Override
  public String build() {
    return build;
  }

  /**
   * @return value of attribute {@code buildOrThrow}
   */
  @Override
  public String buildOrThrow() {
    return buildOrThrow;
  }

  /**
   * @return value of attribute {@code canBuild}
   */
  @Override
  public String canBuild() {
    return canBuild;
  }

  /**
   * @return value of attribute {@code toBuilder}
   */
  @Override
  public String toBuilder() {
    return toBuilder;
  }

  /**
   * @return value of attribute {@code underrideHashCode}
   */
  @Override
  public String underrideHashCode() {
    return underrideHashCode;
  }

  /**
   * @return value of attribute {@code underrideEquals}
   */
  @Override
  public String underrideEquals() {
    return underrideEquals;
  }

  /**
   * @return value of attribute {@code underrideToString}
   */
  @Override
  public String underrideToString() {
    return underrideToString;
  }

  /**
   * @return value of attribute {@code delegateToString}
   */
  @Override
  public String delegateToString() {
    return delegateToString;
  }

  /**
   * @return value of attribute {@code includeHashCode}
   */
  @Override
  public String includeHashCode() {
    return includeHashCode;
  }

  /**
   * @return value of attribute {@code isInitialized}
   */
  @Override
  public String isInitialized() {
    return isInitialized;
  }

  /**
   * @return value of attribute {@code isSet}
   */
  @Override
  public String isSet() {
    return isSet;
  }

  /**
   * @return value of attribute {@code set}
   */
  @Override
  public String set() {
    return set;
  }

  /**
   * @return value of attribute {@code unset}
   */
  @Override
  public String unset() {
    return unset;
  }

  /**
   * @return value of attribute {@code clear}
   */
  @Override
  public String clear() {
    return clear;
  }

  /**
   * @return value of attribute {@code create}
   */
  @Override
  public String create() {
    return create;
  }

  /**
   * @return value of attribute {@code toImmutable}
   */
  @Override
  public String toImmutable() {
    return toImmutable;
  }

  /**
   * @return value of attribute {@code typeBuilder}
   */
  @Override
  public String typeBuilder() {
    return typeBuilder;
  }

  /**
   * @return value of attribute {@code typeInnerBuilder}
   */
  @Override
  public String typeInnerBuilder() {
    return typeInnerBuilder;
  }

  /**
   * @return value of attribute {@code typeAbstract}
   */
  @Override
  public String[] typeAbstract() {
    return typeAbstract.clone();
  }

  /**
   * @return value of attribute {@code typeImmutable}
   */
  @Override
  public String typeImmutable() {
    return typeImmutable;
  }

  /**
   * @return value of attribute {@code typeImmutableEnclosing}
   */
  @Override
  public String typeImmutableEnclosing() {
    return typeImmutableEnclosing;
  }

  /**
   * @return value of attribute {@code typeImmutableNested}
   */
  @Override
  public String typeImmutableNested() {
    return typeImmutableNested;
  }

  /**
   * @return value of attribute {@code typeModifiable}
   */
  @Override
  public String typeModifiable() {
    return typeModifiable;
  }

  /**
   * @return value of attribute {@code typeInnerModifiable}
   */
  @Override
  public String typeInnerModifiable() {
    return typeInnerModifiable;
  }

  /**
   * @return value of attribute {@code typeWith}
   */
  @Override
  public String typeWith() {
    return typeWith;
  }

  /**
   * @return value of attribute {@code packageGenerated}
   */
  @Override
  public String packageGenerated() {
    return packageGenerated;
  }

  /**
   * @return annotation mirror for value of attribute {@code defaults}
   */
  public AnnotationMirror defaultsMirror() {
    return defaults;
  }

  /**
   * @return reified annotation mirror for {@link org.immutables.value.processor.meta.ValueMirrors.Immutable}
   */
  @Override
  public ImmutableMirror defaults() {
    return defaultsMirror;
  }

  /**
   * @return value of attribute {@code strictBuilder}
   */
  @Override
  public boolean strictBuilder() {
    return strictBuilder;
  }

  /**
   * @return value of attribute {@code strictModifiable}
   */
  @Override
  public boolean strictModifiable() {
    return strictModifiable;
  }

  /**
   * @return value of attribute {@code validationMethod}
   */
  @Override
  public ValueMirrors.Style.ValidationMethod validationMethod() {
    return validationMethod;
  }

  /**
   * @return value of attribute {@code allParameters}
   */
  @Override
  public boolean allParameters() {
    return allParameters;
  }

  /**
   * @return value of attribute {@code defaultAsDefault}
   */
  @Override
  public boolean defaultAsDefault() {
    return defaultAsDefault;
  }

  /**
   * @return value of attribute {@code headerComments}
   */
  @Override
  public boolean headerComments() {
    return headerComments;
  }

  /**
   * @return value of attribute {@code jdkOnly}
   */
  @Override
  public boolean jdkOnly() {
    return jdkOnly;
  }

  /**
   * @return value of attribute {@code jdk9Collections}
   */
  @Override
  public boolean jdk9Collections() {
    return jdk9Collections;
  }

  /**
   * @return type name for value of attribute {@code passAnnotations}
   */
  public String[] passAnnotationsName() {
    return passAnnotationsName.clone();
  }

  /**
   * @return type mirror for value of attribute {@code passAnnotations}
   */
  public TypeMirror[] passAnnotationsMirror() {
    return passAnnotations.clone();
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #passAnnotationsMirror} or {@link passAnnotationsName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<? extends Annotation>[] passAnnotations() {
    throw new UnsupportedOperationException("Use 'passAnnotationsMirror()' or 'passAnnotationsName()'");
  }

  /**
   * @return type name for value of attribute {@code additionalJsonAnnotations}
   */
  public String[] additionalJsonAnnotationsName() {
    return additionalJsonAnnotationsName.clone();
  }

  /**
   * @return type mirror for value of attribute {@code additionalJsonAnnotations}
   */
  public TypeMirror[] additionalJsonAnnotationsMirror() {
    return additionalJsonAnnotations.clone();
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #additionalJsonAnnotationsMirror} or {@link additionalJsonAnnotationsName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<? extends Annotation>[] additionalJsonAnnotations() {
    throw new UnsupportedOperationException("Use 'additionalJsonAnnotationsMirror()' or 'additionalJsonAnnotationsName()'");
  }

  /**
   * @return value of attribute {@code visibility}
   */
  @Override
  public ValueMirrors.Style.ImplementationVisibility visibility() {
    return visibility;
  }

  /**
   * @return value of attribute {@code visibilityString}
   */
  @Override
  public String visibilityString() {
    return visibilityString;
  }

  /**
   * @return value of attribute {@code optionalAcceptNullable}
   */
  @Override
  public boolean optionalAcceptNullable() {
    return optionalAcceptNullable;
  }

  /**
   * @return value of attribute {@code generateSuppressAllWarnings}
   */
  @Override
  public boolean generateSuppressAllWarnings() {
    return generateSuppressAllWarnings;
  }

  /**
   * @return value of attribute {@code privateNoargConstructor}
   */
  @Override
  public boolean privateNoargConstructor() {
    return privateNoargConstructor;
  }

  /**
   * @return value of attribute {@code protectedNoargConstructor}
   */
  @Override
  public boolean protectedNoargConstructor() {
    return protectedNoargConstructor;
  }

  /**
   * @return value of attribute {@code attributelessSingleton}
   */
  @Override
  public boolean attributelessSingleton() {
    return attributelessSingleton;
  }

  /**
   * @return value of attribute {@code unsafeDefaultAndDerived}
   */
  @Override
  public boolean unsafeDefaultAndDerived() {
    return unsafeDefaultAndDerived;
  }

  /**
   * @return value of attribute {@code clearBuilder}
   */
  @Override
  public boolean clearBuilder() {
    return clearBuilder;
  }

  /**
   * @return value of attribute {@code deferCollectionAllocation}
   */
  @Override
  public boolean deferCollectionAllocation() {
    return deferCollectionAllocation;
  }

  /**
   * @return value of attribute {@code deepImmutablesDetection}
   */
  @Override
  public boolean deepImmutablesDetection() {
    return deepImmutablesDetection;
  }

  /**
   * @return value of attribute {@code overshadowImplementation}
   */
  @Override
  public boolean overshadowImplementation() {
    return overshadowImplementation;
  }

  /**
   * @return value of attribute {@code implementationNestedInBuilder}
   */
  @Override
  public boolean implementationNestedInBuilder() {
    return implementationNestedInBuilder;
  }

  /**
   * @return value of attribute {@code forceJacksonPropertyNames}
   */
  @Override
  public boolean forceJacksonPropertyNames() {
    return forceJacksonPropertyNames;
  }

  /**
   * @return value of attribute {@code forceJacksonIgnoreFields}
   */
  @Override
  public boolean forceJacksonIgnoreFields() {
    return forceJacksonIgnoreFields;
  }

  /**
   * @return value of attribute {@code forceEqualsInWithers}
   */
  @Override
  public boolean forceEqualsInWithers() {
    return forceEqualsInWithers;
  }

  /**
   * @return value of attribute {@code jacksonIntegration}
   */
  @Override
  public boolean jacksonIntegration() {
    return jacksonIntegration;
  }

  /**
   * @return value of attribute {@code weakInterning}
   */
  @Override
  public boolean weakInterning() {
    return weakInterning;
  }

  /**
   * @return value of attribute {@code alwaysPublicInitializers}
   */
  @Override
  public boolean alwaysPublicInitializers() {
    return alwaysPublicInitializers;
  }

  /**
   * @return value of attribute {@code builderVisibility}
   */
  @Override
  public ValueMirrors.Style.BuilderVisibility builderVisibility() {
    return builderVisibility;
  }

  /**
   * @return value of attribute {@code builderVisibilityString}
   */
  @Override
  public String builderVisibilityString() {
    return builderVisibilityString;
  }

  /**
   * @return type name for value of attribute {@code throwForInvalidImmutableState}
   */
  public String throwForInvalidImmutableStateName() {
    return throwForInvalidImmutableStateName;
  }

  /**
   * @return type mirror for value of attribute {@code throwForInvalidImmutableState}
   */
  public TypeMirror throwForInvalidImmutableStateMirror() {
    return throwForInvalidImmutableState;
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #throwForInvalidImmutableStateMirror} or {@link throwForInvalidImmutableStateName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<? extends Exception> throwForInvalidImmutableState() {
    throw new UnsupportedOperationException("Use 'throwForInvalidImmutableStateMirror()' or 'throwForInvalidImmutableStateName()'");
  }

  /**
   * @return type name for value of attribute {@code throwForNullPointer}
   */
  public String throwForNullPointerName() {
    return throwForNullPointerName;
  }

  /**
   * @return type mirror for value of attribute {@code throwForNullPointer}
   */
  public TypeMirror throwForNullPointerMirror() {
    return throwForNullPointer;
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #throwForNullPointerMirror} or {@link throwForNullPointerName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<? extends Exception> throwForNullPointer() {
    throw new UnsupportedOperationException("Use 'throwForNullPointerMirror()' or 'throwForNullPointerName()'");
  }

  /**
   * @return value of attribute {@code depluralize}
   */
  @Override
  public boolean depluralize() {
    return depluralize;
  }

  /**
   * @return value of attribute {@code depluralizeDictionary}
   */
  @Override
  public String[] depluralizeDictionary() {
    return depluralizeDictionary.clone();
  }

  /**
   * @return type name for value of attribute {@code immutableCopyOfRoutines}
   */
  public String[] immutableCopyOfRoutinesName() {
    return immutableCopyOfRoutinesName.clone();
  }

  /**
   * @return type mirror for value of attribute {@code immutableCopyOfRoutines}
   */
  public TypeMirror[] immutableCopyOfRoutinesMirror() {
    return immutableCopyOfRoutines.clone();
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #immutableCopyOfRoutinesMirror} or {@link immutableCopyOfRoutinesName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<?>[] immutableCopyOfRoutines() {
    throw new UnsupportedOperationException("Use 'immutableCopyOfRoutinesMirror()' or 'immutableCopyOfRoutinesName()'");
  }

  /**
   * @return value of attribute {@code stagedBuilder}
   */
  @Override
  public boolean stagedBuilder() {
    return stagedBuilder;
  }

  /**
   * @return value of attribute {@code builtinContainerAttributes}
   */
  @Override
  public boolean builtinContainerAttributes() {
    return builtinContainerAttributes;
  }

  /**
   * @return value of attribute {@code beanFriendlyModifiables}
   */
  @Override
  public boolean beanFriendlyModifiables() {
    return beanFriendlyModifiables;
  }

  /**
   * @return value of attribute {@code allMandatoryParameters}
   */
  @Override
  public boolean allMandatoryParameters() {
    return allMandatoryParameters;
  }

  /**
   * @return value of attribute {@code redactedMask}
   */
  @Override
  public String redactedMask() {
    return redactedMask;
  }

  /**
   * @return value of attribute {@code attributeBuilderDetection}
   */
  @Override
  public boolean attributeBuilderDetection() {
    return attributeBuilderDetection;
  }

  /**
   * @return value of attribute {@code transientDerivedFields}
   */
  @Override
  public boolean transientDerivedFields() {
    return transientDerivedFields;
  }

  /**
   * @return value of attribute {@code finalInstanceFields}
   */
  @Override
  public boolean finalInstanceFields() {
    return finalInstanceFields;
  }

  /**
   * @return value of attribute {@code attributeBuilder}
   */
  @Override
  public String[] attributeBuilder() {
    return attributeBuilder.clone();
  }

  /**
   * @return value of attribute {@code getBuilder}
   */
  @Override
  public String getBuilder() {
    return getBuilder;
  }

  /**
   * @return value of attribute {@code setBuilder}
   */
  @Override
  public String setBuilder() {
    return setBuilder;
  }

  /**
   * @return value of attribute {@code addBuilder}
   */
  @Override
  public String addBuilder() {
    return addBuilder;
  }

  /**
   * @return value of attribute {@code addAllBuilder}
   */
  @Override
  public String addAllBuilder() {
    return addAllBuilder;
  }

  /**
   * @return value of attribute {@code getBuilders}
   */
  @Override
  public String getBuilders() {
    return getBuilders;
  }

  /**
   * @return value of attribute {@code nullableAnnotation}
   */
  @Override
  public String nullableAnnotation() {
    return nullableAnnotation;
  }

  /**
   * @return value of attribute {@code limitStringLengthInToString}
   */
  @Override
  public int limitStringLengthInToString() {
    return limitStringLengthInToString;
  }

  /**
   * @return type name for value of attribute {@code allowedClasspathAnnotations}
   */
  public String[] allowedClasspathAnnotationsName() {
    return allowedClasspathAnnotationsName.clone();
  }

  /**
   * @return type mirror for value of attribute {@code allowedClasspathAnnotations}
   */
  public TypeMirror[] allowedClasspathAnnotationsMirror() {
    return allowedClasspathAnnotations.clone();
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #allowedClasspathAnnotationsMirror} or {@link allowedClasspathAnnotationsName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<? extends Annotation>[] allowedClasspathAnnotations() {
    throw new UnsupportedOperationException("Use 'allowedClasspathAnnotationsMirror()' or 'allowedClasspathAnnotationsName()'");
  }

  /**
   * @return type name for value of attribute {@code fallbackNullableAnnotation}
   */
  public String fallbackNullableAnnotationName() {
    return fallbackNullableAnnotationName;
  }

  /**
   * @return type mirror for value of attribute {@code fallbackNullableAnnotation}
   */
  public TypeMirror fallbackNullableAnnotationMirror() {
    return fallbackNullableAnnotation;
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #fallbackNullableAnnotationMirror} or {@link fallbackNullableAnnotationName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<? extends Annotation> fallbackNullableAnnotation() {
    throw new UnsupportedOperationException("Use 'fallbackNullableAnnotationMirror()' or 'fallbackNullableAnnotationName()'");
  }

  /**
   * @return value of attribute {@code jakarta}
   */
  @Override
  public boolean jakarta() {
    return jakarta;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Style.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return ValueMirrors.Style.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "get".hashCode() ^ Arrays.hashCode(get);
    h += 127 * "init".hashCode() ^ init.hashCode();
    h += 127 * "with".hashCode() ^ with.hashCode();
    h += 127 * "withUnaryOperator".hashCode() ^ withUnaryOperator.hashCode();
    h += 127 * "add".hashCode() ^ add.hashCode();
    h += 127 * "addAll".hashCode() ^ addAll.hashCode();
    h += 127 * "put".hashCode() ^ put.hashCode();
    h += 127 * "putAll".hashCode() ^ putAll.hashCode();
    h += 127 * "copyOf".hashCode() ^ copyOf.hashCode();
    h += 127 * "of".hashCode() ^ of.hashCode();
    h += 127 * "instance".hashCode() ^ instance.hashCode();
    h += 127 * "builder".hashCode() ^ builder.hashCode();
    h += 127 * "newBuilder".hashCode() ^ newBuilder.hashCode();
    h += 127 * "from".hashCode() ^ from.hashCode();
    h += 127 * "build".hashCode() ^ build.hashCode();
    h += 127 * "buildOrThrow".hashCode() ^ buildOrThrow.hashCode();
    h += 127 * "canBuild".hashCode() ^ canBuild.hashCode();
    h += 127 * "toBuilder".hashCode() ^ toBuilder.hashCode();
    h += 127 * "underrideHashCode".hashCode() ^ underrideHashCode.hashCode();
    h += 127 * "underrideEquals".hashCode() ^ underrideEquals.hashCode();
    h += 127 * "underrideToString".hashCode() ^ underrideToString.hashCode();
    h += 127 * "delegateToString".hashCode() ^ delegateToString.hashCode();
    h += 127 * "includeHashCode".hashCode() ^ includeHashCode.hashCode();
    h += 127 * "isInitialized".hashCode() ^ isInitialized.hashCode();
    h += 127 * "isSet".hashCode() ^ isSet.hashCode();
    h += 127 * "set".hashCode() ^ set.hashCode();
    h += 127 * "unset".hashCode() ^ unset.hashCode();
    h += 127 * "clear".hashCode() ^ clear.hashCode();
    h += 127 * "create".hashCode() ^ create.hashCode();
    h += 127 * "toImmutable".hashCode() ^ toImmutable.hashCode();
    h += 127 * "typeBuilder".hashCode() ^ typeBuilder.hashCode();
    h += 127 * "typeInnerBuilder".hashCode() ^ typeInnerBuilder.hashCode();
    h += 127 * "typeAbstract".hashCode() ^ Arrays.hashCode(typeAbstract);
    h += 127 * "typeImmutable".hashCode() ^ typeImmutable.hashCode();
    h += 127 * "typeImmutableEnclosing".hashCode() ^ typeImmutableEnclosing.hashCode();
    h += 127 * "typeImmutableNested".hashCode() ^ typeImmutableNested.hashCode();
    h += 127 * "typeModifiable".hashCode() ^ typeModifiable.hashCode();
    h += 127 * "typeInnerModifiable".hashCode() ^ typeInnerModifiable.hashCode();
    h += 127 * "typeWith".hashCode() ^ typeWith.hashCode();
    h += 127 * "packageGenerated".hashCode() ^ packageGenerated.hashCode();
    h += 127 * "defaults".hashCode() ^ defaults.hashCode();
    h += 127 * "strictBuilder".hashCode() ^ Booleans.hashCode(strictBuilder);
    h += 127 * "strictModifiable".hashCode() ^ Booleans.hashCode(strictModifiable);
    h += 127 * "validationMethod".hashCode() ^ validationMethod.hashCode();
    h += 127 * "allParameters".hashCode() ^ Booleans.hashCode(allParameters);
    h += 127 * "defaultAsDefault".hashCode() ^ Booleans.hashCode(defaultAsDefault);
    h += 127 * "headerComments".hashCode() ^ Booleans.hashCode(headerComments);
    h += 127 * "jdkOnly".hashCode() ^ Booleans.hashCode(jdkOnly);
    h += 127 * "jdk9Collections".hashCode() ^ Booleans.hashCode(jdk9Collections);
    h += 127 * "passAnnotations".hashCode() ^ Arrays.hashCode(passAnnotationsName);
    h += 127 * "additionalJsonAnnotations".hashCode() ^ Arrays.hashCode(additionalJsonAnnotationsName);
    h += 127 * "visibility".hashCode() ^ visibility.hashCode();
    h += 127 * "visibilityString".hashCode() ^ visibilityString.hashCode();
    h += 127 * "optionalAcceptNullable".hashCode() ^ Booleans.hashCode(optionalAcceptNullable);
    h += 127 * "generateSuppressAllWarnings".hashCode() ^ Booleans.hashCode(generateSuppressAllWarnings);
    h += 127 * "privateNoargConstructor".hashCode() ^ Booleans.hashCode(privateNoargConstructor);
    h += 127 * "protectedNoargConstructor".hashCode() ^ Booleans.hashCode(protectedNoargConstructor);
    h += 127 * "attributelessSingleton".hashCode() ^ Booleans.hashCode(attributelessSingleton);
    h += 127 * "unsafeDefaultAndDerived".hashCode() ^ Booleans.hashCode(unsafeDefaultAndDerived);
    h += 127 * "clearBuilder".hashCode() ^ Booleans.hashCode(clearBuilder);
    h += 127 * "deferCollectionAllocation".hashCode() ^ Booleans.hashCode(deferCollectionAllocation);
    h += 127 * "deepImmutablesDetection".hashCode() ^ Booleans.hashCode(deepImmutablesDetection);
    h += 127 * "overshadowImplementation".hashCode() ^ Booleans.hashCode(overshadowImplementation);
    h += 127 * "implementationNestedInBuilder".hashCode() ^ Booleans.hashCode(implementationNestedInBuilder);
    h += 127 * "forceJacksonPropertyNames".hashCode() ^ Booleans.hashCode(forceJacksonPropertyNames);
    h += 127 * "forceJacksonIgnoreFields".hashCode() ^ Booleans.hashCode(forceJacksonIgnoreFields);
    h += 127 * "forceEqualsInWithers".hashCode() ^ Booleans.hashCode(forceEqualsInWithers);
    h += 127 * "jacksonIntegration".hashCode() ^ Booleans.hashCode(jacksonIntegration);
    h += 127 * "weakInterning".hashCode() ^ Booleans.hashCode(weakInterning);
    h += 127 * "alwaysPublicInitializers".hashCode() ^ Booleans.hashCode(alwaysPublicInitializers);
    h += 127 * "builderVisibility".hashCode() ^ builderVisibility.hashCode();
    h += 127 * "builderVisibilityString".hashCode() ^ builderVisibilityString.hashCode();
    h += 127 * "throwForInvalidImmutableState".hashCode() ^ throwForInvalidImmutableStateName.hashCode();
    h += 127 * "throwForNullPointer".hashCode() ^ throwForNullPointerName.hashCode();
    h += 127 * "depluralize".hashCode() ^ Booleans.hashCode(depluralize);
    h += 127 * "depluralizeDictionary".hashCode() ^ Arrays.hashCode(depluralizeDictionary);
    h += 127 * "immutableCopyOfRoutines".hashCode() ^ Arrays.hashCode(immutableCopyOfRoutinesName);
    h += 127 * "stagedBuilder".hashCode() ^ Booleans.hashCode(stagedBuilder);
    h += 127 * "builtinContainerAttributes".hashCode() ^ Booleans.hashCode(builtinContainerAttributes);
    h += 127 * "beanFriendlyModifiables".hashCode() ^ Booleans.hashCode(beanFriendlyModifiables);
    h += 127 * "allMandatoryParameters".hashCode() ^ Booleans.hashCode(allMandatoryParameters);
    h += 127 * "redactedMask".hashCode() ^ redactedMask.hashCode();
    h += 127 * "attributeBuilderDetection".hashCode() ^ Booleans.hashCode(attributeBuilderDetection);
    h += 127 * "transientDerivedFields".hashCode() ^ Booleans.hashCode(transientDerivedFields);
    h += 127 * "finalInstanceFields".hashCode() ^ Booleans.hashCode(finalInstanceFields);
    h += 127 * "attributeBuilder".hashCode() ^ Arrays.hashCode(attributeBuilder);
    h += 127 * "getBuilder".hashCode() ^ getBuilder.hashCode();
    h += 127 * "setBuilder".hashCode() ^ setBuilder.hashCode();
    h += 127 * "addBuilder".hashCode() ^ addBuilder.hashCode();
    h += 127 * "addAllBuilder".hashCode() ^ addAllBuilder.hashCode();
    h += 127 * "getBuilders".hashCode() ^ getBuilders.hashCode();
    h += 127 * "nullableAnnotation".hashCode() ^ nullableAnnotation.hashCode();
    h += 127 * "limitStringLengthInToString".hashCode() ^ Ints.hashCode(limitStringLengthInToString);
    h += 127 * "allowedClasspathAnnotations".hashCode() ^ Arrays.hashCode(allowedClasspathAnnotationsName);
    h += 127 * "fallbackNullableAnnotation".hashCode() ^ fallbackNullableAnnotationName.hashCode();
    h += 127 * "jakarta".hashCode() ^ Booleans.hashCode(jakarta);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof StyleMirror) {
      StyleMirror otherMirror = (StyleMirror) other;
      return Arrays.equals(get, otherMirror.get)
          && init.equals(otherMirror.init)
          && with.equals(otherMirror.with)
          && withUnaryOperator.equals(otherMirror.withUnaryOperator)
          && add.equals(otherMirror.add)
          && addAll.equals(otherMirror.addAll)
          && put.equals(otherMirror.put)
          && putAll.equals(otherMirror.putAll)
          && copyOf.equals(otherMirror.copyOf)
          && of.equals(otherMirror.of)
          && instance.equals(otherMirror.instance)
          && builder.equals(otherMirror.builder)
          && newBuilder.equals(otherMirror.newBuilder)
          && from.equals(otherMirror.from)
          && build.equals(otherMirror.build)
          && buildOrThrow.equals(otherMirror.buildOrThrow)
          && canBuild.equals(otherMirror.canBuild)
          && toBuilder.equals(otherMirror.toBuilder)
          && underrideHashCode.equals(otherMirror.underrideHashCode)
          && underrideEquals.equals(otherMirror.underrideEquals)
          && underrideToString.equals(otherMirror.underrideToString)
          && delegateToString.equals(otherMirror.delegateToString)
          && includeHashCode.equals(otherMirror.includeHashCode)
          && isInitialized.equals(otherMirror.isInitialized)
          && isSet.equals(otherMirror.isSet)
          && set.equals(otherMirror.set)
          && unset.equals(otherMirror.unset)
          && clear.equals(otherMirror.clear)
          && create.equals(otherMirror.create)
          && toImmutable.equals(otherMirror.toImmutable)
          && typeBuilder.equals(otherMirror.typeBuilder)
          && typeInnerBuilder.equals(otherMirror.typeInnerBuilder)
          && Arrays.equals(typeAbstract, otherMirror.typeAbstract)
          && typeImmutable.equals(otherMirror.typeImmutable)
          && typeImmutableEnclosing.equals(otherMirror.typeImmutableEnclosing)
          && typeImmutableNested.equals(otherMirror.typeImmutableNested)
          && typeModifiable.equals(otherMirror.typeModifiable)
          && typeInnerModifiable.equals(otherMirror.typeInnerModifiable)
          && typeWith.equals(otherMirror.typeWith)
          && packageGenerated.equals(otherMirror.packageGenerated)
          && defaults.equals(otherMirror.defaults)
          && strictBuilder == otherMirror.strictBuilder
          && strictModifiable == otherMirror.strictModifiable
          && validationMethod.equals(otherMirror.validationMethod)
          && allParameters == otherMirror.allParameters
          && defaultAsDefault == otherMirror.defaultAsDefault
          && headerComments == otherMirror.headerComments
          && jdkOnly == otherMirror.jdkOnly
          && jdk9Collections == otherMirror.jdk9Collections
          && Arrays.equals(passAnnotationsName, otherMirror.passAnnotationsName)
          && Arrays.equals(additionalJsonAnnotationsName, otherMirror.additionalJsonAnnotationsName)
          && visibility.equals(otherMirror.visibility)
          && visibilityString.equals(otherMirror.visibilityString)
          && optionalAcceptNullable == otherMirror.optionalAcceptNullable
          && generateSuppressAllWarnings == otherMirror.generateSuppressAllWarnings
          && privateNoargConstructor == otherMirror.privateNoargConstructor
          && protectedNoargConstructor == otherMirror.protectedNoargConstructor
          && attributelessSingleton == otherMirror.attributelessSingleton
          && unsafeDefaultAndDerived == otherMirror.unsafeDefaultAndDerived
          && clearBuilder == otherMirror.clearBuilder
          && deferCollectionAllocation == otherMirror.deferCollectionAllocation
          && deepImmutablesDetection == otherMirror.deepImmutablesDetection
          && overshadowImplementation == otherMirror.overshadowImplementation
          && implementationNestedInBuilder == otherMirror.implementationNestedInBuilder
          && forceJacksonPropertyNames == otherMirror.forceJacksonPropertyNames
          && forceJacksonIgnoreFields == otherMirror.forceJacksonIgnoreFields
          && forceEqualsInWithers == otherMirror.forceEqualsInWithers
          && jacksonIntegration == otherMirror.jacksonIntegration
          && weakInterning == otherMirror.weakInterning
          && alwaysPublicInitializers == otherMirror.alwaysPublicInitializers
          && builderVisibility.equals(otherMirror.builderVisibility)
          && builderVisibilityString.equals(otherMirror.builderVisibilityString)
          && throwForInvalidImmutableStateName.equals(otherMirror.throwForInvalidImmutableStateName)
          && throwForNullPointerName.equals(otherMirror.throwForNullPointerName)
          && depluralize == otherMirror.depluralize
          && Arrays.equals(depluralizeDictionary, otherMirror.depluralizeDictionary)
          && Arrays.equals(immutableCopyOfRoutinesName, otherMirror.immutableCopyOfRoutinesName)
          && stagedBuilder == otherMirror.stagedBuilder
          && builtinContainerAttributes == otherMirror.builtinContainerAttributes
          && beanFriendlyModifiables == otherMirror.beanFriendlyModifiables
          && allMandatoryParameters == otherMirror.allMandatoryParameters
          && redactedMask.equals(otherMirror.redactedMask)
          && attributeBuilderDetection == otherMirror.attributeBuilderDetection
          && transientDerivedFields == otherMirror.transientDerivedFields
          && finalInstanceFields == otherMirror.finalInstanceFields
          && Arrays.equals(attributeBuilder, otherMirror.attributeBuilder)
          && getBuilder.equals(otherMirror.getBuilder)
          && setBuilder.equals(otherMirror.setBuilder)
          && addBuilder.equals(otherMirror.addBuilder)
          && addAllBuilder.equals(otherMirror.addAllBuilder)
          && getBuilders.equals(otherMirror.getBuilders)
          && nullableAnnotation.equals(otherMirror.nullableAnnotation)
          && limitStringLengthInToString == otherMirror.limitStringLengthInToString
          && Arrays.equals(allowedClasspathAnnotationsName, otherMirror.allowedClasspathAnnotationsName)
          && fallbackNullableAnnotationName.equals(otherMirror.fallbackNullableAnnotationName)
          && jakarta == otherMirror.jakarta;
    }
    return false;
  }

  @Override
  public String toString() {
    return "StyleMirror:" + annotationMirror;
  }

  private static class GetExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'get' in @" + QUALIFIED_NAME);
    }
  }

  private static class InitExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'init' in @" + QUALIFIED_NAME);
    }
  }

  private static class WithExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'with' in @" + QUALIFIED_NAME);
    }
  }

  private static class WithUnaryOperatorExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'withUnaryOperator' in @" + QUALIFIED_NAME);
    }
  }

  private static class AddExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'add' in @" + QUALIFIED_NAME);
    }
  }

  private static class AddAllExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'addAll' in @" + QUALIFIED_NAME);
    }
  }

  private static class PutExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'put' in @" + QUALIFIED_NAME);
    }
  }

  private static class PutAllExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'putAll' in @" + QUALIFIED_NAME);
    }
  }

  private static class CopyOfExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'copyOf' in @" + QUALIFIED_NAME);
    }
  }

  private static class OfExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'of' in @" + QUALIFIED_NAME);
    }
  }

  private static class InstanceExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'instance' in @" + QUALIFIED_NAME);
    }
  }

  private static class BuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'builder' in @" + QUALIFIED_NAME);
    }
  }

  private static class NewBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'newBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class FromExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'from' in @" + QUALIFIED_NAME);
    }
  }

  private static class BuildExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'build' in @" + QUALIFIED_NAME);
    }
  }

  private static class BuildOrThrowExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'buildOrThrow' in @" + QUALIFIED_NAME);
    }
  }

  private static class CanBuildExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'canBuild' in @" + QUALIFIED_NAME);
    }
  }

  private static class ToBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'toBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class UnderrideHashCodeExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'underrideHashCode' in @" + QUALIFIED_NAME);
    }
  }

  private static class UnderrideEqualsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'underrideEquals' in @" + QUALIFIED_NAME);
    }
  }

  private static class UnderrideToStringExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'underrideToString' in @" + QUALIFIED_NAME);
    }
  }

  private static class DelegateToStringExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'delegateToString' in @" + QUALIFIED_NAME);
    }
  }

  private static class IncludeHashCodeExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'includeHashCode' in @" + QUALIFIED_NAME);
    }
  }

  private static class IsInitializedExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'isInitialized' in @" + QUALIFIED_NAME);
    }
  }

  private static class IsSetExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'isSet' in @" + QUALIFIED_NAME);
    }
  }

  private static class SetExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'set' in @" + QUALIFIED_NAME);
    }
  }

  private static class UnsetExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'unset' in @" + QUALIFIED_NAME);
    }
  }

  private static class ClearExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'clear' in @" + QUALIFIED_NAME);
    }
  }

  private static class CreateExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'create' in @" + QUALIFIED_NAME);
    }
  }

  private static class ToImmutableExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'toImmutable' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeInnerBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeInnerBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeAbstractExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeAbstract' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeImmutableExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeImmutable' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeImmutableEnclosingExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeImmutableEnclosing' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeImmutableNestedExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeImmutableNested' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeModifiableExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeModifiable' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeInnerModifiableExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeInnerModifiable' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeWithExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeWith' in @" + QUALIFIED_NAME);
    }
  }

  private static class PackageGeneratedExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'packageGenerated' in @" + QUALIFIED_NAME);
    }
  }

  private static class DefaultsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    AnnotationMirror value;

    @Override
    public Void visitAnnotation(AnnotationMirror value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    AnnotationMirror get() {
      return value;
    }

    public ImmutableMirror mirror() {
      return ImmutableMirror.from(value).get();
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'defaults' in @" + QUALIFIED_NAME);
    }
  }

  private static class StrictBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'strictBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class StrictModifiableExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'strictModifiable' in @" + QUALIFIED_NAME);
    }
  }

  private static class ValidationMethodExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    ValueMirrors.Style.ValidationMethod value;

    @Override
    public Void visitEnumConstant(VariableElement value, Void p) {
      this.value = ValueMirrors.Style.ValidationMethod.valueOf(value.getSimpleName().toString());
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    ValueMirrors.Style.ValidationMethod get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'validationMethod' in @" + QUALIFIED_NAME);
    }
  }

  private static class AllParametersExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'allParameters' in @" + QUALIFIED_NAME);
    }
  }

  private static class DefaultAsDefaultExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'defaultAsDefault' in @" + QUALIFIED_NAME);
    }
  }

  private static class HeaderCommentsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'headerComments' in @" + QUALIFIED_NAME);
    }
  }

  private static class JdkOnlyExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'jdkOnly' in @" + QUALIFIED_NAME);
    }
  }

  private static class Jdk9CollectionsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'jdk9Collections' in @" + QUALIFIED_NAME);
    }
  }

  private static class PassAnnotationsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror[] values;
    int position;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new TypeMirror[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    TypeMirror[] get() {
      return values;
    }

    public String[] name() {
      String[] names = new String[values.length];
      for (int i = 0; i < values.length; i++) {
        names[i] = values[i].toString();
      }
      return names;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'passAnnotations' in @" + QUALIFIED_NAME);
    }
  }

  private static class AdditionalJsonAnnotationsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror[] values;
    int position;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new TypeMirror[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    TypeMirror[] get() {
      return values;
    }

    public String[] name() {
      String[] names = new String[values.length];
      for (int i = 0; i < values.length; i++) {
        names[i] = values[i].toString();
      }
      return names;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'additionalJsonAnnotations' in @" + QUALIFIED_NAME);
    }
  }

  private static class VisibilityExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    ValueMirrors.Style.ImplementationVisibility value;

    @Override
    public Void visitEnumConstant(VariableElement value, Void p) {
      this.value = ValueMirrors.Style.ImplementationVisibility.valueOf(value.getSimpleName().toString());
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    ValueMirrors.Style.ImplementationVisibility get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'visibility' in @" + QUALIFIED_NAME);
    }
  }

  private static class VisibilityStringExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'visibilityString' in @" + QUALIFIED_NAME);
    }
  }

  private static class OptionalAcceptNullableExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'optionalAcceptNullable' in @" + QUALIFIED_NAME);
    }
  }

  private static class GenerateSuppressAllWarningsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'generateSuppressAllWarnings' in @" + QUALIFIED_NAME);
    }
  }

  private static class PrivateNoargConstructorExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'privateNoargConstructor' in @" + QUALIFIED_NAME);
    }
  }

  private static class ProtectedNoargConstructorExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'protectedNoargConstructor' in @" + QUALIFIED_NAME);
    }
  }

  private static class AttributelessSingletonExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'attributelessSingleton' in @" + QUALIFIED_NAME);
    }
  }

  private static class UnsafeDefaultAndDerivedExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'unsafeDefaultAndDerived' in @" + QUALIFIED_NAME);
    }
  }

  private static class ClearBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'clearBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class DeferCollectionAllocationExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'deferCollectionAllocation' in @" + QUALIFIED_NAME);
    }
  }

  private static class DeepImmutablesDetectionExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'deepImmutablesDetection' in @" + QUALIFIED_NAME);
    }
  }

  private static class OvershadowImplementationExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'overshadowImplementation' in @" + QUALIFIED_NAME);
    }
  }

  private static class ImplementationNestedInBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'implementationNestedInBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class ForceJacksonPropertyNamesExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'forceJacksonPropertyNames' in @" + QUALIFIED_NAME);
    }
  }

  private static class ForceJacksonIgnoreFieldsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'forceJacksonIgnoreFields' in @" + QUALIFIED_NAME);
    }
  }

  private static class ForceEqualsInWithersExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'forceEqualsInWithers' in @" + QUALIFIED_NAME);
    }
  }

  private static class JacksonIntegrationExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'jacksonIntegration' in @" + QUALIFIED_NAME);
    }
  }

  private static class WeakInterningExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'weakInterning' in @" + QUALIFIED_NAME);
    }
  }

  private static class AlwaysPublicInitializersExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'alwaysPublicInitializers' in @" + QUALIFIED_NAME);
    }
  }

  private static class BuilderVisibilityExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    ValueMirrors.Style.BuilderVisibility value;

    @Override
    public Void visitEnumConstant(VariableElement value, Void p) {
      this.value = ValueMirrors.Style.BuilderVisibility.valueOf(value.getSimpleName().toString());
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    ValueMirrors.Style.BuilderVisibility get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'builderVisibility' in @" + QUALIFIED_NAME);
    }
  }

  private static class BuilderVisibilityStringExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'builderVisibilityString' in @" + QUALIFIED_NAME);
    }
  }

  private static class ThrowForInvalidImmutableStateExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror value;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    TypeMirror get() {
      return value;
    }

    public String name() {
      return value.toString();
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'throwForInvalidImmutableState' in @" + QUALIFIED_NAME);
    }
  }

  private static class ThrowForNullPointerExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror value;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    TypeMirror get() {
      return value;
    }

    public String name() {
      return value.toString();
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'throwForNullPointer' in @" + QUALIFIED_NAME);
    }
  }

  private static class DepluralizeExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'depluralize' in @" + QUALIFIED_NAME);
    }
  }

  private static class DepluralizeDictionaryExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'depluralizeDictionary' in @" + QUALIFIED_NAME);
    }
  }

  private static class ImmutableCopyOfRoutinesExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror[] values;
    int position;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new TypeMirror[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    TypeMirror[] get() {
      return values;
    }

    public String[] name() {
      String[] names = new String[values.length];
      for (int i = 0; i < values.length; i++) {
        names[i] = values[i].toString();
      }
      return names;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'immutableCopyOfRoutines' in @" + QUALIFIED_NAME);
    }
  }

  private static class StagedBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'stagedBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class BuiltinContainerAttributesExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'builtinContainerAttributes' in @" + QUALIFIED_NAME);
    }
  }

  private static class BeanFriendlyModifiablesExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'beanFriendlyModifiables' in @" + QUALIFIED_NAME);
    }
  }

  private static class AllMandatoryParametersExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'allMandatoryParameters' in @" + QUALIFIED_NAME);
    }
  }

  private static class RedactedMaskExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'redactedMask' in @" + QUALIFIED_NAME);
    }
  }

  private static class AttributeBuilderDetectionExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'attributeBuilderDetection' in @" + QUALIFIED_NAME);
    }
  }

  private static class TransientDerivedFieldsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'transientDerivedFields' in @" + QUALIFIED_NAME);
    }
  }

  private static class FinalInstanceFieldsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'finalInstanceFields' in @" + QUALIFIED_NAME);
    }
  }

  private static class AttributeBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'attributeBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class GetBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'getBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class SetBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'setBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class AddBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'addBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class AddAllBuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'addAllBuilder' in @" + QUALIFIED_NAME);
    }
  }

  private static class GetBuildersExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'getBuilders' in @" + QUALIFIED_NAME);
    }
  }

  private static class NullableAnnotationExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'nullableAnnotation' in @" + QUALIFIED_NAME);
    }
  }

  private static class LimitStringLengthInToStringExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    int value;

    @Override
    public Void visitInt(int value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    int get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'limitStringLengthInToString' in @" + QUALIFIED_NAME);
    }
  }

  private static class AllowedClasspathAnnotationsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror[] values;
    int position;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new TypeMirror[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    TypeMirror[] get() {
      return values;
    }

    public String[] name() {
      String[] names = new String[values.length];
      for (int i = 0; i < values.length; i++) {
        names[i] = values[i].toString();
      }
      return names;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'allowedClasspathAnnotations' in @" + QUALIFIED_NAME);
    }
  }

  private static class FallbackNullableAnnotationExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror value;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    TypeMirror get() {
      return value;
    }

    public String name() {
      return value.toString();
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'fallbackNullableAnnotation' in @" + QUALIFIED_NAME);
    }
  }

  private static class JakartaExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'jakarta' in @" + QUALIFIED_NAME);
    }
  }
}
