package org.immutables.value.processor.meta;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
/**
 * CriteriaRepositoryMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.criteria.Criteria.Repository}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.CriteriaMirrors.CriteriaRepository} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class CriteriaRepositoryMirror implements CriteriaMirrors.CriteriaRepository {
  public static final String QUALIFIED_NAME = "org.immutables.criteria.Criteria.Repository";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.CriteriaMirrors.CriteriaRepository";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Repository";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code CriteriaRepositoryMirror}, present if this annotation found
   */
  public static Optional<CriteriaRepositoryMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code CriteriaRepositoryMirror}, present if this annotation found
   */
  public static Optional<CriteriaRepositoryMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new CriteriaRepositoryMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code CriteriaRepositoryMirror}s
   */
  public static ImmutableList<CriteriaRepositoryMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<CriteriaRepositoryMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new CriteriaRepositoryMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code CriteriaRepositoryMirror}
   */
  public static CriteriaRepositoryMirror from(TypeElement element) {
    return new CriteriaRepositoryMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code CriteriaRepositoryMirror}, present if mirror matched this annotation type
   */
  public static Optional<CriteriaRepositoryMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final TypeMirror[] facets;
  private final String[] facetsName;

  private CriteriaRepositoryMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    TypeMirror[] facets = null;
    String[] facetsName = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("facets".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Repository");
        }
        FacetsExtractor facetsExtractor$ = new FacetsExtractor();
        annotationValue$.accept(facetsExtractor$, null);

        facets = facetsExtractor$.get();
        facetsName = facetsExtractor$.name();
        continue;
      }
    }
    this.facets = Preconditions.checkNotNull(facets, "default attribute 'facets'");
    this.facetsName = Preconditions.checkNotNull(facetsName, "default attribute 'facets'");
  }

  private CriteriaRepositoryMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    TypeMirror[] facets = null;
    String[] facetsName = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("facets".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'facets' attribute of @Repository");
        }
        FacetsExtractor facetsExtractor$ = new FacetsExtractor();
        annotationValue$.accept(facetsExtractor$, null);

        facets = facetsExtractor$.get();
        facetsName = facetsExtractor$.name();
        continue;
      }
    }
    this.facets = Preconditions.checkNotNull(facets, "value for 'facets'");
    this.facetsName = Preconditions.checkNotNull(facetsName, "Value for 'facets'");
  }

  /**
   * @return type name for value of attribute {@code facets}
   */
  public String[] facetsName() {
    return facetsName.clone();
  }

  /**
   * @return type mirror for value of attribute {@code facets}
   */
  public TypeMirror[] facetsMirror() {
    return facets.clone();
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #facetsMirror} or {@link facetsName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<?>[] facets() {
    throw new UnsupportedOperationException("Use 'facetsMirror()' or 'facetsName()'");
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code CriteriaRepository.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return CriteriaMirrors.CriteriaRepository.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "facets".hashCode() ^ Arrays.hashCode(facetsName);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof CriteriaRepositoryMirror) {
      CriteriaRepositoryMirror otherMirror = (CriteriaRepositoryMirror) other;
      return Arrays.equals(facetsName, otherMirror.facetsName);
    }
    return false;
  }

  @Override
  public String toString() {
    return "CriteriaRepositoryMirror:" + annotationMirror;
  }

  private static class FacetsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror[] values;
    int position;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new TypeMirror[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    TypeMirror[] get() {
      return values;
    }

    public String[] name() {
      String[] names = new String[values.length];
      for (int i = 0; i < values.length; i++) {
        names[i] = values[i].toString();
      }
      return names;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'facets' in @" + QUALIFIED_NAME);
    }
  }
}
