package org.immutables.value.processor.meta;

import java.lang.annotation.Annotation;
import java.util.Collections;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;

/**
 * IdMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.mongo.Mongo.Id}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.MongoMirrors.Id} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class IdMirror implements MongoMirrors.Id {
  public static final String QUALIFIED_NAME = "org.immutables.mongo.Mongo.Id";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.MongoMirrors.Id";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Id";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code IdMirror}, present if this annotation found
   */
  public static Optional<IdMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code IdMirror}, present if this annotation found
   */
  public static Optional<IdMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new IdMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code IdMirror}s
   */
  public static ImmutableList<IdMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<IdMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new IdMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code IdMirror}
   */
  public static IdMirror from(TypeElement element) {
    return new IdMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code IdMirror}, present if mirror matched this annotation type
   */
  public static Optional<IdMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;

  private IdMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

  }

  private IdMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Id.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return MongoMirrors.Id.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof IdMirror) {
      IdMirror otherMirror = (IdMirror) other;
      return true;
    }
    return false;
  }

  @Override
  public String toString() {
    return "IdMirror:" + annotationMirror;
  }
}
