package org.immutables.value.processor.meta;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Booleans;
import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;

/**
 * RepositoryMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.mongo.Mongo.Repository}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.MongoMirrors.Repository} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class RepositoryMirror implements MongoMirrors.Repository {
  public static final String QUALIFIED_NAME = "org.immutables.mongo.Mongo.Repository";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.MongoMirrors.Repository";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Repository";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code RepositoryMirror}, present if this annotation found
   */
  public static Optional<RepositoryMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code RepositoryMirror}, present if this annotation found
   */
  public static Optional<RepositoryMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new RepositoryMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code RepositoryMirror}s
   */
  public static ImmutableList<RepositoryMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<RepositoryMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new RepositoryMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code RepositoryMirror}
   */
  public static RepositoryMirror from(TypeElement element) {
    return new RepositoryMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code RepositoryMirror}, present if mirror matched this annotation type
   */
  public static Optional<RepositoryMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String value;
  private final String collection;
  private final boolean readonly;
  private final boolean index;

  private RepositoryMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String value = null;
    String collection = null;
    boolean readonly = false;
    boolean index = false;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Repository");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        continue;
      }
      if ("collection".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Repository");
        }
        CollectionExtractor collectionExtractor$ = new CollectionExtractor();
        annotationValue$.accept(collectionExtractor$, null);

        collection = collectionExtractor$.get();
        continue;
      }
      if ("readonly".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Repository");
        }
        ReadonlyExtractor readonlyExtractor$ = new ReadonlyExtractor();
        annotationValue$.accept(readonlyExtractor$, null);

        readonly = readonlyExtractor$.get();
        continue;
      }
      if ("index".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Repository");
        }
        IndexExtractor indexExtractor$ = new IndexExtractor();
        annotationValue$.accept(indexExtractor$, null);

        index = indexExtractor$.get();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "default attribute 'value'");
    this.collection = Preconditions.checkNotNull(collection, "default attribute 'collection'");
    this.readonly = readonly;
    this.index = index;
  }

  private RepositoryMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String value = null;
    String collection = null;
    boolean readonly = false;
    boolean index = false;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'value' attribute of @Repository");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        continue;
      }
      if ("collection".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'collection' attribute of @Repository");
        }
        CollectionExtractor collectionExtractor$ = new CollectionExtractor();
        annotationValue$.accept(collectionExtractor$, null);

        collection = collectionExtractor$.get();
        continue;
      }
      if ("readonly".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'readonly' attribute of @Repository");
        }
        ReadonlyExtractor readonlyExtractor$ = new ReadonlyExtractor();
        annotationValue$.accept(readonlyExtractor$, null);

        readonly = readonlyExtractor$.get();
        continue;
      }
      if ("index".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'index' attribute of @Repository");
        }
        IndexExtractor indexExtractor$ = new IndexExtractor();
        annotationValue$.accept(indexExtractor$, null);

        index = indexExtractor$.get();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "value for 'value'");
    this.collection = Preconditions.checkNotNull(collection, "value for 'collection'");
    this.readonly = readonly;
    this.index = index;
  }

  /**
   * @return value of attribute {@code value}
   */
  @Override
  public String value() {
    return value;
  }

  /**
   * @return value of attribute {@code collection}
   */
  @Override
  public String collection() {
    return collection;
  }

  /**
   * @return value of attribute {@code readonly}
   */
  @Override
  public boolean readonly() {
    return readonly;
  }

  /**
   * @return value of attribute {@code index}
   */
  @Override
  public boolean index() {
    return index;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Repository.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return MongoMirrors.Repository.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "value".hashCode() ^ value.hashCode();
    h += 127 * "collection".hashCode() ^ collection.hashCode();
    h += 127 * "readonly".hashCode() ^ Booleans.hashCode(readonly);
    h += 127 * "index".hashCode() ^ Booleans.hashCode(index);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof RepositoryMirror) {
      RepositoryMirror otherMirror = (RepositoryMirror) other;
      return value.equals(otherMirror.value)
          && collection.equals(otherMirror.collection)
          && readonly == otherMirror.readonly
          && index == otherMirror.index;
    }
    return false;
  }

  @Override
  public String toString() {
    return "RepositoryMirror:" + annotationMirror;
  }

  private static class ValueExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'value' in @" + QUALIFIED_NAME);
    }
  }

  private static class CollectionExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'collection' in @" + QUALIFIED_NAME);
    }
  }

  private static class ReadonlyExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'readonly' in @" + QUALIFIED_NAME);
    }
  }

  private static class IndexExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'index' in @" + QUALIFIED_NAME);
    }
  }
}
