package org.immutables.value.processor.meta;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
/**
 * SerializedNameMirror used to parse data of AnnotationMirror for original annotation {@code com.google.gson.annotations.SerializedName}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.GsonMirrors.SerializedName} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class SerializedNameMirror implements GsonMirrors.SerializedName {
  public static final String QUALIFIED_NAME = "com.google.gson.annotations.SerializedName";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.GsonMirrors.SerializedName";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "SerializedName";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code SerializedNameMirror}, present if this annotation found
   */
  public static Optional<SerializedNameMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code SerializedNameMirror}, present if this annotation found
   */
  public static Optional<SerializedNameMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new SerializedNameMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code SerializedNameMirror}s
   */
  public static ImmutableList<SerializedNameMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<SerializedNameMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new SerializedNameMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code SerializedNameMirror}
   */
  public static SerializedNameMirror from(TypeElement element) {
    return new SerializedNameMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code SerializedNameMirror}, present if mirror matched this annotation type
   */
  public static Optional<SerializedNameMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String value;
  private final String[] alternate;

  private SerializedNameMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String value = null;
    String[] alternate = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @SerializedName");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        continue;
      }
      if ("alternate".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @SerializedName");
        }
        AlternateExtractor alternateExtractor$ = new AlternateExtractor();
        annotationValue$.accept(alternateExtractor$, null);

        alternate = alternateExtractor$.get();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "default attribute 'value'");
    this.alternate = Preconditions.checkNotNull(alternate, "default attribute 'alternate'");
  }

  private SerializedNameMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String value = null;
    String[] alternate = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'value' attribute of @SerializedName");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        continue;
      }
      if ("alternate".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'alternate' attribute of @SerializedName");
        }
        AlternateExtractor alternateExtractor$ = new AlternateExtractor();
        annotationValue$.accept(alternateExtractor$, null);

        alternate = alternateExtractor$.get();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "value for 'value'");
    this.alternate = Preconditions.checkNotNull(alternate, "value for 'alternate'");
  }

  /**
   * @return value of attribute {@code value}
   */
  @Override
  public String value() {
    return value;
  }

  /**
   * @return value of attribute {@code alternate}
   */
  @Override
  public String[] alternate() {
    return alternate.clone();
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code SerializedName.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return GsonMirrors.SerializedName.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "value".hashCode() ^ value.hashCode();
    h += 127 * "alternate".hashCode() ^ Arrays.hashCode(alternate);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof SerializedNameMirror) {
      SerializedNameMirror otherMirror = (SerializedNameMirror) other;
      return value.equals(otherMirror.value)
          && Arrays.equals(alternate, otherMirror.alternate);
    }
    return false;
  }

  @Override
  public String toString() {
    return "SerializedNameMirror:" + annotationMirror;
  }

  private static class ValueExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'value' in @" + QUALIFIED_NAME);
    }
  }

  private static class AlternateExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'alternate' in @" + QUALIFIED_NAME);
    }
  }
}
