package org.immutables.value.processor.meta;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;

/**
 * IncludeMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.value.Value.Include}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.ValueMirrors.Include} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class IncludeMirror implements ValueMirrors.Include {
  public static final String QUALIFIED_NAME = "org.immutables.value.Value.Include";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.ValueMirrors.Include";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Include";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code IncludeMirror}, present if this annotation found
   */
  public static Optional<IncludeMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code IncludeMirror}, present if this annotation found
   */
  public static Optional<IncludeMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new IncludeMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code IncludeMirror}s
   */
  public static ImmutableList<IncludeMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<IncludeMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new IncludeMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code IncludeMirror}
   */
  public static IncludeMirror from(TypeElement element) {
    return new IncludeMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code IncludeMirror}, present if mirror matched this annotation type
   */
  public static Optional<IncludeMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final TypeMirror[] value;
  private final String[] valueName;

  private IncludeMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    TypeMirror[] value = null;
    String[] valueName = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Include");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        valueName = valueExtractor$.name();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "default attribute 'value'");
    this.valueName = Preconditions.checkNotNull(valueName, "default attribute 'value'");
  }

  private IncludeMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    TypeMirror[] value = null;
    String[] valueName = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'value' attribute of @Include");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        valueName = valueExtractor$.name();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "value for 'value'");
    this.valueName = Preconditions.checkNotNull(valueName, "Value for 'value'");
  }

  /**
   * @return type name for value of attribute {@code value}
   */
  public String[] valueName() {
    return valueName.clone();
  }

  /**
   * @return type mirror for value of attribute {@code value}
   */
  public TypeMirror[] valueMirror() {
    return value.clone();
  }

  /**
   * @deprecated Always throws UnsupportedOperationException. Use {@link #valueMirror} or {@link valueName}.
   */
  @Deprecated
  @Override
  public java.lang.Class<?>[] value() {
    throw new UnsupportedOperationException("Use 'valueMirror()' or 'valueName()'");
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Include.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return ValueMirrors.Include.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "value".hashCode() ^ Arrays.hashCode(valueName);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof IncludeMirror) {
      IncludeMirror otherMirror = (IncludeMirror) other;
      return Arrays.equals(valueName, otherMirror.valueName);
    }
    return false;
  }

  @Override
  public String toString() {
    return "IncludeMirror:" + annotationMirror;
  }

  private static class ValueExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    TypeMirror[] values;
    int position;

    @Override
    public Void visitType(TypeMirror value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new TypeMirror[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    TypeMirror[] get() {
      return values;
    }

    public String[] name() {
      String[] names = new String[values.length];
      for (int i = 0; i < values.length; i++) {
        names[i] = values[i].toString();
      }
      return names;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'value' in @" + QUALIFIED_NAME);
    }
  }
}
