package org.immutables.value.processor.meta;

import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;

/**
 * OkNamedMirror used to parse data of AnnotationMirror for original annotation {@code com.squareup.moshi.Json}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.OkJsonMirrors.OkNamed} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class OkNamedMirror implements OkJsonMirrors.OkNamed {
  public static final String QUALIFIED_NAME = "com.squareup.moshi.Json";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.OkJsonMirrors.OkNamed";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Json";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code OkNamedMirror}, present if this annotation found
   */
  public static Optional<OkNamedMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code OkNamedMirror}, present if this annotation found
   */
  public static Optional<OkNamedMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new OkNamedMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code OkNamedMirror}s
   */
  public static ImmutableList<OkNamedMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<OkNamedMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new OkNamedMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code OkNamedMirror}
   */
  public static OkNamedMirror from(TypeElement element) {
    return new OkNamedMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code OkNamedMirror}, present if mirror matched this annotation type
   */
  public static Optional<OkNamedMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String name;

  private OkNamedMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String name = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("name".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Json");
        }
        NameExtractor nameExtractor$ = new NameExtractor();
        annotationValue$.accept(nameExtractor$, null);

        name = nameExtractor$.get();
        continue;
      }
    }
    this.name = Preconditions.checkNotNull(name, "default attribute 'name'");
  }

  private OkNamedMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String name = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("name".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'name' attribute of @Json");
        }
        NameExtractor nameExtractor$ = new NameExtractor();
        annotationValue$.accept(nameExtractor$, null);

        name = nameExtractor$.get();
        continue;
      }
    }
    this.name = Preconditions.checkNotNull(name, "value for 'name'");
  }

  /**
   * @return value of attribute {@code name}
   */
  @Override
  public String name() {
    return name;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code OkNamed.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return OkJsonMirrors.OkNamed.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "name".hashCode() ^ name.hashCode();
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof OkNamedMirror) {
      OkNamedMirror otherMirror = (OkNamedMirror) other;
      return name.equals(otherMirror.name);
    }
    return false;
  }

  @Override
  public String toString() {
    return "OkNamedMirror:" + annotationMirror;
  }

  private static class NameExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'name' in @" + QUALIFIED_NAME);
    }
  }
}
