package org.immutables.value.processor.meta;

import java.lang.annotation.Annotation;
import java.util.Collections;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;

/**
 * RedactedMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.value.Value.Redacted}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.ValueMirrors.Redacted} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class RedactedMirror implements ValueMirrors.Redacted {
  public static final String QUALIFIED_NAME = "org.immutables.value.Value.Redacted";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.ValueMirrors.Redacted";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Redacted";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code RedactedMirror}, present if this annotation found
   */
  public static Optional<RedactedMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code RedactedMirror}, present if this annotation found
   */
  public static Optional<RedactedMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new RedactedMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code RedactedMirror}s
   */
  public static ImmutableList<RedactedMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<RedactedMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new RedactedMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code RedactedMirror}
   */
  public static RedactedMirror from(TypeElement element) {
    return new RedactedMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code RedactedMirror}, present if mirror matched this annotation type
   */
  public static Optional<RedactedMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;

  private RedactedMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

  }

  private RedactedMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Redacted.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return ValueMirrors.Redacted.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof RedactedMirror) {
      RedactedMirror otherMirror = (RedactedMirror) other;
      return true;
    }
    return false;
  }

  @Override
  public String toString() {
    return "RedactedMirror:" + annotationMirror;
  }
}
