package org.immutables.value.processor.meta;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import java.lang.annotation.Annotation;
import java.util.Collections;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;

/**
 * OkQualifierMirror used to parse data of AnnotationMirror for original annotation {@code com.squareup.moshi.JsonQualifier}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.OkJsonMirrors.OkQualifier} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class OkQualifierMirror implements OkJsonMirrors.OkQualifier {
  public static final String QUALIFIED_NAME = "com.squareup.moshi.JsonQualifier";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.OkJsonMirrors.OkQualifier";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "JsonQualifier";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code OkQualifierMirror}, present if this annotation found
   */
  public static Optional<OkQualifierMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code OkQualifierMirror}, present if this annotation found
   */
  public static Optional<OkQualifierMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new OkQualifierMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code OkQualifierMirror}s
   */
  public static ImmutableList<OkQualifierMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<OkQualifierMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new OkQualifierMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code OkQualifierMirror}
   */
  public static OkQualifierMirror from(TypeElement element) {
    return new OkQualifierMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code OkQualifierMirror}, present if mirror matched this annotation type
   */
  public static Optional<OkQualifierMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;

  private OkQualifierMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

  }

  private OkQualifierMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code OkQualifier.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return OkJsonMirrors.OkQualifier.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof OkQualifierMirror) {
      OkQualifierMirror otherMirror = (OkQualifierMirror) other;
      return true;
    }
    return false;
  }

  @Override
  public String toString() {
    return "OkQualifierMirror:" + annotationMirror;
  }
}
