/** Improving: An unconstrained collection of scala code.
 *  Copyright 2005-2010 Paul Phillips
 * 
 *  Distributed under the "Simplified BSD License" in LICENSE.txt.
 */

package improving
package security

/** LoggingSM
 *
 *  Inspired by the java version by Mark Petrovic available at
 *    http://www.onjava.com/pub/a/onjava/2007/01/03/discovering-java-security-requirements.html
 *
 * @author Paul Phillips <paulp@improving.org>
 */

import java.lang.reflect.Field
import java.net.URL
import java.security.{ AccessController, AccessControlContext, CodeSource, Permission, ProtectionDomain }
import scala.collection.{ mutable, immutable, generic }
import scala.util.Properties.{ javaHome, userHome }
import scala.tools.util.PathResolver
import scala.tools.nsc.{ GenericRunnerSettings, CompilerCommand, ObjectRunner, GenericRunnerCommand }
import scala.tools.nsc.io.File

class LoggingSM(permFilter: Permission => Boolean) extends SecurityManager {
  def this() = this(_ => true)
  import LoggingSM.Rule
  
  private var cache     = mutable.Set[Rule]()
  
  val thisCodeSource    = this.getClass.getProtectionDomain.getCodeSource
  val thisCodeSourceURL = thisCodeSource.getLocation.toString
  def codebases         = cache map (_.url) toList
  def perms             = (cache map (_.permStr)).toList.sorted.distinct
  def rules             = cache groupBy (_.url) toMap
  
  def asPolicyFile(cb: String): String = perms.mkString("grant " + cb + " {\n", "\n", "\n}\n")
  def asPolicyFile(cb: URL): String    = asPolicyFile("\"" + cb + "\"")
  def asPolicyFile(): String           = asPolicyFile("")
  
  // helper bits so we don't recurse forever on security checks we caused ourselves
  val recursionMarker               = (this.getClass.getName, "buildRules")   // name/class marker tuple
  def isRecur(x: StackTraceElement) = recursionMarker == (x.getClassName, x.getMethodName)
  def isRecursion(t: Throwable): Boolean = {
    // Writing this as:
    //    t.getStackTrace exists isRecur
    // causes an interesting ClassCircularityError by way of the closure.  Who knows.
    val arr = t.getStackTrace
    var index = 0
    while (index < arr.length) {
      if (isRecur(arr(index)))
        return true

      index += 1
    }
    false
  }
  
  // these two are the choke points for all permissions checks
  override def checkPermission(perm: Permission): Unit =
    checkPermission(perm, AccessController.getContext)  
  override def checkPermission(perm: Permission, context: Object): Unit = context match {
    case x: AccessControlContext if !isRecursion(new Throwable) => buildRules(perm, x)
    case _                                                      => ()
  }
  // overridden for our convenience -- just widening it to public
  override def getClassContext() = super.getClassContext()
  
  private def buildRules(perm: Permission, context: AccessControlContext) =
    getProtectionDomains(context) foreach (pd => addToCache(perm, pd))
  
  private def getProtectionDomains(context: AccessControlContext): List[ProtectionDomain] = {    
    for (fs <- Option(classOf[AccessControlContext].getDeclaredFields) ; f <- fs ; if f.getName == "context") {
      f setAccessible true
      (f get context) match {
        case xs: Array[ProtectionDomain]  => return xs.toList
        case _                            => ()
      }
    }
    Nil
  }
  
  def install(): Boolean = {
    try   { System.setSecurityManager(this) ; true }
    catch { case _: SecurityException => false }
  }
  
  def addToCache(perm: Permission, pd: ProtectionDomain) = {
    if (permFilter(perm))
      Rule(perm, pd) foreach (cache += _)
  }
  override def toString() = "SecurityManager (" + cache.size + " rules in cache)"
}

object LoggingSM {	
  val defaultPolicyFiles  = List(javaHome + "/lib/security/java.policy", userHome + "/.java.policy") map (x => File(x))
  val defaultSecurityFile = File(javaHome + "/lib/security/java.security")
  
  def withManager[T](sm: SecurityManager)(body: => T): T = {
    val oldsm = System.getSecurityManager

    try { System.setSecurityManager(sm) ; body }
    finally System.setSecurityManager(oldsm)    
  }
  
  def logging[T](body: => Unit): LoggingSM = {
    val sm = new LoggingSM()
    withManager(sm) { body ; sm }
  }

  def main(args: Array[String]): Unit = {
    val thisProg = this.getClass.getName.stripSuffix("$")
    if (args == null || args.isEmpty)
      return println("Usage: scala " + thisProg + " <class> <arguments>")

    val s  = new GenericRunnerSettings(println)
    s.howtorun.value = "object"
    s.nocompdaemon.value = true
    val urls = new PathResolver(s).asURLs

    val sm = logging {            
      try ObjectRunner.run(urls, args.head, args.tail)
      catch { case _ => () }
    }
    
    println("")
    println("// security policy autogenerated by " + thisProg + " with args:")
    println("//   " + args.mkString(" "))
    println("\n" + sm.asPolicyFile)
  }
  
  // factory method validates input and invokes private constructor
  object Rule {
    def apply(perm: Permission, pd: ProtectionDomain): Option[Rule] = {
      for (cs <- Option(pd.getCodeSource) ; url <- Option(cs.getLocation)) yield      
        new Rule(perm, pd, url)
    }
  }
  class Rule private (
    val perm: Permission,
    val pd: ProtectionDomain,
    val url: URL)
  {
    def esc(s: String) = s.replace("\"","\\\"").replace("\r","\\\r") 
    lazy val permStr  = """  permission %s "%s", "%s";""".format(perm.getClass.getName, esc(perm.getName), perm.getActions)

    override def toString = permStr
  }
  implicit val ruleOrdering: Ordering[Rule] = Ordering[String] on (_.toString)
}
