package org.infinispan.functional.impl;

import org.infinispan.commands.functional.WriteOnlyKeyCommand;
import org.infinispan.commands.functional.WriteOnlyKeyValueCommand;
import org.infinispan.commands.functional.WriteOnlyManyCommand;
import org.infinispan.commands.functional.WriteOnlyManyEntriesCommand;
import org.infinispan.commons.api.functional.EntryView.WriteEntryView;
import org.infinispan.commons.api.functional.FunctionalMap.WriteOnlyMap;
import org.infinispan.commons.api.functional.Listeners.WriteListeners;
import org.infinispan.commons.api.functional.Param;
import org.infinispan.commons.api.functional.Param.WaitMode;
import org.infinispan.commons.util.CloseableIterator;
import org.infinispan.commons.util.CloseableIteratorSet;
import org.infinispan.context.InvocationContext;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Stream;

import static org.infinispan.functional.impl.WaitModes.withWaitFuture;
import static org.infinispan.functional.impl.WaitModes.withWaitIterator;

public class WriteOnlyMapImpl<K, V> extends AbstractFunctionalMap<K, V> implements WriteOnlyMap<K, V> {
   private static final Log log = LogFactory.getLog(WriteOnlyMapImpl.class);
   private final Params params;

   private WriteOnlyMapImpl(Params params, FunctionalMapImpl<K, V> functionalMap) {
      super(functionalMap);
      this.params = params;
   }

   public static <K, V> WriteOnlyMap<K, V> create(FunctionalMapImpl<K, V> functionalMap) {
      return new WriteOnlyMapImpl<>(Params.from(functionalMap.params.params), functionalMap);
   }

   private static <K, V> WriteOnlyMap<K, V> create(Params params, FunctionalMapImpl<K, V> functionalMap) {
      return new WriteOnlyMapImpl<>(params, functionalMap);
   }

   @Override
   public CompletableFuture<Void> eval(K key, Consumer<WriteEntryView<V>> f) {
      log.tracef("Invoked eval(k=%s, %s)", key, params);
      Param<WaitMode> waitMode = params.get(WaitMode.ID);
      WriteOnlyKeyCommand cmd = fmap.cmdFactory().buildWriteOnlyKeyCommand(key, f);
      InvocationContext ctx = fmap.invCtxFactory().createInvocationContext(true, 1);
      return withWaitFuture(waitMode, fmap.asyncExec(), () -> (Void) fmap.chain().invoke(ctx, cmd));
   }

   @Override
   public CompletableFuture<Void> eval(K key, V value, BiConsumer<V, WriteEntryView<V>> f) {
      log.tracef("Invoked eval(k=%s, v=%s, %s)", key, value, params);
      Param<WaitMode> waitMode = params.get(WaitMode.ID);
      WriteOnlyKeyValueCommand cmd = fmap.cmdFactory().buildWriteOnlyKeyValueCommand(key, value, f);
      InvocationContext ctx = fmap.invCtxFactory().createInvocationContext(true, 1);
      return withWaitFuture(waitMode, fmap.asyncExec(), () -> (Void) fmap.chain().invoke(ctx, cmd));
   }

   @Override
   public CloseableIterator<Void> evalMany(Map<? extends K, ? extends V> entries, BiConsumer<V, WriteEntryView<V>> f) {
      log.tracef("Invoked evalMany(entries=%s, %s)", entries, params);
      Param<WaitMode> waitMode = params.get(WaitMode.ID);
      WriteOnlyManyEntriesCommand cmd = fmap.cmdFactory().buildWriteOnlyManyEntriesCommand(entries, f);
      InvocationContext ctx = fmap.invCtxFactory().createInvocationContext(true, entries.size());
      return withWaitIterator(waitMode, () -> (Stream<Void>) fmap.chain().invoke(ctx, cmd));
   }

   @Override
   public CloseableIterator<Void> evalMany(Set<? extends K> keys, Consumer<WriteEntryView<V>> f) {
      log.tracef("Invoked evalMany(keys=%s, %s)", keys, params);
      Param<WaitMode> waitMode = params.get(WaitMode.ID);
      WriteOnlyManyCommand cmd = fmap.cmdFactory().buildWriteOnlyManyCommand(keys, f);
      InvocationContext ctx = fmap.invCtxFactory().createInvocationContext(true, keys.size());
      return withWaitIterator(waitMode, () -> (Stream<Void>) fmap.chain().invoke(ctx, cmd));
   }

   @Override
   public CloseableIterator<Void> evalAll(Consumer<WriteEntryView<V>> f) {
      log.tracef("Invoked evalAll(%s)", params);
      Param<WaitMode> waitMode = params.get(WaitMode.ID);
      CloseableIteratorSet<K> keys = fmap.cache.keySet();
      WriteOnlyManyCommand cmd = fmap.cmdFactory().buildWriteOnlyManyCommand(keys, f);
      InvocationContext ctx = fmap.invCtxFactory().createInvocationContext(true, keys.size());
      return withWaitIterator(waitMode, () -> (Stream<Void>) fmap.chain().invoke(ctx, cmd));
   }

   @Override
   public CompletableFuture<Void> truncate() {
      log.tracef("Invoked truncate(%s)", params);
      Param<WaitMode> waitMode = params.get(WaitMode.ID);
      return withWaitFuture(waitMode, fmap.asyncExec(), () -> {
         fmap.cache.clear();
         return null;
      });
   }

   @Override
   public WriteOnlyMap<K, V> withParams(Param<?>... ps) {
      if (ps == null || ps.length == 0)
         return this;

      if (params.containsAll(ps))
         return this; // We already have all specified params

      return create(params.addAll(ps), fmap);
   }

   @Override
   public WriteListeners<K, V> listeners() {
      return fmap.notifier();
   }

}
