/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.util.concurrent;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import org.infinispan.commands.CommandInvocationId;
import org.infinispan.remoting.transport.Address;
import org.infinispan.statetransfer.OutdatedTopologyException;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

public class CommandAckCollector {
    private static final Log log = LogFactory.getLog(CommandAckCollector.class);
    private static final boolean trace = log.isTraceEnabled();
    private final ConcurrentHashMap<CommandInvocationId, Collector<?>> collectorMap = new ConcurrentHashMap();

    public void create(CommandInvocationId id, Collection<Address> owners, int topologyId) {
        this.collectorMap.putIfAbsent(id, new SingleKeyCollector(id, owners, topologyId));
        if (trace) {
            log.tracef("Created new collector for %s. Owners=%s", (Object)id, (Object)owners);
        }
    }

    public void create(CommandInvocationId id, Object returnValue, Collection<Address> owners, int topologyId) {
        this.collectorMap.putIfAbsent(id, new SingleKeyCollector(id, returnValue, owners, topologyId));
        if (trace) {
            log.tracef("Created new collector for %s. ReturnValue=%s. Owners=%s", (Object)id, returnValue, (Object)owners);
        }
    }

    public void createMultiKeyCollector(CommandInvocationId id, Collection<Address> primary, Map<Address, Collection<Integer>> backups, int topologyId) {
        this.collectorMap.putIfAbsent(id, new MultiKeyCollector(id, primary, backups, topologyId));
        if (trace) {
            log.tracef("Created new collector for %s. PrimarySegments=%s. BackupSegments", (Object)id, (Object)primary, (Object)backups);
        }
    }

    public void multiKeyPrimaryAck(CommandInvocationId id, Address from, Map<Object, Object> returnValue, int topologyId) {
        MultiKeyCollector collector = (MultiKeyCollector)this.collectorMap.get(id);
        if (collector != null) {
            collector.primaryAck(returnValue, from, topologyId);
        }
    }

    public void multiKeyBackupAck(CommandInvocationId id, Address from, Collection<Integer> segments, int topologyId) {
        MultiKeyCollector collector = (MultiKeyCollector)this.collectorMap.get(id);
        if (collector != null) {
            collector.backupAck(from, segments, topologyId);
        }
    }

    public void backupAck(CommandInvocationId id, Address from, int topologyId) {
        SingleKeyCollector collector = (SingleKeyCollector)this.collectorMap.get(id);
        if (collector != null) {
            collector.backupAck(topologyId, from);
        }
    }

    public void primaryAck(CommandInvocationId id, Object returnValue, boolean success, Address from, int topologyId) {
        SingleKeyCollector collector = (SingleKeyCollector)this.collectorMap.get(id);
        if (collector != null) {
            collector.primaryAck(topologyId, returnValue, success, from);
        }
    }

    public void completeExceptionally(CommandInvocationId id, Throwable throwable, int topologyId) {
        Collector<?> collector = this.collectorMap.get(id);
        if (collector != null) {
            collector.completeExceptionally(throwable, topologyId);
        }
    }

    public <T> CompletableFuture<T> getCollectorCompletableFuture(CommandInvocationId id, boolean cleanupAfterCompleted) {
        Collector<?> collector = this.collectorMap.get(id);
        if (collector != null) {
            if (trace) {
                log.tracef("[Collector#%s] Waiting for acks asynchronously.", (Object)id);
            }
            CompletableFuture<?> future = collector.getFuture();
            if (cleanupAfterCompleted) {
                return future.whenComplete((t, throwable) -> this.collectorMap.remove(id));
            }
            return future;
        }
        return null;
    }

    public List<CommandInvocationId> getPendingCommands() {
        return new ArrayList<CommandInvocationId>(this.collectorMap.keySet());
    }

    public boolean hasPendingBackupAcks(CommandInvocationId id) {
        Collector<?> collector = this.collectorMap.get(id);
        return collector != null && collector.hasPendingBackupAcks();
    }

    public void onMembersChange(Collection<Address> members) {
        HashSet<Address> currentMembers = new HashSet<Address>(members);
        for (Collector<?> collector : this.collectorMap.values()) {
            collector.onMembersChange(currentMembers);
        }
    }

    public void dispose(CommandInvocationId id) {
        if (trace) {
            log.tracef("[Collector#%s] Dispose collector.", (Object)id);
        }
        this.collectorMap.remove(id);
    }

    private static class MultiKeyCollector
    implements Collector<Map<Object, Object>> {
        private final CommandInvocationId id;
        private Map<Object, Object> returnValue;
        private final Collection<Address> primary;
        private final Map<Address, Collection<Integer>> backups;
        private final CompletableFuture<Map<Object, Object>> future;
        private final int topologyId;

        MultiKeyCollector(CommandInvocationId id, Collection<Address> primary, Map<Address, Collection<Integer>> backups, int topologyId) {
            this.id = id;
            this.topologyId = topologyId;
            this.returnValue = null;
            this.backups = backups;
            this.primary = new HashSet<Address>(primary);
            this.future = new CompletableFuture();
        }

        @Override
        public synchronized void completeExceptionally(Throwable throwable, int topologyId) {
            if (trace) {
                log.tracef(throwable, "[Collector#%s] completed exceptionally. TopologyId=%s (expected=%s)", (Object)this.id, (Object)topologyId, (Object)this.topologyId);
            }
            if (this.topologyId != topologyId) {
                return;
            }
            this.doCompleteExceptionally(throwable);
        }

        @Override
        public synchronized boolean hasPendingBackupAcks() {
            return !this.backups.isEmpty();
        }

        @Override
        public CompletableFuture<Map<Object, Object>> getFuture() {
            return this.future;
        }

        @Override
        public void onMembersChange(Collection<Address> members) {
            if (!members.containsAll(this.primary)) {
                if (trace) {
                    log.tracef("[Collector#%s] A primary Owner left the cluster.", (Object)this.id);
                }
                this.doCompleteExceptionally(OutdatedTopologyException.getCachedInstance());
            } else if (this.backups.keySet().retainAll(members)) {
                if (trace) {
                    log.tracef("[Collector#%s] Some backups left the cluster.", (Object)this.id);
                }
                this.checkCompleted();
            }
        }

        synchronized void primaryAck(Map<Object, Object> returnValue, Address from, int topologyId) {
            if (trace) {
                log.tracef("[Collector#%s] PutMap Primary ACK. Address=%s. TopologyId=%s (expected=%s)", this.id, from, topologyId, this.topologyId);
            }
            if (this.topologyId != topologyId) {
                return;
            }
            if (returnValue != null) {
                if (this.returnValue == null) {
                    this.returnValue = new HashMap<Object, Object>(returnValue.size());
                }
                this.returnValue.putAll(returnValue);
            }
            if (this.primary.remove(from)) {
                this.checkCompleted();
            }
        }

        synchronized void backupAck(Address from, Collection<Integer> segments, int topologyId) {
            if (trace) {
                log.tracef("[Collector#%s] PutMap Backup ACK. Address=%s. TopologyId=%s (expected=%s). Segments=%s", this.id, from, topologyId, this.topologyId, segments);
            }
            if (this.topologyId != topologyId) {
                return;
            }
            Collection pendingSegments = this.backups.getOrDefault(from, Collections.emptyList());
            if (pendingSegments.removeAll(segments) && pendingSegments.isEmpty()) {
                this.backups.remove(from);
                this.checkCompleted();
            }
        }

        private void checkCompleted() {
            if (this.primary.isEmpty() && this.backups.isEmpty()) {
                if (trace) {
                    log.tracef("[Collector#%s] Ready! Return value=%ss.", (Object)this.id, (Object)this.returnValue);
                }
                this.future.complete(this.returnValue);
            }
        }

        private void doCompleteExceptionally(Throwable throwable) {
            this.returnValue = null;
            this.primary.clear();
            this.backups.clear();
            this.future.completeExceptionally(throwable);
        }
    }

    private static class SingleKeyCollector
    implements Collector<Object> {
        private final CommandInvocationId id;
        private final CompletableFuture<Object> future;
        private final Collection<Address> owners;
        private final Address primaryOwner;
        private final int topologyId;
        private Object returnValue;

        private SingleKeyCollector(CommandInvocationId id, Collection<Address> owners, int topologyId) {
            this.id = id;
            this.primaryOwner = owners.iterator().next();
            this.topologyId = topologyId;
            this.future = new CompletableFuture();
            this.owners = new HashSet<Address>(owners);
        }

        private SingleKeyCollector(CommandInvocationId id, Object returnValue, Collection<Address> owners, int topologyId) {
            this.id = id;
            this.returnValue = returnValue;
            this.primaryOwner = owners.iterator().next();
            this.topologyId = topologyId;
            HashSet<Address> tmpOwners = new HashSet<Address>(owners);
            tmpOwners.remove(this.primaryOwner);
            if (tmpOwners.isEmpty()) {
                this.owners = Collections.emptyList();
                this.future = CompletableFuture.completedFuture(returnValue);
            } else {
                this.future = new CompletableFuture();
                this.owners = tmpOwners;
            }
        }

        @Override
        public synchronized void completeExceptionally(Throwable throwable, int topologyId) {
            if (trace) {
                log.tracef(throwable, "[Collector#%s] completed exceptionally. TopologyId=%s (expected=%s)", (Object)this.id, (Object)topologyId, (Object)this.topologyId);
            }
            if (this.topologyId != topologyId) {
                return;
            }
            this.doCompleteExceptionally(throwable);
        }

        @Override
        public synchronized boolean hasPendingBackupAcks() {
            return this.owners.size() > 1 || this.owners.size() == 1 && !this.primaryOwner.equals(this.owners.iterator().next());
        }

        @Override
        public CompletableFuture<Object> getFuture() {
            return this.future;
        }

        @Override
        public synchronized void onMembersChange(Collection<Address> members) {
            if (!members.contains(this.primaryOwner)) {
                if (trace) {
                    log.tracef("[Collector#%s] The Primary Owner left the cluster.", (Object)this.id);
                }
                this.doCompleteExceptionally(OutdatedTopologyException.getCachedInstance());
            } else if (this.owners.retainAll(members) && this.owners.isEmpty()) {
                if (trace) {
                    log.tracef("[Collector#%s] Some backups left the cluster.", (Object)this.id);
                }
                this.markReady();
            }
        }

        synchronized void primaryAck(int topologyId, Object returnValue, boolean success, Address from) {
            if (trace) {
                log.tracef("[Collector#%s] Primary ACK. Success=%s. ReturnValue=%s. Address=%s, TopologyId=%s (expected=%s)", this.id, success, returnValue, from, topologyId, this.topologyId);
            }
            if (this.topologyId != topologyId || !this.owners.remove(from)) {
                return;
            }
            this.returnValue = returnValue;
            if (!success) {
                this.owners.clear();
                this.future.complete(returnValue);
                if (trace) {
                    log.tracef("[Collector#%s] Ready! Command not succeed on primary.", (Object)this.id);
                }
            } else if (this.owners.isEmpty()) {
                this.markReady();
            }
        }

        synchronized void backupAck(int topologyId, Address from) {
            if (trace) {
                log.tracef("[Collector#%s] Backup ACK. Address=%s, TopologyId=%s (expected=%s)", this.id, from, topologyId, this.topologyId);
            }
            if (this.topologyId == topologyId && this.owners.remove(from) && this.owners.isEmpty()) {
                this.markReady();
            }
        }

        private void markReady() {
            if (trace) {
                log.tracef("[Collector#%s] Ready! Return value=%ss.", (Object)this.id, this.returnValue);
            }
            this.future.complete(this.returnValue);
        }

        private void doCompleteExceptionally(Throwable throwable) {
            this.owners.clear();
            this.future.completeExceptionally(throwable);
        }
    }

    private static interface Collector<T> {
        public void completeExceptionally(Throwable var1, int var2);

        public boolean hasPendingBackupAcks();

        public CompletableFuture<T> getFuture();

        public void onMembersChange(Collection<Address> var1);
    }
}

