/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.persistence.sifs;

import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.infinispan.commons.marshall.Marshaller;
import org.infinispan.persistence.sifs.EntryHeader;
import org.infinispan.persistence.sifs.EntryPosition;
import org.infinispan.persistence.sifs.EntryRecord;
import org.infinispan.persistence.sifs.FileProvider;
import org.infinispan.persistence.sifs.Index;
import org.infinispan.persistence.sifs.IndexRequest;
import org.infinispan.persistence.sifs.TemporaryTable;
import org.infinispan.util.TimeService;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

class Compactor
extends Thread {
    private static final Log log = LogFactory.getLog(Compactor.class);
    private static final boolean trace = log.isTraceEnabled();
    private final ConcurrentMap<Integer, Stats> fileStats = new ConcurrentHashMap<Integer, Stats>();
    private final BlockingQueue<Integer> scheduledCompaction = new LinkedBlockingQueue<Integer>();
    private final BlockingQueue<IndexRequest> indexQueue;
    private final FileProvider fileProvider;
    private final TemporaryTable temporaryTable;
    private final Marshaller marshaller;
    private final TimeService timeService;
    private final int maxFileSize;
    private final double compactionThreshold;
    private Index index;
    private volatile boolean clearSignal = false;
    private volatile boolean terminateSignal = false;
    private volatile CountDownLatch compactorResume;
    private volatile CountDownLatch compactorStop;

    public Compactor(FileProvider fileProvider, TemporaryTable temporaryTable, BlockingQueue<IndexRequest> indexQueue, Marshaller marshaller, TimeService timeService, int maxFileSize, double compactionThreshold) {
        super("BCS-Compactor");
        this.fileProvider = fileProvider;
        this.temporaryTable = temporaryTable;
        this.indexQueue = indexQueue;
        this.marshaller = marshaller;
        this.timeService = timeService;
        this.maxFileSize = maxFileSize;
        this.compactionThreshold = compactionThreshold;
        this.start();
    }

    public void setIndex(Index index) {
        this.index = index;
    }

    public void releaseStats(int file) {
        this.fileStats.remove(file);
    }

    public void free(int file, int size) {
        if (file < 0) {
            return;
        }
        this.recordFreeSpace(this.getStats(file), file, size);
    }

    public void completeFile(int file) {
        Stats stats = this.getStats(file);
        stats.setCompleted();
        if (stats.readyToBeScheduled(this.compactionThreshold, stats.getFree())) {
            this.schedule(file, stats);
        }
    }

    private Stats getStats(int file) {
        int fileSize;
        Stats stats = (Stats)this.fileStats.get(file);
        if (stats == null) {
            fileSize = (int)this.fileProvider.getFileSize(file);
            stats = new Stats(fileSize, 0);
            Stats other = this.fileStats.putIfAbsent(file, stats);
            if (other != null) {
                if (fileSize > other.getTotal()) {
                    other.setTotal(fileSize);
                }
                return other;
            }
        }
        if (stats.getTotal() < 0 && (fileSize = (int)this.fileProvider.getFileSize(file)) >= 0) {
            stats.setTotal(fileSize);
        }
        return stats;
    }

    private void recordFreeSpace(Stats stats, int file, int size) {
        if (stats.addFree(size, this.compactionThreshold)) {
            this.schedule(file, stats);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void schedule(int file, Stats stats) {
        try {
            Stats stats2 = stats;
            synchronized (stats2) {
                if (!stats.isScheduled()) {
                    log.debug((Object)String.format("Scheduling file %d for compaction: %d/%d free", file, stats.free.get(), stats.total));
                    stats.setScheduled();
                    this.scheduledCompaction.put(file);
                }
            }
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        try {
            FileProvider.Log logFile = null;
            int currentOffset = 0;
            while (true) {
                Integer scheduledFile = null;
                try {
                    scheduledFile = this.scheduledCompaction.poll(1L, TimeUnit.MINUTES);
                }
                catch (InterruptedException e) {
                    // empty catch block
                }
                if (this.terminateSignal) {
                    if (logFile != null) {
                        logFile.close();
                        this.completeFile(logFile.fileId);
                    }
                    break;
                }
                if (this.clearSignal) {
                    this.pauseCompactor(logFile);
                    logFile = null;
                    continue;
                }
                if (scheduledFile == null) {
                    if (logFile == null) continue;
                    logFile.close();
                    this.completeFile(logFile.fileId);
                    logFile = null;
                    continue;
                }
                if (log.isDebugEnabled()) {
                    log.debugf("Compacting file %d", (Object)scheduledFile);
                }
                int scheduledOffset = 0;
                FileProvider.Handle handle = this.fileProvider.getFile(scheduledFile);
                if (handle == null) {
                    throw new IllegalStateException("Compactor should not get deleted file for compaction!");
                }
                try {
                    while (!this.clearSignal && !this.terminateSignal) {
                        int entryOffset;
                        byte[] serializedValue;
                        byte[] serializedMetadata;
                        EntryHeader header = EntryRecord.readEntryHeader(handle, scheduledOffset);
                        if (header == null) {
                            break;
                        }
                        byte[] serializedKey = EntryRecord.readKey(handle, header, scheduledOffset);
                        Object key = this.marshaller.objectFromByteBuffer(serializedKey);
                        boolean drop = true;
                        EntryPosition entry = this.temporaryTable.get(key);
                        if (entry != null) {
                            EntryPosition entryPosition = entry;
                            synchronized (entryPosition) {
                                if (entry.file == scheduledFile && entry.offset == scheduledOffset) {
                                    drop = false;
                                } else if (trace) {
                                    log.tracef("Key for %d:%d was found in temporary table on %d:%d", new Object[]{scheduledFile, scheduledOffset, entry.file, entry.offset});
                                }
                            }
                        } else {
                            EntryPosition position = this.index.getPosition(key, serializedKey);
                            if (position != null && position.file == scheduledFile && position.offset == scheduledOffset) {
                                drop = false;
                            } else if (trace) {
                                if (position != null) {
                                    log.tracef("Key for %d:%d was found in index on %d:%d", new Object[]{scheduledFile, scheduledOffset, position.file, position.offset});
                                } else {
                                    log.tracef("Key for %d:%d was not found in index!", (Object)scheduledFile, (Object)scheduledOffset);
                                }
                            }
                        }
                        if (drop) {
                            if (trace) {
                                log.tracef("Drop %d:%d (%s)", (Object)scheduledFile, (Object)scheduledOffset, (Object)(header.valueLength() > 0 ? "record" : "tombstone"));
                            }
                            scheduledOffset += header.totalLength();
                            continue;
                        }
                        if (logFile == null || currentOffset + header.totalLength() > this.maxFileSize) {
                            if (logFile != null) {
                                logFile.close();
                                this.completeFile(logFile.fileId);
                            }
                            currentOffset = 0;
                            logFile = this.fileProvider.getFileForLog();
                            if (log.isDebugEnabled()) {
                                log.debugf("Compacting to %d", (Object)logFile.fileId);
                            }
                        }
                        if (header.valueLength() > 0 && (header.expiryTime() < 0L || header.expiryTime() > this.timeService.wallClockTime())) {
                            serializedMetadata = EntryRecord.readMetadata(handle, header, scheduledOffset);
                            serializedValue = EntryRecord.readValue(handle, header, scheduledOffset);
                            entryOffset = currentOffset;
                        } else {
                            serializedMetadata = null;
                            serializedValue = null;
                            entryOffset = ~currentOffset;
                        }
                        EntryRecord.writeEntry(logFile.fileChannel, serializedKey, serializedMetadata, serializedValue, header.seqId(), header.expiryTime());
                        this.temporaryTable.setConditionally(key, logFile.fileId, entryOffset, scheduledFile, scheduledOffset);
                        if (trace) {
                            log.tracef("Update %d:%d -> %d:%d | %d,%d", new Object[]{scheduledFile, scheduledOffset, logFile.fileId, entryOffset, logFile.fileChannel.position(), logFile.fileChannel.size()});
                        }
                        this.indexQueue.put(new IndexRequest(key, serializedKey, logFile.fileId, entryOffset, header.totalLength(), scheduledFile, scheduledOffset));
                        currentOffset += header.totalLength();
                        scheduledOffset += header.totalLength();
                    }
                }
                finally {
                    handle.close();
                }
                if (this.terminateSignal) {
                    if (logFile != null) {
                        logFile.close();
                        this.completeFile(logFile.fileId);
                    }
                    return;
                }
                if (this.clearSignal) {
                    this.pauseCompactor(logFile);
                    logFile = null;
                    continue;
                }
                log.debugf("Finished compacting %d, scheduling delete", (Object)scheduledFile);
                this.indexQueue.put(IndexRequest.deleteFileRequest(scheduledFile));
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    private void pauseCompactor(FileProvider.Log logFile) throws IOException, InterruptedException {
        if (logFile != null) {
            logFile.close();
            this.completeFile(logFile.fileId);
        }
        this.compactorStop.countDown();
        this.compactorResume.await();
    }

    public void clearAndPause() throws InterruptedException {
        this.compactorResume = new CountDownLatch(1);
        this.compactorStop = new CountDownLatch(1);
        this.clearSignal = true;
        this.scheduledCompaction.put(-1);
        this.compactorStop.await();
        this.scheduledCompaction.clear();
        this.fileStats.clear();
    }

    public void resumeAfterPause() {
        this.clearSignal = false;
        this.compactorResume.countDown();
    }

    public void stopOperations() throws InterruptedException {
        this.terminateSignal = true;
        this.scheduledCompaction.put(-1);
        this.join();
    }

    private static class Stats {
        private final AtomicInteger free;
        private volatile int total;
        private volatile boolean completed = false;
        private volatile boolean scheduled = false;

        private Stats(int total, int free) {
            this.free = new AtomicInteger(free);
            this.total = total;
        }

        public int getTotal() {
            return this.total;
        }

        public void setTotal(int total) {
            this.total = total;
        }

        public boolean addFree(int size, double compactionThreshold) {
            int free = this.free.addAndGet(size);
            return this.readyToBeScheduled(compactionThreshold, free);
        }

        public int getFree() {
            return this.free.get();
        }

        public boolean readyToBeScheduled(double compactionThreshold, int free) {
            int total = this.total;
            return this.completed && !this.scheduled && total >= 0 && (double)free > (double)total * compactionThreshold;
        }

        public boolean isScheduled() {
            return this.scheduled;
        }

        public void setScheduled() {
            this.scheduled = true;
        }

        public boolean isCompleted() {
            return this.completed;
        }

        public void setCompleted() {
            this.completed = true;
        }
    }
}

