/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.infinispan.query.impl;

import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

import org.hibernate.search.cfg.SearchConfiguration;
import org.hibernate.search.spi.SearchFactoryBuilder;
import org.hibernate.search.spi.SearchFactoryIntegrator;
import org.infinispan.config.Configuration;
import org.infinispan.config.CustomInterceptorConfig;
import org.infinispan.factories.ComponentRegistry;
import org.infinispan.interceptors.DistLockingInterceptor;
import org.infinispan.interceptors.InterceptorChain;
import org.infinispan.interceptors.LockingInterceptor;
import org.infinispan.interceptors.base.CommandInterceptor;
import org.infinispan.lifecycle.AbstractModuleLifecycle;
import org.infinispan.query.backend.LocalQueryInterceptor;
import org.infinispan.query.backend.QueryInterceptor;
import org.infinispan.query.backend.SearchableCacheConfiguration;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

/**
 * Lifecycle of the Query module: initializes the Hibernate Search engine and shuts it down
 * it at cache stop.
 * 
 * @author Sanne Grinovero <sanne@hibernate.org> (C) 2011 Red Hat Inc.
 */
public class LifecycleManager extends AbstractModuleLifecycle {
   
   private static final Log log = LogFactory.getLog(LifecycleManager.class);
   
   private final Map<String,SearchFactoryIntegrator> searchFactoriesToShutdown = new TreeMap<String,SearchFactoryIntegrator>();

   /**
    * Registers the Search interceptor in the cache before it gets started
    */
   @Override
   public void cacheStarting(ComponentRegistry cr, Configuration configuration, String cacheName) {
      Configuration cfg = cr.getComponent(Configuration.class);
      if (cfg.isIndexingEnabled()) {
         log.info("Registering Query interceptor");
         SearchFactoryIntegrator searchFactory = getSearchFactory(cfg.getIndexingProperties(), cr);
         createQueryInterceptorIfNeeded(cr, cfg, searchFactory);
      }
   }
   
   private void createQueryInterceptorIfNeeded(ComponentRegistry cr, Configuration cfg, SearchFactoryIntegrator searchFactory) {
      QueryInterceptor queryInterceptor = cr.getComponent(QueryInterceptor.class);
      if (queryInterceptor == null) {
         queryInterceptor = buildQueryInterceptor(cfg, searchFactory);
         cr.registerComponent(queryInterceptor, QueryInterceptor.class);
         if (cfg.getCacheMode().isDistributed()) {
            cfg.getCustomInterceptors().add(
                  new CustomInterceptorConfig(queryInterceptor, false, false,
                        -1, DistLockingInterceptor.class.getName(), ""));
         } else {
            cfg.getCustomInterceptors().add(
                  new CustomInterceptorConfig(queryInterceptor, false, false,
                        -1, LockingInterceptor.class.getName(), ""));
         }
      }
   }

   private QueryInterceptor buildQueryInterceptor(Configuration cfg, SearchFactoryIntegrator searchFactory) {
      if ( cfg.isIndexLocalOnly() ) {
         return new LocalQueryInterceptor(searchFactory);
      }
      else {
         return new QueryInterceptor(searchFactory);
      }
   }

   @Override
   public void cacheStarted(ComponentRegistry cr, String cacheName) {
      if ( ! verifyChainContainsQueryInterceptor(cr) ) {
         throw new IllegalStateException( "It was expected to find the Query interceptor registered in the InterceptorChain but it wasn't found" );
      }
   }

   private boolean verifyChainContainsQueryInterceptor(ComponentRegistry cr) {
      Configuration cfg = cr.getComponent(Configuration.class);
      if (cfg.isIndexingEnabled()) {
         InterceptorChain interceptorChain = cr.getComponent(InterceptorChain.class);
         CommandInterceptor chainElement = interceptorChain.getFirstInChain();
         if (chainElement instanceof QueryInterceptor) {
            return true;
         }
         while (chainElement.hasNext()) {
            chainElement = chainElement.getNext();
            if (chainElement instanceof QueryInterceptor) {
               return true;
            }
         }
         return false;
      }
      else {
         return true;
      }
   }

   private SearchFactoryIntegrator getSearchFactory(Properties indexingProperties, ComponentRegistry cr) {
       SearchFactoryIntegrator searchFactory = cr.getComponent(SearchFactoryIntegrator.class);
       //defend against multiple initialization:
       if (searchFactory==null) {
          // Set up the search factory for Hibernate Search first.
          SearchConfiguration config = new SearchableCacheConfiguration(new Class[0], indexingProperties);
          searchFactory = new SearchFactoryBuilder().configuration(config).buildSearchFactory();
          cr.registerComponent(searchFactory, SearchFactoryIntegrator.class);
       }
      return searchFactory;
   }
   
   @Override
   public void cacheStopping(ComponentRegistry cr, String cacheName) {
      //TODO move this to cacheStopped event (won't work right now as the ComponentRegistry is half empty at that point: ISPN-1006)
      SearchFactoryIntegrator searchFactoryImplementor = cr.getComponent(SearchFactoryIntegrator.class);
      if (searchFactoryImplementor != null) {
         searchFactoriesToShutdown.put(cacheName, searchFactoryImplementor);
      }
   }
   
   @Override
   public void cacheStopped(ComponentRegistry cr, String cacheName) {
      SearchFactoryIntegrator searchFactoryIntegrator = searchFactoriesToShutdown.get(cacheName);
      if (searchFactoryIntegrator != null) {
         searchFactoryIntegrator.close();
      }
   }

}
