package org.infinitenature.herbariorum.client;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.infinitenature.herbariorum.entities.Institution;
import org.infinitenature.herbariorum.helper.InstitutionComparator;
import org.infinitenature.herbariorum.resource.CountryResource;
import org.infinitenature.herbariorum.resource.InstitutionResource;
import org.infinitenature.herbariorum.response.CountryResponse;
import org.infinitenature.herbariorum.response.InstitutionResponse;

/**
 * One InstantClient per Session. No longterm use.
 */
public class HerbariorumInstantClient implements  Client
{
   private Map<String, List<Institution>> institutionsByCountryAndName = new HashMap<>();
   private Map<String, Integer> countInstitutionsByCountryAndName = new HashMap<>();
   private Map<String, List<String>> countriesByName = new HashMap<>();
   private Map<String, Integer> countCountriesByName = new HashMap<>();

   private Herbariorum herbariorum = new Herbariorum(
         "http://sweetgum.nybg.org");

   /**
    * Parameter country is mandatory, no country returns no institutions.
    * Parameter name is a filter, emtpy filter returns all institutions in the country.
    * returns a list of institutions.
    *
    * @param country
    * @param name
    * @return
    */
   public List<Institution> findInstitutionsByCountryAndName(String country,
         String name)
   {
      if (country == null)
      {
         return new ArrayList<>();
      }
      if (institutionsByCountryAndName.get(country + name) != null)
      {
         return institutionsByCountryAndName.get(country + name);
      }
      InstitutionResource institutionResource = herbariorum.institutions();
      InstitutionResponse institutionResponse = institutionResource
            .findByCountry(country);

      List<Institution> institutions = institutionResponse.getInstitutions()
            .stream()
            .filter(institution -> institution.getOrganization().contains(name))
            .collect(Collectors.toList());

      Collections.sort(institutions, new InstitutionComparator());
      institutionsByCountryAndName.put(country + name, institutions);
      return institutions;
   }

   /**
    * Parameter country is mandatory, no country no institutions.
    * Parameter name is a filter, empty filter returns the number of all institutions in the country.
    * returns the number of institutions
    *
    * @param country
    * @param name
    * @return
    */
   public int countInstitutionsByCountryAndName(String country, String name)
   {
      if (country == null)
      {
         return 0;
      }
      if (countInstitutionsByCountryAndName.get(country + name) != null)
      {
         return countInstitutionsByCountryAndName.get(country + name);
      }
      InstitutionResource institutionResource = herbariorum.institutions();
      InstitutionResponse institutionResponse = institutionResource
            .findByCountry(country);

      int numberOfInstitutions = (int) institutionResponse.getInstitutions()
            .stream()
            .filter(institution -> institution.getOrganization().contains(name))
            .count();

      countInstitutionsByCountryAndName
            .put(country + name, numberOfInstitutions);

      return numberOfInstitutions;
   }

   /**
    * Parameter name is a filter, empty filter returns all countries.
    *
    * @param name
    * @return
    */
   public List<String> findCountriesByName(String name)
   {
      if (countriesByName.get(name) != null)
      {
         return countriesByName.get(name);
      }

      CountryResource countryResource = herbariorum.countries();
      CountryResponse countryResponse = countryResource.findCountries();

      List<String> countries = countryResponse.getCountries().stream()
            .filter(country -> country.contains(name))
            .collect(Collectors.toList());

      countriesByName.put(name, countries);

      return countries;
   }

   /**
    * Parameter name is a filter, empty filter returns number of all countries.
    *
    * @param name
    * @return
    */
   public int countCountriesByName(String name)
   {
      if(countCountriesByName.get(name) != null)
      {
         return countCountriesByName.get(name);
      }

      CountryResource countryResource = herbariorum.countries();
      CountryResponse countryResponse = countryResource.findCountries();

      int numberOfCountries = (int) countryResponse.getCountries().stream()
            .filter(country -> country.contains(name)).count();

      countCountriesByName.put(name, numberOfCountries);

      return numberOfCountries;
   }

   public void setHerbariorum(Herbariorum herbariorum)
   {
      this.herbariorum = herbariorum;
   }

   @Override
   public void setInstitutions(List<Institution> institutions)
   {

   }

   @Override
   public void setCountries(Collection<String> countries)
   {

   }

   @Override
   public void setGermanCountries()
   {

   }

}
