/*
 * Copyright 2022-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.instancio.junit;

import org.instancio.documentation.ExperimentalApi;
import org.instancio.junit.internal.InstancioSourceArgumentsProvider;
import org.instancio.settings.Keys;
import org.junit.jupiter.params.provider.ArgumentsSource;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Provides arguments for {@code @ParameterizedTest} methods, supporting
 * multiple arguments per test run. Each argument will be a fully-populated
 * instance generated by Instancio.
 *
 * <p>Example usage:
 * <pre><code>
 * &#064;ExtendWith(InstancioExtension.class)
 * class ExampleTest {
 *     &#064;ParameterizedTest
 *     &#064;InstancioSource(samples = 3)
 *     void someTestMethod(String s, int n, Person person) {
 *         // ... use generated arguments
 *     }
 * }
 * </code></pre>
 *
 * <p>The above example will run the test method 3 times, each time with
 * a different set of arguments: a random string, a random integer, and
 * a randomly populated instance of the {@code Person} class.
 *
 * @since 1.1.8
 */
@Target({ElementType.ANNOTATION_TYPE, ElementType.METHOD})
@Retention(RetentionPolicy.RUNTIME)
@Documented
@ArgumentsSource(InstancioSourceArgumentsProvider.class)
public @interface InstancioSource {

    /**
     * Specifies the number of times to run the test, generating a new
     * set of arguments each time.
     *
     * <p>The default value is {@code 0}, which means the number will be
     * determined from the {@link Keys#INSTANCIO_SOURCE_SAMPLES} setting.
     *
     * @return the number of samples to generate
     * @since 5.0.0
     */
    @ExperimentalApi
    int samples() default 0;
}