/*
 * Decompiled with CFR 0.152.
 */
package org.itsallcode.openfasttrace.report.plaintext;

import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.itsallcode.openfasttrace.api.ReportSettings;
import org.itsallcode.openfasttrace.api.core.ItemStatus;
import org.itsallcode.openfasttrace.api.core.LinkStatus;
import org.itsallcode.openfasttrace.api.core.LinkedSpecificationItem;
import org.itsallcode.openfasttrace.api.core.Location;
import org.itsallcode.openfasttrace.api.core.Newline;
import org.itsallcode.openfasttrace.api.core.Trace;
import org.itsallcode.openfasttrace.api.core.TracedLink;
import org.itsallcode.openfasttrace.api.report.Reportable;
import org.itsallcode.openfasttrace.report.plaintext.TextFormatter;
import org.itsallcode.openfasttrace.report.plaintext.TextFormatterFactory;

public class PlainTextReport
implements Reportable {
    private final Trace trace;
    private static final Comparator<LinkedSpecificationItem> LINKED_ITEM_BY_ID = Comparator.comparing(LinkedSpecificationItem::getId);
    private int nonEmptySections = 0;
    private final ReportSettings settings;
    private final TextFormatter formatter;

    public PlainTextReport(Trace trace, ReportSettings settings) {
        this.trace = trace;
        this.settings = settings;
        this.formatter = TextFormatterFactory.createFormatter(settings.getColorScheme());
    }

    public void renderToStream(OutputStream outputStream) {
        Charset charset = StandardCharsets.UTF_8;
        try (PrintStream report = new PrintStream(outputStream, false, charset);){
            this.renderToPrintStream(report);
        }
    }

    private void renderToPrintStream(PrintStream report) {
        switch (this.settings.getReportVerbosity()) {
            case QUIET: {
                break;
            }
            case MINIMAL: {
                this.renderResultStatus(report);
                break;
            }
            case SUMMARY: {
                this.renderSummary(report);
                break;
            }
            case FAILURES: {
                this.renderFailureIds(report);
                break;
            }
            case FAILURE_SUMMARIES: {
                this.renderFailureSummaries(report);
                this.separateItemsFromSummary(report);
                this.renderSummary(report);
                break;
            }
            case FAILURE_DETAILS: {
                this.renderFailureDetails(report, this.settings.showOrigin());
                this.separateItemsFromSummary(report);
                this.renderSummary(report);
                break;
            }
            case ALL: {
                this.renderAll(report, this.settings.showOrigin());
                report.print(this.settings.getNewline());
                this.renderSummary(report);
                break;
            }
            default: {
                throw new IllegalStateException("Unable to create stream for unknown verbosity level " + this.settings.getReportVerbosity());
            }
        }
    }

    private void separateItemsFromSummary(PrintStream report) {
        if (this.trace.countDefects() > 0) {
            report.print(this.settings.getNewline());
        }
    }

    private void renderResultStatus(PrintStream report) {
        report.print(this.translateStatus(this.trace.hasNoDefects()));
        report.print(this.settings.getNewline().toString());
    }

    private String translateStatus(boolean ok) {
        return ok ? this.formatter.formatOk("ok") : this.formatter.formatNotOk("not ok");
    }

    private void renderSummary(PrintStream report) {
        report.print(this.translateStatus(this.trace.hasNoDefects()));
        report.print(" - ");
        report.print(this.trace.count());
        report.print(" total");
        if (this.trace.countDefects() != 0) {
            report.print(", ");
            report.print(this.trace.countDefects());
            report.print(" defect");
        }
        report.print(this.settings.getNewline());
    }

    private void renderFailureIds(PrintStream report) {
        this.trace.getDefectIds().stream().sorted().forEachOrdered(id -> {
            report.print(id);
            report.print(this.settings.getNewline().toString());
        });
    }

    private void renderFailureSummaries(PrintStream report) {
        this.trace.getDefectItems().stream().sorted(LINKED_ITEM_BY_ID).forEachOrdered(item -> this.renderItemSummary(report, (LinkedSpecificationItem)item));
    }

    private void renderItemSummary(PrintStream report, LinkedSpecificationItem item) {
        report.print(this.translateStatus(!item.isDefect()));
        this.renderItemLinkCounts(report, item);
        report.print(this.formatter.formatStrong(item.getId().toString()));
        report.print(" ");
        this.renderMaturity(report, item);
        report.print(this.translateArtifactTypeCoverage(item));
        this.renderDuplicatesCount(report, item);
        report.print(this.settings.getNewline());
    }

    private String translateArtifactTypeCoverage(LinkedSpecificationItem item) {
        Comparator<String> byTypeName = Comparator.comparing(a -> a.replaceFirst("[-+]", ""));
        Stream<String> uncoveredStream = item.getUncoveredArtifactTypes().stream().map(x -> this.formatter.formatNotOk("-" + x));
        return "(" + Stream.concat(Stream.concat(uncoveredStream, item.getCoveredArtifactTypes().stream().map(this.formatter::formatOk)), item.getOverCoveredArtifactTypes().stream().map(x -> this.formatter.formatNotOk("+" + x))).sorted(byTypeName).collect(Collectors.joining(", ")) + ")";
    }

    private void renderItemLinkCounts(PrintStream report, LinkedSpecificationItem item) {
        int incomingLinks = item.countIncomingLinks();
        int incomingBadLinks = item.countIncomingBadLinks();
        int incomingGoodLinks = incomingLinks - incomingBadLinks;
        int outgoingLinks = item.countOutgoingLinks();
        int outgoingBadLinks = item.countOutgoingBadLinks();
        int outgoingGoodLinks = outgoingLinks - outgoingBadLinks;
        report.print(" [ in: ");
        report.print(this.formatCountXofY(incomingGoodLinks, incomingLinks));
        report.print(" | out: ");
        report.print(this.formatCountXofY(outgoingGoodLinks, outgoingLinks));
        report.print(" ] ");
    }

    private void renderDuplicatesCount(PrintStream report, LinkedSpecificationItem item) {
        int duplicateLinks = item.countDuplicateLinks();
        if (duplicateLinks != 0) {
            report.print(" [has ");
            report.print(this.formatter.formatNotOk(duplicateLinks + " duplicate" + (duplicateLinks > 1 ? "s" : "")));
            report.print("]");
        }
    }

    private String formatCountXofY(int countGood, int count) {
        if (countGood == 0 && count == 0) {
            return " 0 /  0  ";
        }
        return countGood == count ? this.formatter.formatOk(String.format("%2d / %2d \u2714", countGood, count)) : this.formatter.formatNotOk(String.format("%2d / %2d \u2718", countGood, count));
    }

    private void renderMaturity(PrintStream report, LinkedSpecificationItem item) {
        ItemStatus status = item.getStatus();
        if (status != ItemStatus.APPROVED) {
            report.print("[");
            report.print(status);
            report.print("] ");
        }
    }

    private void renderFailureDetails(PrintStream report, boolean showOrigin) {
        this.trace.getDefectItems().stream().sorted(LINKED_ITEM_BY_ID).forEachOrdered(item -> this.renderItemDetails(report, (LinkedSpecificationItem)item, showOrigin));
    }

    private void renderAll(PrintStream report, boolean showOrigin) {
        this.trace.getItems().stream().sorted(LINKED_ITEM_BY_ID).forEachOrdered(item -> this.renderItemDetails(report, (LinkedSpecificationItem)item, showOrigin));
    }

    private void renderItemDetails(PrintStream report, LinkedSpecificationItem item, boolean showOrigin) {
        this.renderItemSummary(report, item);
        this.renderDescription(report, item);
        if (showOrigin) {
            this.renderOrigin(report, item);
        }
        this.renderLinks(report, item, showOrigin);
        this.renderTags(report, item);
        this.renderItemDetailsEnd(report);
    }

    private void renderOrigin(PrintStream report, Location location) {
        report.print("(");
        report.print(location.getPath());
        report.print(":");
        report.print(location.getLine());
        report.print(")");
    }

    private void renderEmptyItemDetailsLine(PrintStream report) {
        report.print(this.settings.getNewline());
    }

    private void renderDescription(PrintStream report, LinkedSpecificationItem item) {
        String description = item.getDescription();
        if (description != null && !description.isEmpty()) {
            this.renderEmptyItemDetailsLine(report);
            for (String line : description.split(Newline.anyNewlineReqEx())) {
                report.print("  ");
                report.print(line);
                report.print(this.settings.getNewline());
            }
            ++this.nonEmptySections;
        }
    }

    private void renderLinks(PrintStream report, LinkedSpecificationItem item, boolean showOrigin) {
        if (item.hasLinks()) {
            this.renderEmptyItemDetailsLine(report);
            this.renderOrderedLinks(report, item, showOrigin);
            ++this.nonEmptySections;
        }
    }

    private void renderOrderedLinks(PrintStream report, LinkedSpecificationItem item, boolean showOrigin) {
        item.getTracedLinks().stream().sorted(Comparator.comparing(a -> a.getOtherLinkEnd().getId())).forEachOrdered(link -> this.renderLink(report, (TracedLink)link, showOrigin));
    }

    private void renderLink(PrintStream report, TracedLink link, boolean showOrigin) {
        Location location;
        LinkStatus status = link.getStatus();
        report.print("  [");
        if (status == LinkStatus.COVERS || status == LinkStatus.COVERED_SHALLOW) {
            report.print(this.formatter.formatOk(PlainTextReport.padStatus(status)));
        } else {
            report.print(this.formatter.formatNotOk(PlainTextReport.padStatus(status)));
        }
        report.print("] ");
        report.print(status.isIncoming() ? "\u2190 " : "\u2192 ");
        report.print(link.getOtherLinkEnd().getId());
        report.print(this.settings.getNewline());
        if (showOrigin && (location = link.getOtherLinkEnd().getLocation()) != null) {
            report.print("         ");
            this.renderOrigin(report, location);
            report.print(this.settings.getNewline());
        }
    }

    private static String padStatus(LinkStatus status) {
        return String.format("%-17s", status.toString().toLowerCase());
    }

    private void renderTags(PrintStream report, LinkedSpecificationItem item) {
        List tags = item.getTags();
        if (tags != null && !tags.equals(Collections.emptyList())) {
            this.renderEmptyItemDetailsLine(report);
            report.print("  #: ");
            report.print(String.join((CharSequence)", ", tags));
            report.print(this.settings.getNewline());
            ++this.nonEmptySections;
        }
    }

    private void renderOrigin(PrintStream report, LinkedSpecificationItem item) {
        Location location = item.getLocation();
        if (location != null) {
            this.renderEmptyItemDetailsLine(report);
            report.print("  (");
            report.print(location.getPath());
            report.print(":");
            report.print(location.getLine());
            report.print(")");
            report.print(this.settings.getNewline());
        }
    }

    private void renderItemDetailsEnd(PrintStream report) {
        if (this.nonEmptySections > 0) {
            this.renderEmptyItemDetailsLine(report);
        }
    }
}

