package org.iworkz.genesis.vertx.maven;

import java.io.File;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Scanner;
import java.util.concurrent.Callable;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.stream.Collectors;

import org.eclipse.aether.artifact.Artifact;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Promise;
import io.vertx.core.Verticle;
import io.vertx.core.Vertx;
import io.vertx.core.impl.logging.Logger;
import io.vertx.core.impl.logging.LoggerFactory;
import io.vertx.core.json.DecodeException;
import io.vertx.core.json.JsonObject;
import io.vertx.core.spi.VerticleFactory;
import io.vertx.maven.MavenCoords;
import io.vertx.maven.MavenVerticleFactory;
import io.vertx.maven.Resolver;
import io.vertx.maven.ResolverOptions;
import io.vertx.maven.resolver.ResolutionOptions;

/**
 * based on the implementation of io.vertx.maven.MavenVerticleFactory
 */
public class GenesisMavenVerticleFactory extends MavenVerticleFactory {

    private static final Logger log = LoggerFactory.getLogger(GenesisMavenVerticleFactory.class);

    private static final String PREFIX = "genesis-maven";

    private Map<byte[], MavenServiceOptions> serviceOptionsMap = new HashMap<>();
    private Vertx vertx;
    
    static {
        String remoteRepositoriesEnv = System.getenv("MAVEN_REMOTE_REPOS");
        if (remoteRepositoriesEnv != null && !remoteRepositoriesEnv.isEmpty()) {
            System.setProperty(REMOTE_REPOS_SYS_PROP, remoteRepositoriesEnv);
        }
    }

    public GenesisMavenVerticleFactory() {
        super();
    }

    public GenesisMavenVerticleFactory(ResolverOptions options) {
        this(Resolver.create(options));
    }

    public GenesisMavenVerticleFactory(Resolver resolver) {
        super(resolver);
    }

    @Override
    public void init(Vertx vertx) {
        super.init(vertx);
        this.vertx = vertx;
        log.info("Init " + getClass().getCanonicalName());
    }

    @Override
    public String prefix() {
        return PREFIX;
    }

    @Override
    public void createVerticle(String verticleName, DeploymentOptions deploymentOptions, ClassLoader classLoader,
                               Promise<Callable<Verticle>> promise) {

        log.info("Create verticle " + verticleName);
        vertx.<Callable<Verticle>>executeBlocking(executionPromise -> {
            try {
                String identifierNoPrefix = VerticleFactory.removePrefix(verticleName);
                String coordsString = identifierNoPrefix;
                String serviceName = null;
                int pos = identifierNoPrefix.lastIndexOf("::");
                if (pos != -1) {
                    coordsString = identifierNoPrefix.substring(0, pos);
                    serviceName = identifierNoPrefix.substring(pos + 2);
                }

                MavenServiceOptions serviceOptions =
                        createServiceOptions(coordsString, identifierNoPrefix, serviceName, classLoader);

                DeploymentOptions effectiveDeploymentOptions =
                        createDeplomentOptionsWithClassLoader(deploymentOptions, serviceOptions);

                AbstractVerticle wrapperVerticle = createWrapperVerticle(serviceOptions.getMainVerticleName(),
                                                                         serviceOptions, effectiveDeploymentOptions);
                executionPromise.complete(() -> wrapperVerticle);
            } catch (Exception e) {
                log.error(e);
                Throwable cause = findRootMavenServiceFactoryException(e);
                executionPromise.fail(cause);
            }
        }, false, ar -> {
            if (ar.succeeded()) {
                log.info("Successfully deployed " + prefix() + " verticle" + verticleName);
                promise.complete(ar.result());
            } else {
                log.error("Failed to deploy " + prefix() + " verticle" + verticleName, ar.cause());
                promise.fail(ar.cause());
            }
        });
    }

    private DeploymentOptions createDeplomentOptionsWithClassLoader(DeploymentOptions deploymentOptions,
                                                                    MavenServiceOptions serviceOptions) {
        DeploymentOptions effectiveDeploymentOptions = null;
        if (serviceOptions.getDeploymentOptions() != null) {
            JsonObject mergedDeploymentOptions = deploymentOptions.toJson();
            mergedDeploymentOptions.mergeIn(serviceOptions.getDeploymentOptions());
            effectiveDeploymentOptions = new DeploymentOptions(mergedDeploymentOptions);
        } else {
            effectiveDeploymentOptions = deploymentOptions;
        }
        effectiveDeploymentOptions.setClassLoader(serviceOptions.getUrlClassLoader());
        return effectiveDeploymentOptions;
    }

    protected Throwable findRootMavenServiceFactoryException(Throwable ex) {
        if (ex.getCause() instanceof MavenServiceFactoryException) {
            return findRootMavenServiceFactoryException(ex.getCause());
        } else {
            return ex;
        }
    }

    protected JsonObject getServiceDescriptor(ClassLoader classLoader, String serviceName) {
        String descriptorFile = serviceName + ".json";
        try (InputStream is = classLoader.getResourceAsStream(descriptorFile)) {
            if (is == null) {
                throw new MavenServiceFactoryException("Cannot find service descriptor file " + descriptorFile
                        + " on classpath");
            }
            return readDescriptor(is, descriptorFile);
        } catch (Exception e) {
            throw new MavenServiceFactoryException("Failed to read " + descriptorFile, e);
        }
    }

    protected JsonObject readDescriptor(InputStream is, String descriptorFileName) {
        try (Scanner scanner = new Scanner(is, "UTF-8").useDelimiter("\\A")) {
            String conf = scanner.next();
            return new JsonObject(conf);
        } catch (NoSuchElementException e) {
            throw new IllegalArgumentException(descriptorFileName + " is empty");
        } catch (DecodeException e) {
            throw new IllegalArgumentException(descriptorFileName + " contains invalid json");
        }
    }

    protected MavenServiceOptions createServiceOptions(String coordsString, String identifierNoPrefix,
                                                       String serviceName, ClassLoader classLoader) {
        MavenCoords coords = new MavenCoords(coordsString);
        if (coords.version() == null) {
            throw new IllegalArgumentException("Invalid service identifier, missing version: " + coordsString);
        }
        List<Artifact> artifacts = resolveArtifacts(coordsString);
        MavenServiceOptions serviceOptions = null;
        for (Artifact result : artifacts) {
            if (result.getGroupId().equals(coords.owner()) && result.getArtifactId().equals(coords.serviceName())) {
                File file = result.getFile();
                byte[] checksum = checksumOf(file);
                synchronized (serviceOptionsMap) {
                    serviceOptions = serviceOptionsMap.computeIfAbsent(checksum, c -> {
                        URLClassLoader urlClassLoader = createClassLoader(artifacts, classLoader);
                        log.info("Register maven service: " + coordsString);
                        return createServiceOptions(urlClassLoader, file, checksum, identifierNoPrefix, serviceName);
                    });
                    serviceOptions.incrementUsage();
                }
            }
        }
        return serviceOptions;
    }

    protected MavenServiceOptions createServiceOptions(URLClassLoader urlClassLoader, File file, byte[] checksum,
                                                       String identifierNoPrefix, String serviceName) {

        String mainVerticleClassName;
        JsonObject deploymentOptions = null;

        /* when service name is null we look at the Main-Verticle in META-INF/MANIFEST.MF */
        if (serviceName == null) {
            mainVerticleClassName = getMainVerticleNameFromManifest(file, identifierNoPrefix);
        } else {
            /* when service name is defined look at the service descriptor */
            JsonObject descriptor = getServiceDescriptor(urlClassLoader, serviceName);
            if (descriptor == null) {
                throw new MavenServiceFactoryException("Cannot find service descriptor file " + serviceName + ".json");
            }
            mainVerticleClassName = descriptor.getString("main");
            if (mainVerticleClassName == null) {
                throw new MavenServiceFactoryException(serviceName + ".json does not contain a main field");
            }
            deploymentOptions = descriptor.getJsonObject("options", new JsonObject());
        }
        log.info("Create service options: " + mainVerticleClassName);
        return new MavenServiceOptions(checksum, mainVerticleClassName, urlClassLoader, deploymentOptions);
    }

    protected List<Artifact> resolveArtifacts(String coordsString) {
        try {
            return getResolver().resolve(coordsString, new ResolutionOptions());
        } catch (NullPointerException e) {
            // Sucks, but aether throws a NPE if repository name is invalid....
            throw new MavenServiceFactoryException("Cannot find module " + coordsString
                    + ". Maybe repository URL is invalid?");
        }
    }

    protected String getMainVerticleNameFromManifest(File file, String identifierNoPrefix) {

        try (JarFile jarFile = new JarFile(file)) {
            String serviceIdentifer = null;
            Manifest manifest = jarFile.getManifest();
            if (manifest != null) {
                serviceIdentifer = (String) manifest.getMainAttributes().get(new Attributes.Name("Main-Verticle"));
                if (serviceIdentifer == null) {
                    throw new IllegalStateException("Failed to get name of 'Main-Verticle' in manifest "
                            + identifierNoPrefix);
                }

            } else {
                throw new IllegalStateException("Invalid service identifier, missing 'Main-Verticle' attribute: "
                        + identifierNoPrefix);
            }
            return serviceIdentifer;
        } catch (Exception ex) {
            throw new MavenServiceFactoryException("Invalid service identifier", ex);
        }

    }

    protected URLClassLoader createClassLoader(List<Artifact> artifacts, ClassLoader parentClassLoader) {
        // Generate the classpath - if the jar is already on the Vert.x classpath (e.g. the Vert.x dependencies, netty
        // etc)
        // then we don't add it to the classpath for the module
        List<String> classpath =
                artifacts.stream().map(res -> res.getFile().getAbsolutePath()).collect(Collectors.toList());
        URL[] urls = new URL[classpath.size()];
        int index = 0;
        List<String> extraCP = new ArrayList<>(urls.length);
        for (String pathElement : classpath) {
            File claspathFile = new File(pathElement);
            extraCP.add(claspathFile.getAbsolutePath());
            try {
                URL url = claspathFile.toURI().toURL();
                urls[index++] = url;
            } catch (MalformedURLException e) {
                throw new IllegalStateException(e);
            }
        }
        return new URLClassLoader(urls, parentClassLoader);
    }

    protected byte[] checksumOf(File file) {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            computeDigest(file, md);
            return md.digest();
        } catch (Exception e) {
            throw new MavenServiceFactoryException("Failed to get checksum of file: " + file.getAbsolutePath(), e);
        }
    }

    protected void computeDigest(File file, MessageDigest md) {
        byte[] buffer = new byte[1024];
        try (InputStream is = Files.newInputStream(file.toPath());
             DigestInputStream dis = new DigestInputStream(is, md)) {
            while (is.read(buffer) > 0)
                ;
        } catch (Exception e) {
            throw new MavenServiceFactoryException("Failed to compute digest of file: " + file.getAbsolutePath(), e);
        }
    }

    protected AbstractVerticle createWrapperVerticle(String fullQualifiedVerticleName,
                                                     MavenServiceOptions serviceOptions,
                                                     DeploymentOptions deploymentOptions) {

        return new AbstractVerticle() {

            @Override
            public void start(Promise<Void> startPromise) {
                if (deploymentOptions.getConfig() == null) {
                    deploymentOptions.setConfig(new JsonObject());
                }
                deploymentOptions.getConfig().mergeIn(context.config());
                vertx.deployVerticle(fullQualifiedVerticleName, deploymentOptions)
                     .onSuccess(deploymentId -> log.info("Successfully deployed verticle: "
                             + serviceOptions.getMainVerticleName()))
                     .<Void>mapEmpty()
                     .onComplete(startPromise);
            }

            @Override
            public void stop(Promise<Void> stopPromise) {
                removeDeployment();
                stopPromise.complete();
            }

            protected void removeDeployment() {
                log.info("Successfully undeployed verticle: " + serviceOptions.getMainVerticleName());
                synchronized (serviceOptionsMap) {
                    if (serviceOptions.decrementAndGetUsage() == 0) {
                        serviceOptionsMap.remove(serviceOptions.getChecksum());
                        try {
                            serviceOptions.getUrlClassLoader().close();
                        } catch (Exception e) {
                            log.error("Failed to close classloader for verticle: "
                                    + serviceOptions.getMainVerticleName());
                            e.printStackTrace();
                        }
                        log.info("Unregister verticle: " + serviceOptions.getMainVerticleName());
                    }
                }
            }
        };
    }

}
