/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.auth;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import software.amazon.awssdk.SdkClientException;
import software.amazon.awssdk.annotation.SdkTestInternalApi;
import software.amazon.awssdk.auth.AbstractAwsSigner;
import software.amazon.awssdk.auth.AwsCredentials;
import software.amazon.awssdk.auth.AwsSessionCredentials;
import software.amazon.awssdk.auth.Presigner;
import software.amazon.awssdk.auth.RegionAwareSigner;
import software.amazon.awssdk.auth.SdkClock;
import software.amazon.awssdk.auth.ServiceAwareSigner;
import software.amazon.awssdk.auth.SigningAlgorithm;
import software.amazon.awssdk.auth.internal.Aws4SignerRequestParams;
import software.amazon.awssdk.auth.internal.Aws4SignerUtils;
import software.amazon.awssdk.auth.internal.SignerKey;
import software.amazon.awssdk.http.SdkHttpFullRequest;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.internal.collections.FifoCache;
import software.amazon.awssdk.util.CredentialUtils;
import software.amazon.awssdk.util.DateUtils;
import software.amazon.awssdk.util.SdkHttpUtils;
import software.amazon.awssdk.util.StringUtils;
import software.amazon.awssdk.utils.BinaryUtils;

public class Aws4Signer
extends AbstractAwsSigner
implements ServiceAwareSigner,
RegionAwareSigner,
Presigner {
    private static final Log LOG = LogFactory.getLog(Aws4Signer.class);
    private static final int SIGNER_CACHE_MAX_SIZE = 300;
    private static final FifoCache<SignerKey> SIGNER_CACHE = new FifoCache(300);
    private static final List<String> LIST_OF_HEADERS_TO_IGNORE_IN_LOWER_CASE = Arrays.asList("connection", "x-amzn-trace-id");
    protected String serviceName;
    protected String regionName;
    private Date overriddenDate;
    private boolean doubleUrlEncode;
    private final SdkClock clock;

    public Aws4Signer() {
        this(true);
    }

    public Aws4Signer(boolean doubleUrlEncoding) {
        this(doubleUrlEncoding, SdkClock.STANDARD);
    }

    @SdkTestInternalApi
    public Aws4Signer(SdkClock clock) {
        this(true, clock);
    }

    private Aws4Signer(boolean doubleUrlEncode, SdkClock clock) {
        this.doubleUrlEncode = doubleUrlEncode;
        this.clock = clock;
    }

    @SdkTestInternalApi
    public void setOverrideDate(Date overriddenDate) {
        this.overriddenDate = overriddenDate != null ? new Date(overriddenDate.getTime()) : null;
    }

    public String getRegionName() {
        return this.regionName;
    }

    @Override
    public void setRegionName(String regionName) {
        this.regionName = regionName;
    }

    public String getServiceName() {
        return this.serviceName;
    }

    @Override
    public void setServiceName(String serviceName) {
        this.serviceName = serviceName;
    }

    @Override
    public SdkHttpFullRequest sign(SdkHttpFullRequest request, AwsCredentials credentials) {
        if (CredentialUtils.isAnonymous(credentials)) {
            return request;
        }
        return (SdkHttpFullRequest)((SdkHttpFullRequest.Builder)((SdkHttpFullRequest.Builder)request.toBuilder()).apply(b -> this.doSign((SdkHttpFullRequest.Builder)b, credentials))).build();
    }

    private SdkHttpFullRequest.Builder doSign(SdkHttpFullRequest.Builder mutableRequest, AwsCredentials credentials) {
        AwsCredentials sanitizedCredentials = this.sanitizeCredentials(credentials);
        if (sanitizedCredentials instanceof AwsSessionCredentials) {
            this.addSessionCredentials(mutableRequest, (AwsSessionCredentials)sanitizedCredentials);
        }
        Aws4SignerRequestParams signerParams = new Aws4SignerRequestParams(mutableRequest, this.overriddenDate, this.regionName, this.serviceName, "AWS4-HMAC-SHA256");
        this.addHostHeader(mutableRequest);
        mutableRequest.header("X-Amz-Date", signerParams.getFormattedSigningDateTime());
        String contentSha256 = this.calculateContentHash(mutableRequest);
        mutableRequest.getFirstHeaderValue("x-amz-content-sha256").filter(h -> h.equals("required")).ifPresent(h -> mutableRequest.header("x-amz-content-sha256", contentSha256));
        String canonicalRequest = this.createCanonicalRequest(mutableRequest, contentSha256);
        String stringToSign = this.createStringToSign(canonicalRequest, signerParams);
        byte[] signingKey = this.deriveSigningKey(sanitizedCredentials, signerParams);
        byte[] signature = this.computeSignature(stringToSign, signingKey);
        mutableRequest.header("Authorization", this.buildAuthorizationHeader(mutableRequest, signature, sanitizedCredentials, signerParams));
        this.processRequestPayload(mutableRequest, signature, signingKey, signerParams);
        return mutableRequest;
    }

    @Override
    public SdkHttpFullRequest presignRequest(SdkHttpFullRequest request, AwsCredentials credentials, Date userSpecifiedExpirationDate) {
        if (CredentialUtils.isAnonymous(credentials)) {
            return request;
        }
        SdkHttpFullRequest.Builder mutableRequest = (SdkHttpFullRequest.Builder)request.toBuilder();
        long expirationInSeconds = this.generateExpirationDate(userSpecifiedExpirationDate);
        this.addHostHeader(mutableRequest);
        AwsCredentials sanitizedCredentials = this.sanitizeCredentials(credentials);
        if (sanitizedCredentials instanceof AwsSessionCredentials) {
            mutableRequest.queryParameter("X-Amz-Security-Token", ((AwsSessionCredentials)sanitizedCredentials).sessionToken());
        }
        Aws4SignerRequestParams signerRequestParams = new Aws4SignerRequestParams(mutableRequest, this.overriddenDate, this.regionName, this.serviceName, "AWS4-HMAC-SHA256");
        String timeStamp = signerRequestParams.getFormattedSigningDateTime();
        this.addPreSignInformationToRequest(mutableRequest, sanitizedCredentials, signerRequestParams, timeStamp, expirationInSeconds);
        String contentSha256 = this.calculateContentHashPresign(mutableRequest);
        String canonicalRequest = this.createCanonicalRequest(mutableRequest, contentSha256);
        String stringToSign = this.createStringToSign(canonicalRequest, signerRequestParams);
        byte[] signingKey = this.deriveSigningKey(sanitizedCredentials, signerRequestParams);
        byte[] signature = this.computeSignature(stringToSign, signingKey);
        mutableRequest.queryParameter("X-Amz-Signature", BinaryUtils.toHex(signature));
        return (SdkHttpFullRequest)mutableRequest.build();
    }

    private String createCanonicalRequest(SdkHttpRequest request, String contentSha256) {
        String path = SdkHttpUtils.appendUri(request.getEndpoint().getPath(), request.getResourcePath());
        String canonicalRequest = request.getHttpMethod().toString() + "\n" + this.getCanonicalizedResourcePath(path, this.doubleUrlEncode) + "\n" + this.getCanonicalizedQueryString(request.getParameters()) + "\n" + this.getCanonicalizedHeaderString(request) + "\n" + this.getSignedHeadersString(request) + "\n" + contentSha256;
        if (LOG.isDebugEnabled()) {
            LOG.debug("AWS4 Canonical Request: '\"" + canonicalRequest + "\"");
        }
        return canonicalRequest;
    }

    private String createStringToSign(String canonicalRequest, Aws4SignerRequestParams signerParams) {
        String stringToSign = signerParams.getSigningAlgorithm() + "\n" + signerParams.getFormattedSigningDateTime() + "\n" + signerParams.getScope() + "\n" + BinaryUtils.toHex(this.hash(canonicalRequest));
        if (LOG.isDebugEnabled()) {
            LOG.debug("AWS4 String to Sign: '\"" + stringToSign + "\"");
        }
        return stringToSign;
    }

    private byte[] deriveSigningKey(AwsCredentials credentials, Aws4SignerRequestParams signerRequestParams) {
        String cacheKey = this.computeSigningCacheKeyName(credentials, signerRequestParams);
        long daysSinceEpochSigningDate = DateUtils.numberOfDaysSinceEpoch(signerRequestParams.getSigningDateTimeMilli());
        SignerKey signerKey = SIGNER_CACHE.get(cacheKey);
        if (signerKey != null && daysSinceEpochSigningDate == signerKey.getNumberOfDaysSinceEpoch()) {
            return signerKey.getSigningKey();
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Generating a new signing key as the signing key not available in the cache for the date " + TimeUnit.DAYS.toMillis(daysSinceEpochSigningDate));
        }
        byte[] signingKey = this.newSigningKey(credentials, signerRequestParams.getFormattedSigningDate(), signerRequestParams.getRegionName(), signerRequestParams.getServiceName());
        SIGNER_CACHE.add(cacheKey, new SignerKey(daysSinceEpochSigningDate, signingKey));
        return signingKey;
    }

    private String computeSigningCacheKeyName(AwsCredentials credentials, Aws4SignerRequestParams signerRequestParams) {
        return credentials.secretAccessKey() + "-" + signerRequestParams.getRegionName() + "-" + signerRequestParams.getServiceName();
    }

    private byte[] computeSignature(String stringToSign, byte[] signingKey) {
        return this.sign(stringToSign.getBytes(Charset.forName("UTF-8")), signingKey, SigningAlgorithm.HmacSHA256);
    }

    private String buildAuthorizationHeader(SdkHttpRequest request, byte[] signature, AwsCredentials credentials, Aws4SignerRequestParams signerParams) {
        String signingCredentials = credentials.accessKeyId() + "/" + signerParams.getScope();
        String credential = "Credential=" + signingCredentials;
        String signerHeaders = "SignedHeaders=" + this.getSignedHeadersString(request);
        String signatureHeader = "Signature=" + BinaryUtils.toHex(signature);
        return "AWS4-HMAC-SHA256" + " " + credential + ", " + signerHeaders + ", " + signatureHeader;
    }

    private void addPreSignInformationToRequest(SdkHttpFullRequest.Builder mutableRequest, AwsCredentials credentials, Aws4SignerRequestParams signerParams, String timeStamp, long expirationInSeconds) {
        String signingCredentials = credentials.accessKeyId() + "/" + signerParams.getScope();
        mutableRequest.queryParameter("X-Amz-Algorithm", "AWS4-HMAC-SHA256");
        mutableRequest.queryParameter("X-Amz-Date", timeStamp);
        mutableRequest.queryParameter("X-Amz-SignedHeaders", this.getSignedHeadersString(mutableRequest));
        mutableRequest.queryParameter("X-Amz-Expires", Long.toString(expirationInSeconds));
        mutableRequest.queryParameter("X-Amz-Credential", signingCredentials);
    }

    @Override
    protected void addSessionCredentials(SdkHttpFullRequest.Builder mutableRequest, AwsSessionCredentials credentials) {
        mutableRequest.header("X-Amz-Security-Token", credentials.sessionToken());
    }

    private String getCanonicalizedHeaderString(SdkHttpRequest request) {
        ArrayList<String> sortedHeaders = new ArrayList<String>(request.getHeaders().keySet());
        Collections.sort(sortedHeaders, String.CASE_INSENSITIVE_ORDER);
        Map<String, List<String>> requestHeaders = request.getHeaders();
        StringBuilder buffer = new StringBuilder();
        for (String header : sortedHeaders) {
            if (this.shouldExcludeHeaderFromSigning(header)) continue;
            String key = StringUtils.lowerCase(header);
            for (String headerValue : requestHeaders.get(header)) {
                StringUtils.appendCompactedString(buffer, key);
                buffer.append(":");
                if (headerValue != null) {
                    StringUtils.appendCompactedString(buffer, headerValue);
                }
                buffer.append("\n");
            }
        }
        return buffer.toString();
    }

    private String getSignedHeadersString(SdkHttpRequest request) {
        ArrayList<String> sortedHeaders = new ArrayList<String>(request.getHeaders().keySet());
        Collections.sort(sortedHeaders, String.CASE_INSENSITIVE_ORDER);
        StringBuilder buffer = new StringBuilder();
        for (String header : sortedHeaders) {
            if (this.shouldExcludeHeaderFromSigning(header)) continue;
            if (buffer.length() > 0) {
                buffer.append(";");
            }
            buffer.append(StringUtils.lowerCase(header));
        }
        return buffer.toString();
    }

    private boolean shouldExcludeHeaderFromSigning(String header) {
        return LIST_OF_HEADERS_TO_IGNORE_IN_LOWER_CASE.contains(StringUtils.lowerCase(header));
    }

    private void addHostHeader(SdkHttpFullRequest.Builder mutableRequest) {
        URI endpoint = mutableRequest.getEndpoint();
        StringBuilder hostHeaderBuilder = new StringBuilder(endpoint.getHost());
        if (SdkHttpUtils.isUsingNonDefaultPort(endpoint)) {
            hostHeaderBuilder.append(":").append(endpoint.getPort());
        }
        mutableRequest.header("Host", hostHeaderBuilder.toString());
    }

    protected String calculateContentHash(SdkHttpFullRequest.Builder request) {
        InputStream payloadStream = this.getBinaryRequestPayloadStream(request.getContent());
        payloadStream.mark(Aws4Signer.getReadLimit(request));
        String contentSha256 = BinaryUtils.toHex(this.hash(payloadStream));
        try {
            payloadStream.reset();
        }
        catch (IOException e) {
            throw new SdkClientException("Unable to reset stream after calculating AWS4 signature", e);
        }
        return contentSha256;
    }

    protected void processRequestPayload(SdkHttpFullRequest.Builder request, byte[] signature, byte[] signingKey, Aws4SignerRequestParams signerRequestParams) {
    }

    protected String calculateContentHashPresign(SdkHttpFullRequest.Builder request) {
        return this.calculateContentHash(request);
    }

    private long generateExpirationDate(Date expirationDate) {
        long expirationInSeconds;
        long l = expirationInSeconds = expirationDate != null ? (expirationDate.getTime() - this.clock.currentTimeMillis()) / 1000L : 604800L;
        if (expirationInSeconds > 604800L) {
            throw new SdkClientException("Requests that are pre-signed by SigV4 algorithm are valid for at most 7 days. The expiration date set on the current request [" + Aws4SignerUtils.formatTimestamp(expirationDate.getTime()) + "] has exceeded this limit.");
        }
        return expirationInSeconds;
    }

    private byte[] newSigningKey(AwsCredentials credentials, String dateStamp, String regionName, String serviceName) {
        byte[] kSecret = ("AWS4" + credentials.secretAccessKey()).getBytes(Charset.forName("UTF-8"));
        byte[] kDate = this.sign(dateStamp, kSecret, SigningAlgorithm.HmacSHA256);
        byte[] kRegion = this.sign(regionName, kDate, SigningAlgorithm.HmacSHA256);
        byte[] kService = this.sign(serviceName, kRegion, SigningAlgorithm.HmacSHA256);
        return this.sign("aws4_request", kService, SigningAlgorithm.HmacSHA256);
    }
}

