/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.auth;

import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import java.io.IOException;
import java.net.URISyntaxException;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import software.amazon.awssdk.SdkClientException;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.auth.AwsCredentials;
import software.amazon.awssdk.auth.AwsCredentialsProvider;
import software.amazon.awssdk.auth.AwsSessionCredentials;
import software.amazon.awssdk.internal.CredentialsEndpointProvider;
import software.amazon.awssdk.internal.EC2CredentialsUtils;
import software.amazon.awssdk.util.ComparableUtils;
import software.amazon.awssdk.util.DateUtils;
import software.amazon.awssdk.util.json.Jackson;
import software.amazon.awssdk.utils.Validate;
import software.amazon.awssdk.utils.cache.CachedSupplier;
import software.amazon.awssdk.utils.cache.NonBlocking;
import software.amazon.awssdk.utils.cache.RefreshResult;

@SdkInternalApi
class EC2CredentialsProvider
implements AwsCredentialsProvider,
AutoCloseable {
    private final CredentialsEndpointProvider credentialsEndpointProvider;
    private final CachedSupplier<AwsCredentials> credentialsCache;

    public EC2CredentialsProvider(CredentialsEndpointProvider credentialsEndpointProvider, boolean asyncRefreshEnabled, String asyncThreadName) {
        this.credentialsEndpointProvider = credentialsEndpointProvider;
        CachedSupplier.Builder cacheBuilder = CachedSupplier.builder(this::refreshCredentials);
        if (asyncRefreshEnabled) {
            cacheBuilder.prefetchStrategy(new NonBlocking(asyncThreadName));
        }
        this.credentialsCache = cacheBuilder.build();
    }

    private RefreshResult<AwsCredentials> refreshCredentials() {
        try {
            String credentialsResponse = EC2CredentialsUtils.getInstance().readResource(this.credentialsEndpointProvider.getCredentialsEndpoint(), this.credentialsEndpointProvider.getRetryPolicy());
            JsonNode node = Jackson.jsonNodeOf(credentialsResponse);
            JsonNode accessKey = node.get("AccessKeyId");
            JsonNode secretKey = node.get("SecretAccessKey");
            JsonNode token = node.get("Token");
            JsonNode expirationNode = node.get("Expiration");
            Validate.notNull(accessKey, "Failed to load access key.", new Object[0]);
            Validate.notNull(secretKey, "Failed to load secret key.", new Object[0]);
            AwsCredentials credentials = token == null ? new AwsCredentials(accessKey.asText(), secretKey.asText()) : new AwsSessionCredentials(accessKey.asText(), secretKey.asText(), token.asText());
            Instant expiration = this.getExpiration(expirationNode).orElse(null);
            if (expiration != null && Instant.now().isAfter(expiration)) {
                throw new SdkClientException("Credentials obtained from EC2 metadata service are already expired");
            }
            return RefreshResult.builder(credentials).staleTime(this.getStaleTime(expiration)).prefetchTime(this.getPrefetchTime(expiration)).build();
        }
        catch (SdkClientException e) {
            throw e;
        }
        catch (JsonMappingException e) {
            throw new SdkClientException("Unable to parse response returned from service endpoint.", e);
        }
        catch (IOException | RuntimeException | URISyntaxException e) {
            throw new SdkClientException("Unable to load credentials from service endpoint.", e);
        }
    }

    private Optional<Instant> getExpiration(JsonNode expirationNode) {
        return Optional.ofNullable(expirationNode).map(node -> {
            String expirationValue = node.asText().replaceAll("\\+0000$", "Z");
            try {
                return DateUtils.parseIso8601Date(expirationValue).toInstant();
            }
            catch (RuntimeException e) {
                throw new IllegalStateException("Unable to parse credentials expiration date from Amazon EC2 instance.", e);
            }
        });
    }

    private Instant getStaleTime(Instant expiration) {
        return expiration == null ? null : expiration.minus(Duration.ofMinutes(1L));
    }

    private Instant getPrefetchTime(Instant expiration) {
        Instant oneHourFromNow = Instant.now().plus(Duration.ofHours(1L));
        return expiration == null ? oneHourFromNow : (Instant)ComparableUtils.minimum((Comparable[])new Instant[]{oneHourFromNow, expiration.minus(Duration.ofMinutes(15L))});
    }

    @Override
    public AwsCredentials getCredentials() {
        return this.credentialsCache.get();
    }

    @Override
    public void close() {
        this.credentialsCache.close();
    }

    public String toString() {
        return this.getClass().getSimpleName();
    }
}

