/**
 * JBoss, Home of Professional Open Source
 * Copyright Red Hat, Inc., and individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.aerogear.android.store.sql;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.os.AsyncTask;
import android.util.Log;
import com.google.gson.GsonBuilder;
import org.jboss.aerogear.android.core.Callback;
import org.jboss.aerogear.android.core.ReadFilter;
import org.jboss.aerogear.android.core.reflection.Property;
import org.jboss.aerogear.android.core.reflection.Scan;
import org.jboss.aerogear.android.security.EncryptionService;
import org.jboss.aerogear.android.security.InvalidKeyException;
import org.jboss.aerogear.android.security.SecurityManager;
import org.jboss.aerogear.android.security.keystore.KeyStoreBasedEncryptionConfiguration;
import org.jboss.aerogear.android.store.Store;
import org.jboss.aerogear.android.store.generator.IdGenerator;
import org.jboss.aerogear.android.store.util.CryptoEntityUtil;
import org.jboss.aerogear.crypto.RandomUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class EncryptedSQLStore<T> extends SQLiteOpenHelper implements Store<T> {

    private static final String TAG = EncryptedSQLStore.class.getSimpleName();

    private final String COLUMN_ID = "ID";
    private final String COLUMN_DATA = "DATA";
    private final String ID_IV = "IV";

    private final Class<T> modelClass;
    private Context context;
    private final GsonBuilder builder;
    private final IdGenerator idGenerator;
    private final String password;
    private final String TABLE_NAME;

    private SQLiteDatabase database;
    private CryptoEntityUtil<T> cryptoEntityUtil;

    public EncryptedSQLStore(Class<T> modelClass, Context context, GsonBuilder builder,
                             IdGenerator idGenerator, String password) {
        this(modelClass, context, builder, idGenerator, password, modelClass.getSimpleName());
    }

    public EncryptedSQLStore(Class<T> modelClass, Context context, GsonBuilder builder,
                             IdGenerator idGenerator, String password, String tableName) {

        super(context, modelClass.getSimpleName(), null, 2);

        this.modelClass = modelClass;
        this.context = context;
        this.builder = builder;
        this.idGenerator = idGenerator;
        this.password = password;

        this.TABLE_NAME = tableName;
    }

    private String getEncryptTableHelperName() {
        return TABLE_NAME.toUpperCase() + "_ENCRYPT_HELPER";
    }

    @Override
    public void onCreate(SQLiteDatabase sqLiteDatabase) {

        // -- Table for helper data

        String SQL_CREATE_ENCRYPT_HELPER_TABLE = "CREATE TABLE IF NOT EXISTS " + getEncryptTableHelperName() +
                " ( " +
                COLUMN_ID + " TEXT NOT NULL, " +
                COLUMN_DATA + " BLOB NOT NULL " +
                " ) ";
        sqLiteDatabase.execSQL(SQL_CREATE_ENCRYPT_HELPER_TABLE);

        // -- Store iv

        byte[] iv = RandomUtils.randomBytes();

        String SQL_STORE_DATA = "INSERT INTO " + getEncryptTableHelperName() +
                " ( " + COLUMN_ID + ", " + COLUMN_DATA + " ) " +
                " VALUES ( ?, ? ) ";

        sqLiteDatabase.execSQL(SQL_STORE_DATA, new Object[]{ID_IV, iv});

        // -- Table for encrypted data

        String SQL_CREATE_ENTITY_TABLE = "CREATE TABLE IF NOT EXISTS " + TABLE_NAME +
                " ( " +
                COLUMN_ID + " TEXT NOT NULL, " +
                COLUMN_DATA + " BLOB NOT NULL " +
                " ) ";
        sqLiteDatabase.execSQL(SQL_CREATE_ENTITY_TABLE);

    }

    @Override
    public void onUpgrade(SQLiteDatabase sqLiteDatabase, int oldVersion, int newVersion) {
    }

    @Override
    public void onOpen(SQLiteDatabase db) {

        super.onOpen(db);

        String SQL = "SELECT " + COLUMN_DATA + " FROM " + getEncryptTableHelperName() + " WHERE " + COLUMN_ID + " = ?";

        Cursor cursorIV = db.rawQuery(SQL, new String[]{ID_IV});
        cursorIV.moveToFirst();

        try {

            byte[] iv = cursorIV.getBlob(0);

            EncryptionService encryptionService = SecurityManager
                    .config(TABLE_NAME, KeyStoreBasedEncryptionConfiguration.class)
                    .setContext(context)
                    .setAlias(TABLE_NAME)
                    .setKeyStoreFile(TABLE_NAME)
                    .setPassword(password)
                    .asService();

            cryptoEntityUtil = new CryptoEntityUtil<T>(encryptionService, iv, modelClass, builder);

        } finally {
            cursorIV.close();
        }

    }

    /**
     * {@inheritDoc}
     *
     * @throws InvalidKeyException   Will occur if you use the wrong password to retrieve the data
     */
    @Override
    public Collection<T> readAll() throws InvalidKeyException {
        ensureOpen();

        ArrayList<T> dataList = new ArrayList<T>();

        String sql = "SELECT " + COLUMN_DATA + " FROM " + TABLE_NAME;
        Cursor cursor = getReadableDatabase().rawQuery(sql, new String[0]);
        try {
            while (cursor.moveToNext()) {
                byte[] encryptedData = cursor.getBlob(0);
                T decryptedData = cryptoEntityUtil.decrypt(encryptedData);
                dataList.add(decryptedData);
            }
        } finally {
            cursor.close();
        }

        return dataList;
    }

    /**
     * {@inheritDoc}
     *
     * @throws InvalidKeyException   Will occur if you use the wrong password to retrieve the data
     */
    @Override
    public T read(Serializable id) throws InvalidKeyException {
        ensureOpen();

        String sql = "SELECT " + COLUMN_DATA + " FROM " + TABLE_NAME + " WHERE " + COLUMN_ID + " = ?";
        Cursor cursor = getReadableDatabase().rawQuery(sql, new String[]{id.toString()});
        cursor.moveToFirst();

        if (cursor.getCount() == 0) {
            return null;
        }

        try {
            byte[] encryptedData = cursor.getBlob(0);
            return cryptoEntityUtil.decrypt(encryptedData);
        } finally {
            cursor.close();
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<T> readWithFilter(ReadFilter filter) {
        throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     *
     */
    @Override
    public void save(T item)  {
        ensureOpen();

        this.database.beginTransaction();
        try {
            saveItem(item);
            this.database.setTransactionSuccessful();
        } finally {
            this.database.endTransaction();
        }
    }

    /**
     * {@inheritDoc}
     *
     */
    @Override
    public void save(Collection<T> items) {
        ensureOpen();

        this.database.beginTransaction();
        try {
            for (T item : items) {
                saveItem(item);
            }
            this.database.setTransactionSuccessful();
        } finally {
            this.database.endTransaction();
        }
    }

    private void saveItem(T item) {
        String recordIdFieldName = Scan.recordIdFieldNameIn(item.getClass());
        Property property = new Property(item.getClass(), recordIdFieldName);
        Serializable idValue = (Serializable) property.getValue(item);

        if (idValue == null) {
            idValue = idGenerator.generate();
            property.setValue(item, idValue);
        }

        ContentValues values = new ContentValues();
        values.put(COLUMN_ID, idValue.toString());
        values.put(COLUMN_DATA, cryptoEntityUtil.encrypt(item));

        this.database.insert(TABLE_NAME, null, values);
    }

    /**
     * {@inheritDoc}
     *
     */
    @Override
    public void reset() {
        ensureOpen();

        String sql = String.format("DELETE FROM " + TABLE_NAME);
        this.database.execSQL(sql);
    }

    /**
     * {@inheritDoc}
     *
     */
    @Override
    public void remove(Serializable id) {
        ensureOpen();

        String sql = "DELETE FROM " + TABLE_NAME + " WHERE " + COLUMN_ID + " = ?";
        this.database.execSQL(sql, new Object[]{id});
    }

    /**
     * {@inheritDoc}
     *
     */
    @Override
    public boolean isEmpty() {
        ensureOpen();

        String sql = "SELECT COUNT(" + COLUMN_ID + ") FROM " + TABLE_NAME;
        Cursor cursor = getReadableDatabase().rawQuery(sql, null);
        cursor.moveToFirst();
        boolean result = (cursor.getInt(0) == 0);
        cursor.close();
        return result;
    }

    public void open(final Callback<EncryptedSQLStore<T>> onReady) {
        new AsyncTask<Void, Void, Void>() {
            private Exception exception;

            @Override
            protected Void doInBackground(Void... params) {
                try {
                    EncryptedSQLStore.this.database = getWritableDatabase();
                } catch (Exception e) {
                    this.exception = e;
                    Log.e(TAG, "There was an error loading the database", e);
                }
                return null;
            }

            @Override
            protected void onPostExecute(Void result) {
                if (exception != null) {
                    onReady.onFailure(exception);
                } else {
                    onReady.onSuccess(EncryptedSQLStore.this);
                }
            }
        }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, (Void[]) null);
    }

    public void openSync() {
        this.database = getWritableDatabase();
    }

    @Override
    public void close() {
        this.database.close();
    }

    private boolean isOpen() {
        return this.database != null;
    }

    private void ensureOpen() {
        if (!isOpen()) {
            Log.w(TAG, "Store is not opened, trying to open.");
            openSync();
        }
    }

}
