/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.arquillian.container.openshift.express.util;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.CharArrayWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.channels.Channel;

/**
 * General IO stream manipulation utilities.
 *
 * This class provides static utility methods for input/output operations.
 *
 * closeQuietly - these methods close a stream ignoring nulls and exceptions toXxx/read - these methods read data from a stream
 * write - these methods write data to a stream copy - these methods copy all the data from one stream to another contentEquals
 * - these methods compare the content of two streams
 *
 *
 * The byte-to-char methods and char-to-byte methods involve a conversion step. Two methods are provided in each case, one that
 * uses the platform default encoding and the other which allows you to specify an encoding. You are encouraged to always
 * specify an encoding because relying on the platform default can lead to unexpected results, for example when moving from
 * development to production.
 *
 * All the methods in this class that read a stream are buffered internally. This means that there is no cause to use a
 * BufferedInputStream or BufferedReader. The default buffer size of 4K has been shown to be efficient in tests.
 *
 * Wherever possible, the methods in this class do not flush or close the stream. This is to avoid making non-portable
 * assumptions about the streams' origin and further use. Thus the caller is still responsible for closing streams after use.
 *
 * Origin of code: Excalibur.
 *
 * @author <a href="mailto:kpiwko@redhat.com>Karel Piwko</a>
 * @author Peter Donald
 * @author Jeff Turner
 * @author Matthew Hawthorne
 * @author Stephen Colebourne
 * @author Gareth Davis
 * @author Ian Springer
 * @author Niall Pemberton
 * @author Sandy McArthur
 * @since Apache Tika 0.4, copied (partially) from Commons IO 1.4
 */
public class IOUtils {

    /**
     * The default buffer size to use.
     */
    private static final int DEFAULT_BUFFER_SIZE = 1024 * 4;

    private IOUtils() {
    }

    // -----------------------------------------------------------------------
    /**
     * Unconditionally close an Reader.
     *
     * Equivalent to {@link Reader#close()}, except any exceptions will be ignored. This is typically used in finally blocks.
     *
     * @param input the Reader to close, may be null or already closed
     */
    public static void closeQuietly(Reader input) {
        try {
            if (input != null) {
                input.close();
            }
        } catch (IOException ioe) {
            // ignore
        }
    }

    /**
     * Unconditionally close a Channel.
     *
     * Equivalent to {@link Channel#close()}, except any exceptions will be ignored. This is typically used in finally blocks.
     *
     * @param channel the Channel to close, may be null or already closed
     */
    public static void closeQuietly(Channel channel) {
        try {
            if (channel != null) {
                channel.close();
            }
        } catch (IOException ioe) {
            // ignore
        }
    }

    /**
     * Unconditionally close a Writer.
     *
     * Equivalent to {@link Writer#close()}, except any exceptions will be ignored. This is typically used in finally blocks.
     *
     * @param output the Writer to close, may be null or already closed
     */
    public static void closeQuietly(Writer output) {
        try {
            if (output != null) {
                output.close();
            }
        } catch (IOException ioe) {
            // ignore
        }
    }

    /**
     * Unconditionally close an InputStream.
     *
     * Equivalent to {@link InputStream#close()}, except any exceptions will be ignored. This is typically used in finally
     * blocks.
     *
     * @param input the InputStream to close, may be null or already closed
     */
    public static void closeQuietly(InputStream input) {
        try {
            if (input != null) {
                input.close();
            }
        } catch (IOException ioe) {
            // ignore
        }
    }

    /**
     * Unconditionally close an OutputStream.
     *
     * Equivalent to {@link OutputStream#close()}, except any exceptions will be ignored. This is typically used in finally
     * blocks.
     *
     * @param output the OutputStream to close, may be null or already closed
     */
    public static void closeQuietly(OutputStream output) {
        try {
            if (output != null) {
                output.close();
            }
        } catch (IOException ioe) {
            // ignore
        }
    }

    // read toByteArray
    // -----------------------------------------------------------------------
    /**
     * Get the contents of an InputStream as a byte[].
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * @param input the InputStream to read from
     * @return the requested byte array
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     */
    public static byte[] toByteArray(InputStream input) throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        copy(input, output);
        return output.toByteArray();
    }

    /**
     * Get the contents of a Reader as a byte[] using the default character encoding of the platform.
     *
     * This method buffers the input internally, so there is no need to use a BufferedReader.
     *
     * @param input the Reader to read from
     * @return the requested byte array
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     */
    public static byte[] toByteArray(Reader input) throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        copy(input, output);
        return output.toByteArray();
    }

    /**
     * Get the contents of a Reader as a byte[] using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * This method buffers the input internally, so there is no need to use a BufferedReader.
     *
     * @param input the Reader to read from
     * @param encoding the encoding to use, null means platform default
     * @return the requested byte array
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static byte[] toByteArray(Reader input, String encoding) throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        copy(input, output, encoding);
        return output.toByteArray();
    }

    /**
     * Get the contents of a String as a byte[] using the default character encoding of the platform.
     *
     * This is the same as {@link String#getBytes()}.
     *
     * @param input the String to convert
     * @return the requested byte array
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs (never occurs)
     * @deprecated Use {@link String#getBytes()}
     */
    @Deprecated
    public static byte[] toByteArray(String input) throws IOException {
        return input.getBytes();
    }

    // read char[]
    // -----------------------------------------------------------------------
    /**
     * Get the contents of an InputStream as a character array using the default character encoding of the platform.
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * @param is the InputStream to read from
     * @return the requested character array
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static char[] toCharArray(InputStream is) throws IOException {
        CharArrayWriter output = new CharArrayWriter();
        copy(is, output);
        return output.toCharArray();
    }

    /**
     * Get the contents of an InputStream as a character array using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * @param is the InputStream to read from
     * @param encoding the encoding to use, null means platform default
     * @return the requested character array
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static char[] toCharArray(InputStream is, String encoding) throws IOException {
        CharArrayWriter output = new CharArrayWriter();
        copy(is, output, encoding);
        return output.toCharArray();
    }

    /**
     * Get the contents of a Reader as a character array.
     *
     * This method buffers the input internally, so there is no need to use a BufferedReader.
     *
     * @param input the Reader to read from
     * @return the requested character array
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static char[] toCharArray(Reader input) throws IOException {
        CharArrayWriter sw = new CharArrayWriter();
        copy(input, sw);
        return sw.toCharArray();
    }

    // read toString
    // -----------------------------------------------------------------------
    /**
     * Get the contents of an InputStream as a String using the default character encoding of the platform.
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * @param input the InputStream to read from
     * @return the requested String
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     */
    public static String toString(InputStream input) throws IOException {
        StringWriter sw = new StringWriter();
        copy(input, sw);
        return sw.toString();
    }

    /**
     * Get the contents of an InputStream as a String using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * @param input the InputStream to read from
     * @param encoding the encoding to use, null means platform default
     * @return the requested String
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     */
    public static String toString(InputStream input, String encoding) throws IOException {
        StringWriter sw = new StringWriter();
        copy(input, sw, encoding);
        return sw.toString();
    }

    /**
     * Get the contents of a Reader as a String.
     *
     * This method buffers the input internally, so there is no need to use a BufferedReader.
     *
     * @param input the Reader to read from
     * @return the requested String
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs
     */
    public static String toString(Reader input) throws IOException {
        StringWriter sw = new StringWriter();
        copy(input, sw);
        return sw.toString();
    }

    /**
     * Get the contents of a byte[] as a String using the default character encoding of the platform.
     *
     * @param input the byte array to read from
     * @return the requested String
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs (never occurs)
     * @deprecated Use {@link String#String(byte[])}
     */
    @Deprecated
    public static String toString(byte[] input) throws IOException {
        return new String(input);
    }

    /**
     * Get the contents of a byte[] as a String using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * @param input the byte array to read from
     * @param encoding the encoding to use, null means platform default
     * @return the requested String
     * @throws NullPointerException if the input is null
     * @throws IOException if an I/O error occurs (never occurs)
     * @deprecated Use {@link String#String(byte[],String)}
     */
    @Deprecated
    public static String toString(byte[] input, String encoding) throws IOException {
        if (encoding == null) {
            return new String(input);
        } else {
            return new String(input, encoding);
        }
    }

    // -----------------------------------------------------------------------
    /**
     * Convert the specified CharSequence to an input stream, encoded as bytes using the default character encoding of the
     * platform.
     *
     * @param input the CharSequence to convert
     * @return an input stream
     * @since IO 2.0
     */
    public static InputStream toInputStream(CharSequence input) {
        return toInputStream(input.toString());
    }

    /**
     * Convert the specified CharSequence to an input stream, encoded as bytes using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * @param input the CharSequence to convert
     * @param encoding the encoding to use, null means platform default
     * @throws IOException if the encoding is invalid
     * @return an input stream
     * @since IO 2.0
     */
    public static InputStream toInputStream(CharSequence input, String encoding) throws IOException {
        return toInputStream(input.toString(), encoding);
    }

    // -----------------------------------------------------------------------
    /**
     * Convert the specified string to an input stream, encoded as bytes using the default character encoding of the platform.
     *
     * @param input the string to convert
     * @return an input stream
     * @since Commons IO 1.1
     */
    public static InputStream toInputStream(String input) {
        byte[] bytes = input.getBytes();
        return new ByteArrayInputStream(bytes);
    }

    /**
     * Convert the specified string to an input stream, encoded as bytes using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * @param input the string to convert
     * @param encoding the encoding to use, null means platform default
     * @throws IOException if the encoding is invalid
     * @return an input stream
     * @since Commons IO 1.1
     */
    public static InputStream toInputStream(String input, String encoding) throws IOException {
        byte[] bytes = encoding != null ? input.getBytes(encoding) : input.getBytes();
        return new ByteArrayInputStream(bytes);
    }

    // write byte[]
    // -----------------------------------------------------------------------
    /**
     * Writes bytes from a byte[] to an OutputStream.
     *
     * @param data the byte array to write, do not modify during output, null ignored
     * @param output the OutputStream to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(byte[] data, OutputStream output) throws IOException {
        if (data != null) {
            output.write(data);
        }
    }

    /**
     * Writes bytes from a byte[] to chars on a Writer using the default character encoding of the platform.
     *
     * This method uses {@link String#String(byte[])}.
     *
     * @param data the byte array to write, do not modify during output, null ignored
     * @param output the Writer to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(byte[] data, Writer output) throws IOException {
        if (data != null) {
            output.write(new String(data));
        }
    }

    /**
     * Writes bytes from a byte[] to chars on a Writer using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * This method uses {@link String#String(byte[], String)}.
     *
     * @param data the byte array to write, do not modify during output, null ignored
     * @param output the Writer to write to
     * @param encoding the encoding to use, null means platform default
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(byte[] data, Writer output, String encoding) throws IOException {
        if (data != null) {
            if (encoding == null) {
                write(data, output);
            } else {
                output.write(new String(data, encoding));
            }
        }
    }

    // write char[]
    // -----------------------------------------------------------------------
    /**
     * Writes chars from a char[] to a Writer using the default character encoding of the platform.
     *
     * @param data the char array to write, do not modify during output, null ignored
     * @param output the Writer to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(char[] data, Writer output) throws IOException {
        if (data != null) {
            output.write(data);
        }
    }

    /**
     * Writes chars from a char[] to bytes on an OutputStream.
     *
     * This method uses {@link String#String(char[])} and {@link String#getBytes()}.
     *
     * @param data the char array to write, do not modify during output, null ignored
     * @param output the OutputStream to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(char[] data, OutputStream output) throws IOException {
        if (data != null) {
            output.write(new String(data).getBytes());
        }
    }

    /**
     * Writes chars from a char[] to bytes on an OutputStream using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * This method uses {@link String#String(char[])} and {@link String#getBytes(String)}.
     *
     * @param data the char array to write, do not modify during output, null ignored
     * @param output the OutputStream to write to
     * @param encoding the encoding to use, null means platform default
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(char[] data, OutputStream output, String encoding) throws IOException {
        if (data != null) {
            if (encoding == null) {
                write(data, output);
            } else {
                output.write(new String(data).getBytes(encoding));
            }
        }
    }

    // write CharSequence
    // -----------------------------------------------------------------------
    /**
     * Writes chars from a CharSequence to a Writer.
     *
     * @param data the CharSequence to write, null ignored
     * @param output the Writer to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 2.0
     */
    public static void write(CharSequence data, Writer output) throws IOException {
        if (data != null) {
            write(data.toString(), output);
        }
    }

    /**
     * Writes chars from a CharSequence to bytes on an OutputStream using the default character encoding of the platform.
     *
     * This method uses {@link String#getBytes()}.
     *
     * @param data the CharSequence to write, null ignored
     * @param output the OutputStream to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 2.0
     */
    public static void write(CharSequence data, OutputStream output) throws IOException {
        if (data != null) {
            write(data.toString(), output);
        }
    }

    /**
     * Writes chars from a CharSequence to bytes on an OutputStream using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * This method uses {@link String#getBytes(String)}.
     *
     * @param data the CharSequence to write, null ignored
     * @param output the OutputStream to write to
     * @param encoding the encoding to use, null means platform default
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 2.0
     */
    public static void write(CharSequence data, OutputStream output, String encoding) throws IOException {
        if (data != null) {
            write(data.toString(), output, encoding);
        }
    }

    // write String
    // -----------------------------------------------------------------------
    /**
     * Writes chars from a String to a Writer.
     *
     * @param data the String to write, null ignored
     * @param output the Writer to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(String data, Writer output) throws IOException {
        if (data != null) {
            output.write(data);
        }
    }

    /**
     * Writes chars from a String to bytes on an OutputStream using the default character encoding of the platform.
     *
     * This method uses {@link String#getBytes()}.
     *
     * @param data the String to write, null ignored
     * @param output the OutputStream to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(String data, OutputStream output) throws IOException {
        if (data != null) {
            output.write(data.getBytes());
        }
    }

    /**
     * Writes chars from a String to bytes on an OutputStream using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * This method uses {@link String#getBytes(String)}.
     *
     * @param data the String to write, null ignored
     * @param output the OutputStream to write to
     * @param encoding the encoding to use, null means platform default
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void write(String data, OutputStream output, String encoding) throws IOException {
        if (data != null) {
            if (encoding == null) {
                write(data, output);
            } else {
                output.write(data.getBytes(encoding));
            }
        }
    }

    // write StringBuffer
    // -----------------------------------------------------------------------
    /**
     * Writes chars from a StringBuffer to a Writer.
     *
     * @param data the StringBuffer to write, null ignored
     * @param output the Writer to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     * @deprecated replaced by write(CharSequence, Writer)
     */
    @Deprecated
    public static void write(StringBuffer data, Writer output) throws IOException {
        if (data != null) {
            output.write(data.toString());
        }
    }

    /**
     * Writes chars from a StringBuffer to bytes on an OutputStream using the default character encoding of the platform.
     *
     * This method uses {@link String#getBytes()}.
     *
     * @param data the StringBuffer to write, null ignored
     * @param output the OutputStream to write to
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     * @deprecated replaced by write(CharSequence, OutputStream)
     */
    @Deprecated
    public static void write(StringBuffer data, OutputStream output) throws IOException {
        if (data != null) {
            output.write(data.toString().getBytes());
        }
    }

    /**
     * Writes chars from a StringBuffer to bytes on an OutputStream using the specified character encoding.
     *
     * Character encoding names can be found at IANA.
     *
     * This method uses {@link String#getBytes(String)}.
     *
     * @param data the StringBuffer to write, null ignored
     * @param output the OutputStream to write to
     * @param encoding the encoding to use, null means platform default
     * @throws NullPointerException if output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     * @deprecated replaced by write(CharSequence, OutputStream, String)
     */
    @Deprecated
    public static void write(StringBuffer data, OutputStream output, String encoding) throws IOException {
        if (data != null) {
            if (encoding == null) {
                write(data, output);
            } else {
                output.write(data.toString().getBytes(encoding));
            }
        }
    }

    // copy from InputStream
    // -----------------------------------------------------------------------
    /**
     * Copy bytes from an InputStream to an OutputStream.
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * Large streams (over 2GB) will return a bytes copied value of -1 after the copy has completed since the correct number of
     * bytes cannot be returned as an int. For large streams use the copyLarge(InputStream, OutputStream) method.
     *
     * @param input the InputStream to read from
     * @param output the OutputStream to write to
     * @return the number of bytes copied
     * @throws NullPointerException if the input or output is null
     * @throws IOException if an I/O error occurs
     * @throws ArithmeticException if the byte count is too large
     * @since Commons IO 1.1
     */
    public static int copy(InputStream input, OutputStream output) throws IOException {
        long count = copyLarge(input, output);
        if (count > Integer.MAX_VALUE) {
            return -1;
        }
        return (int) count;
    }

    /**
     * Copy bytes from a large (over 2GB) InputStream to an OutputStream.
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * @param input the InputStream to read from
     * @param output the OutputStream to write to
     * @return the number of bytes copied
     * @throws NullPointerException if the input or output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.3
     */
    public static long copyLarge(InputStream input, OutputStream output) throws IOException {
        byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
        long count = 0;
        int n = 0;
        while (-1 != (n = input.read(buffer))) {
            output.write(buffer, 0, n);
            count += n;
        }
        return count;
    }

    /**
     * Copy bytes from an InputStream to chars on a Writer using the default character encoding of the platform.
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * This method uses {@link InputStreamReader}.
     *
     * @param input the InputStream to read from
     * @param output the Writer to write to
     * @throws NullPointerException if the input or output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void copy(InputStream input, Writer output) throws IOException {
        InputStreamReader in = new InputStreamReader(input);
        copy(in, output);
    }

    /**
     * Copy bytes from an InputStream to chars on a Writer using the specified character encoding.
     *
     * This method buffers the input internally, so there is no need to use a BufferedInputStream.
     *
     * Character encoding names can be found at IANA.
     *
     * This method uses {@link InputStreamReader}.
     *
     * @param input the InputStream to read from
     * @param output the Writer to write to
     * @param encoding the encoding to use, null means platform default
     * @throws NullPointerException if the input or output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void copy(InputStream input, Writer output, String encoding) throws IOException {
        if (encoding == null) {
            copy(input, output);
        } else {
            InputStreamReader in = new InputStreamReader(input, encoding);
            copy(in, output);
        }
    }

    // copy from Reader
    // -----------------------------------------------------------------------
    /**
     * Copy chars from a Reader to a Writer.
     *
     * This method buffers the input internally, so there is no need to use a BufferedReader.
     *
     * Large streams (over 2GB) will return a chars copied value of -1 after the copy has completed since the correct number of
     * chars cannot be returned as an int. For large streams use the copyLarge(Reader, Writer) method.
     *
     * @param input the Reader to read from
     * @param output the Writer to write to
     * @return the number of characters copied
     * @throws NullPointerException if the input or output is null
     * @throws IOException if an I/O error occurs
     * @throws ArithmeticException if the character count is too large
     * @since Commons IO 1.1
     */
    public static int copy(Reader input, Writer output) throws IOException {
        long count = copyLarge(input, output);
        if (count > Integer.MAX_VALUE) {
            return -1;
        }
        return (int) count;
    }

    /**
     * Copy chars from a large (over 2GB) Reader to a Writer.
     *
     * This method buffers the input internally, so there is no need to use a BufferedReader.
     *
     * @param input the Reader to read from
     * @param output the Writer to write to
     * @return the number of characters copied
     * @throws NullPointerException if the input or output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.3
     */
    public static long copyLarge(Reader input, Writer output) throws IOException {
        char[] buffer = new char[DEFAULT_BUFFER_SIZE];
        long count = 0;
        int n = 0;
        while (-1 != (n = input.read(buffer))) {
            output.write(buffer, 0, n);
            count += n;
        }
        return count;
    }

    /**
     * Copy chars from a Reader to bytes on an OutputStream using the default character encoding of the platform, and calling
     * flush.
     *
     * This method buffers the input internally, so there is no need to use a BufferedReader.
     *
     * Due to the implementation of OutputStreamWriter, this method performs a flush.
     *
     * This method uses {@link OutputStreamWriter}.
     *
     * @param input the Reader to read from
     * @param output the OutputStream to write to
     * @throws NullPointerException if the input or output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void copy(Reader input, OutputStream output) throws IOException {
        OutputStreamWriter out = new OutputStreamWriter(output);
        copy(input, out);
        // XXX Unless anyone is planning on rewriting OutputStreamWriter, we
        // have to flush here.
        out.flush();
    }

    /**
     * Copy chars from a Reader to bytes on an OutputStream using the specified character encoding, and calling flush.
     *
     * This method buffers the input internally, so there is no need to use a BufferedReader.
     *
     * Character encoding names can be found at IANA.
     *
     * Due to the implementation of OutputStreamWriter, this method performs a flush.
     *
     * This method uses {@link OutputStreamWriter}.
     *
     * @param input the Reader to read from
     * @param output the OutputStream to write to
     * @param encoding the encoding to use, null means platform default
     * @throws NullPointerException if the input or output is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static void copy(Reader input, OutputStream output, String encoding) throws IOException {
        if (encoding == null) {
            copy(input, output);
        } else {
            OutputStreamWriter out = new OutputStreamWriter(output, encoding);
            copy(input, out);
            // XXX Unless anyone is planning on rewriting OutputStreamWriter,
            // we have to flush here.
            out.flush();
        }
    }

    // content equals
    // -----------------------------------------------------------------------
    /**
     * Compare the contents of two Streams to determine if they are equal or not.
     *
     * This method buffers the input internally using BufferedInputStream if they are not already buffered.
     *
     * @param input1 the first stream
     * @param input2 the second stream
     * @return true if the content of the streams are equal or they both don't exist, false otherwise
     * @throws NullPointerException if either input is null
     * @throws IOException if an I/O error occurs
     */
    public static boolean contentEquals(InputStream input1, InputStream input2) throws IOException {
        if (!(input1 instanceof BufferedInputStream)) {
            input1 = new BufferedInputStream(input1);
        }
        if (!(input2 instanceof BufferedInputStream)) {
            input2 = new BufferedInputStream(input2);
        }

        int ch = input1.read();
        while (-1 != ch) {
            int ch2 = input2.read();
            if (ch != ch2) {
                return false;
            }
            ch = input1.read();
        }

        int ch2 = input2.read();
        return (ch2 == -1);
    }

    /**
     * Compare the contents of two Readers to determine if they are equal or not.
     *
     * This method buffers the input internally using BufferedReader if they are not already buffered.
     *
     * @param input1 the first reader
     * @param input2 the second reader
     * @return true if the content of the readers are equal or they both don't exist, false otherwise
     * @throws NullPointerException if either input is null
     * @throws IOException if an I/O error occurs
     * @since Commons IO 1.1
     */
    public static boolean contentEquals(Reader input1, Reader input2) throws IOException {
        if (!(input1 instanceof BufferedReader)) {
            input1 = new BufferedReader(input1);
        }
        if (!(input2 instanceof BufferedReader)) {
            input2 = new BufferedReader(input2);
        }

        int ch = input1.read();
        while (-1 != ch) {
            int ch2 = input2.read();
            if (ch != ch2) {
                return false;
            }
            ch = input1.read();
        }

        int ch2 = input2.read();
        return (ch2 == -1);
    }

}