/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.connector.adapters.jdbc;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import org.jboss.logging.Logger;

/**
 * A wrapper for a statement.
 *
 * @author <a href="mailto:d_jencks@users.sourceforge.net">David Jencks</a>
 * @author <a href="mailto:adrian@jboss.com">Adrian Brock</a>
 * @author <a href="mailto:jesper.pedersen@jboss.org">Jesper Pedersen</a>
 * @version $Revision: 74335 $
 */
public abstract class WrappedStatement extends JBossWrapper implements Statement, StatementAccess {
    /**
     * The spy logger
     */
    protected static Logger spyLogger = Logger.getLogger(Constants.SPY_LOGGER_CATEGORY);
    private final WrappedConnection lc;
    private final Statement s;

    /**
     * The result sets
     */
    private HashMap<WrappedResultSet, Throwable> resultSets;

    /**
     * Whether we are closed
     */
    private AtomicBoolean closed = new AtomicBoolean(false);

    /**
     * Spy functionality
     */
    protected final boolean spy;

    /**
     * The jndi name
     */
    protected final String jndiName;

    /**
     * Constructor
     *
     * @param lc       The connection
     * @param s        The statement
     * @param spy      The spy value
     * @param jndiName The jndi name
     */
    public WrappedStatement(final WrappedConnection lc, Statement s, final boolean spy, final String jndiName) {
        this.lc = lc;
        this.s = s;
        this.spy = spy;
        this.jndiName = jndiName;

        lc.registerStatement(this);
    }

    /**
     * Lock connection
     *
     * @throws SQLException Thrown if an error occurs
     */
    protected void lock() throws SQLException {
        lc.lock();
    }

    /**
     * Unlock connection
     */
    protected void unlock() {
        lc.unlock();
    }

    /**
     * {@inheritDoc}
     */
    public void close() throws SQLException {
        if (closed.get())
            return;

        if (spy)
            spyLogger.debugf("%s [%s] close()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

        closed.set(true);
        lc.unregisterStatement(this);
        internalClose();
    }

    /**
     * {@inheritDoc}
     */
    public boolean execute(String sql) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] execute(%s)", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, sql);

                return s.execute(sql);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] execute(%s, %s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, sql, autoGeneratedKeys);

                return s.execute(sql, autoGeneratedKeys);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] execute(%s, %s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT,
                            sql, Arrays.toString(columnIndexes));

                return s.execute(sql, columnIndexes);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] execute(%s, %s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT,
                            sql, Arrays.toString(columnNames));

                return s.execute(sql, columnNames);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public Connection getConnection() throws SQLException {
        if (spy)
            spyLogger.debugf("%s [%s] getConnection()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

        return lc;
    }

    /**
     * {@inheritDoc}
     */
    public SQLWarning getWarnings() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getWarnings()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getWarnings();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void clearWarnings() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] clearWarnings()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                s.clearWarnings();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public ResultSet executeQuery(String sql) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] executeQuery(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, sql);

                ResultSet result = s.executeQuery(sql);
                return registerResultSet(result);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int executeUpdate(String sql) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] executeUpdate(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, sql);

                return s.executeUpdate(sql);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] executeUpdate(%s, %s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, sql, autoGeneratedKeys);

                return s.executeUpdate(sql, autoGeneratedKeys);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] executeUpdate(%s, %s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT,
                            sql, Arrays.toString(columnIndexes));

                return s.executeUpdate(sql, columnIndexes);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        lock();
        try {
            checkTransaction();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] executeUpdate(%s, %s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT,
                            sql, Arrays.toString(columnNames));

                return s.executeUpdate(sql, columnNames);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getMaxFieldSize() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getMaxFieldSize()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getMaxFieldSize();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setMaxFieldSize(int max) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] setMaxFieldSize(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, max);

                s.setMaxFieldSize(max);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getMaxRows() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getMaxRows()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getMaxRows();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setMaxRows(int max) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] setMaxRows(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, max);

                s.setMaxRows(max);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setEscapeProcessing(boolean enable) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] setEscapeProcessing(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, enable);

                s.setEscapeProcessing(enable);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getQueryTimeout() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getQueryTimeout()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getQueryTimeout();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setQueryTimeout(int timeout) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] setQueryTimeout(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, timeout);

                s.setQueryTimeout(timeout);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void cancel() throws SQLException {
        checkState();
        try {
            if (spy)
                spyLogger.debugf("%s [%s] cancel()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

            s.cancel();
        } catch (Throwable t) {
            throw checkException(t);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setCursorName(String name) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] setCursorName(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, name);

                s.setCursorName(name);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public ResultSet getResultSet() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getResultSet()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                ResultSet result = s.getResultSet();
                if (result == null)
                    return null;
                else
                    return registerResultSet(result);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getUpdateCount() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getUpdateCount()", jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getUpdateCount();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean getMoreResults() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getMoreResults()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getMoreResults();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean getMoreResults(int current) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getMoreResults(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, current);

                return s.getMoreResults(current);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setFetchDirection(int direction) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] setFetchDirection(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, direction);

                s.setFetchDirection(direction);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getFetchDirection() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getFetchDirection()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getFetchDirection();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setFetchSize(int rows) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] setFetchSize(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, rows);

                s.setFetchSize(rows);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getFetchSize() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getFetchSize()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getFetchSize();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getResultSetConcurrency() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getResultSetConcurrency()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getResultSetConcurrency();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getResultSetType() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getResultSetType()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getResultSetType();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void addBatch(String sql) throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] addBatch(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT, sql);

                s.addBatch(sql);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void clearBatch() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] clearBatch()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                s.clearBatch();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int[] executeBatch() throws SQLException {
        lock();
        try {
            checkState();
            try {
                checkConfiguredQueryTimeout();

                if (spy)
                    spyLogger.debugf("%s [%s] executeBatch()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.executeBatch();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public ResultSet getGeneratedKeys() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getGeneratedKeys()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                ResultSet resultSet = s.getGeneratedKeys();
                return registerResultSet(resultSet);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public int getResultSetHoldability() throws SQLException {
        lock();
        try {
            checkState();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] getResultSetHoldability()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return s.getResultSetHoldability();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean isClosed() throws SQLException {
        lock();
        try {
            if (spy)
                spyLogger.debugf("%s [%s] isClosed()",
                        jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

            Statement wrapped = getWrappedObject();
            if (wrapped == null)
                return true;
            return wrapped.isClosed();
        } catch (Throwable t) {
            throw checkException(t);
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean isPoolable() throws SQLException {
        lock();
        try {
            Statement statement = getUnderlyingStatement();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] isPoolable()",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT);

                return statement.isPoolable();
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setPoolable(boolean poolable) throws SQLException {
        lock();
        try {
            Statement statement = getUnderlyingStatement();
            try {
                if (spy)
                    spyLogger.debugf("%s [%s] setPoolable(%s)",
                            jndiName, Constants.SPY_LOGGER_PREFIX_STATEMENT,
                            poolable);

                statement.setPoolable(poolable);
            } catch (Throwable t) {
                throw checkException(t);
            }
        } finally {
            unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    public Statement getUnderlyingStatement() throws SQLException {
        lock();
        try {
            checkState();
            return s;
        } finally {
            unlock();
        }
    }

    /**
     * Get the wrapped statement
     *
     * @return The result
     * @throws SQLException Thrown if an error occurs
     */
    protected Statement getWrappedObject() throws SQLException {
        return getUnderlyingStatement();
    }

    /**
     * Check exception
     *
     * @param t The throwable
     * @return The result
     * @throws SQLException Thrown if an error occurs
     */
    protected SQLException checkException(Throwable t)
            throws SQLException {
        throw lc.checkException(t);
    }

    /**
     * Check transaction
     *
     * @throws SQLException Thrown if an error occurs
     */
    protected void checkTransaction()
            throws SQLException {
        checkState();
        lc.checkTransaction();
    }

    /**
     * Check configured query timeout
     *
     * @throws SQLException Thrown if an error occurs
     */
    protected void checkConfiguredQueryTimeout() throws SQLException {
        lc.checkConfiguredQueryTimeout(this);
    }

    /**
     * Check if transaction is active
     *
     * @throws SQLException Thrown if an error occurs
     */
    protected void checkTransactionActive() throws SQLException {
        lc.checkTransactionActive();
    }

    /**
     * Close
     *
     * @throws SQLException Thrown if an error occurs
     */
    protected void internalClose() throws SQLException {
        closed.set(true);
        try {
            closeResultSets();
        } finally {
            s.close();
        }
    }

    /**
     * Check state
     *
     * @throws SQLException Thrown if an error occurs
     */
    void checkState() throws SQLException {
        if (closed.get())
            throw new SQLException("The statement is closed.");
    }

    /**
     * Wrap the result set
     *
     * @param resultSet The result set
     * @param spy       The spy value
     * @param jndiName  The jndi name
     * @return The result
     */
    protected abstract WrappedResultSet wrapResultSet(ResultSet resultSet, boolean spy, String jndiName);

    /**
     * Register a result set
     *
     * @param resultSet The result set
     * @return The result
     */
    protected ResultSet registerResultSet(ResultSet resultSet) {
        if (resultSet != null)
            resultSet = wrapResultSet(resultSet, spy, jndiName);

        if (lc.getTrackStatements() == BaseWrapperManagedConnectionFactory.TRACK_STATEMENTS_FALSE_INT)
            return resultSet;

        WrappedResultSet wrapped = (WrappedResultSet) resultSet;

        synchronized (this) {
            if (resultSets == null)
                resultSets = new HashMap<WrappedResultSet, Throwable>(1);

            if (lc.getTrackStatements() == BaseWrapperManagedConnectionFactory.TRACK_STATEMENTS_TRUE_INT)
                resultSets.put(wrapped, new Throwable("STACKTRACE"));
            else
                resultSets.put(wrapped, null);
        }
        return resultSet;
    }

    /**
     * Unregister a result set
     *
     * @param resultSet The result set
     */
    protected void unregisterResultSet(WrappedResultSet resultSet) {
        if (lc.getTrackStatements() == BaseWrapperManagedConnectionFactory.TRACK_STATEMENTS_FALSE_INT)
            return;

        synchronized (this) {
            if (resultSets != null)
                resultSets.remove(resultSet);
        }
    }

    /**
     * Close result sets
     */
    protected void closeResultSets() {
        if (lc.getTrackStatements() == BaseWrapperManagedConnectionFactory.TRACK_STATEMENTS_FALSE_INT)
            return;

        synchronized (this) {
            if (resultSets == null)
                return;

            for (Iterator<Map.Entry<WrappedResultSet, Throwable>> i = resultSets.entrySet().iterator(); i.hasNext();) {
                Map.Entry<WrappedResultSet, Throwable> entry = i.next();
                WrappedResultSet resultSet = entry.getKey();

                if (lc.getTrackStatements() == BaseWrapperManagedConnectionFactory.TRACK_STATEMENTS_TRUE_INT) {
                    Throwable stackTrace = entry.getValue();
                    lc.getLogger().warn("Closing a result set you left open! Please close it yourself.", stackTrace);
                }

                try {
                    resultSet.internalClose();
                } catch (Throwable t) {
                    lc.getLogger().warn("Error closing a result set you left open! Please close it yourself.", t);
                }
            }

            resultSets.clear();
        }
    }
}
