/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.as.server.deployment.scanner;

import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import org.jboss.as.controller.OperationFailedException;
import org.jboss.as.controller.client.OperationBuilder;
import org.jboss.as.controller.operations.common.Util;
import org.jboss.as.server.ServerController;
import org.jboss.as.server.deployment.api.ServerDeploymentRepository;
import org.jboss.as.server.deployment.scanner.ExtensibleFilter;
import org.jboss.as.server.deployment.scanner.api.DeploymentScanner;
import org.jboss.dmr.ModelNode;
import org.jboss.dmr.Property;
import org.jboss.logging.Logger;

class FileSystemDeploymentService
implements DeploymentScanner {
    private static final Pattern ARCHIVE_PATTERN = Pattern.compile("^.*\\.[SsWwJjEeRr][Aa][Rr]$");
    private static final Logger log = Logger.getLogger((String)"org.jboss.as.deployment");
    static final String DEPLOYED = ".isdeployed";
    static final String FAILED_DEPLOY = ".faileddeploy";
    static final String DO_DEPLOY = ".dodeploy";
    static final String DEPLOYING = ".deploying";
    static final String UNDEPLOYING = ".undeploying";
    static final String UNDEPLOYED = ".undeployed";
    private File deploymentDir;
    private long scanInterval = 0L;
    private volatile boolean scanEnabled = false;
    private ScheduledFuture<?> scanTask;
    private final Lock scanLock = new ReentrantLock();
    private final Map<String, DeploymentMarker> deployed = new HashMap<String, DeploymentMarker>();
    private final HashSet<String> noticeLogged = new HashSet();
    private final ScheduledExecutorService scheduledExecutor;
    private final ServerController serverController;
    private final ServerDeploymentRepository deploymentRepository;
    private FileFilter filter = new ExtensibleFilter();

    FileSystemDeploymentService(File deploymentDir, long scanInterval, ServerController serverController, ScheduledExecutorService scheduledExecutor, ServerDeploymentRepository deploymentRepository) throws OperationFailedException {
        if (scheduledExecutor == null) {
            throw new IllegalStateException("null scheduled executor");
        }
        if (serverController == null) {
            throw new IllegalStateException("null server controller");
        }
        if (deploymentRepository == null) {
            throw new IllegalStateException("null deployment repository");
        }
        if (deploymentDir == null) {
            throw new IllegalStateException("null deployment dir");
        }
        if (!deploymentDir.exists()) {
            throw new IllegalArgumentException(deploymentDir.getAbsolutePath() + " does not exist");
        }
        if (!deploymentDir.isDirectory()) {
            throw new IllegalArgumentException(deploymentDir.getAbsolutePath() + " is not a directory");
        }
        if (!deploymentDir.canWrite()) {
            throw new IllegalArgumentException(deploymentDir.getAbsolutePath() + " is not writable");
        }
        this.deploymentDir = deploymentDir;
        this.scanInterval = scanInterval;
        this.serverController = serverController;
        this.scheduledExecutor = scheduledExecutor;
        this.deploymentRepository = deploymentRepository;
        this.establishDeployedContentList(deploymentDir);
    }

    @Override
    public boolean isEnabled() {
        return false;
    }

    @Override
    public long getScanInterval() {
        return this.scanInterval;
    }

    @Override
    public synchronized void setScanInterval(long scanInterval) {
        if (scanInterval != this.scanInterval) {
            this.cancelScan();
        }
        this.scanInterval = scanInterval;
        this.startScan();
    }

    public boolean isScanEnabled() {
        return this.scanEnabled;
    }

    @Override
    public synchronized void startScanner() {
        boolean scanEnabled = this.scanEnabled;
        if (scanEnabled) {
            return;
        }
        this.scanEnabled = true;
        this.startScan();
        log.infof("Started %s for directory %s", (Object)this.getClass().getSimpleName(), (Object)this.deploymentDir.getAbsolutePath());
    }

    @Override
    public synchronized void stopScanner() {
        this.scanEnabled = false;
        this.cancelScan();
    }

    private void establishDeployedContentList(File dir) throws OperationFailedException {
        File[] children;
        Set<String> deploymentNames = this.getDeploymentNames();
        for (File child : children = dir.listFiles()) {
            String fileName = child.getName();
            if (child.isDirectory()) {
                if (this.isEEArchive(fileName)) continue;
                this.establishDeployedContentList(child);
                continue;
            }
            if (!fileName.endsWith(DEPLOYED)) continue;
            String deploymentName = fileName.substring(0, fileName.length() - DEPLOYED.length());
            if (deploymentNames.contains(deploymentName)) {
                this.deployed.put(deploymentName, new DeploymentMarker(child.lastModified()));
                continue;
            }
            if (child.delete()) continue;
            log.warnf("Cannot removed extraneous deployment marker file %s", (Object)fileName);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void scan() {
        try {
            this.scanLock.lockInterruptibly();
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            return;
        }
        try {
            if (this.scanEnabled) {
                log.tracef("Scanning directory %s for deployment content changes", (Object)this.deploymentDir.getAbsolutePath());
                ArrayList<ScannerTask> scannerTasks = new ArrayList<ScannerTask>();
                Set<String> registeredDeployments = this.getDeploymentNames();
                HashSet<String> toRemove = new HashSet<String>(this.deployed.keySet());
                this.scanDirectory(this.deploymentDir, scannerTasks, registeredDeployments, toRemove);
                for (String missing : toRemove) {
                    File parent = this.deploymentDir;
                    scannerTasks.add(new UndeployTask(missing, parent));
                }
                if (scannerTasks.size() > 0) {
                    ArrayList<Object> updates = new ArrayList<ModelNode>(scannerTasks.size());
                    for (ScannerTask task : scannerTasks) {
                        ModelNode update = task.getUpdate();
                        if (log.isDebugEnabled()) {
                            log.debugf("Deployment scan of [%s] found update action [%s]", (Object)this.deploymentDir, (Object)update);
                        }
                        updates.add(update);
                    }
                    while (!updates.isEmpty()) {
                        ModelNode composite = this.getCompositeUpdate(updates);
                        ModelNode results = this.serverController.execute(OperationBuilder.Factory.create((ModelNode)composite).build());
                        List resultList = results.get("result").asPropertyList();
                        ArrayList toRetry = new ArrayList();
                        ArrayList<ScannerTask> retryTasks = new ArrayList<ScannerTask>();
                        for (int i = 0; i < resultList.size(); ++i) {
                            ModelNode result = ((Property)resultList.get(i)).getValue();
                            ScannerTask task = (ScannerTask)scannerTasks.get(i);
                            ModelNode outcome = result.get("outcome");
                            if (outcome.isDefined() && "success".equals(outcome.asString())) {
                                task.handleSuccessResult();
                                continue;
                            }
                            if (outcome.isDefined() && "cancelled".equals(outcome.asString())) {
                                toRetry.add(updates.get(i));
                                retryTasks.add(task);
                                continue;
                            }
                            task.handleFailureResult(result);
                        }
                        updates = toRetry;
                        scannerTasks = retryTasks;
                    }
                }
                log.tracef("Scan complete", new Object[0]);
            }
        }
        finally {
            this.scanLock.unlock();
        }
    }

    private void scanDirectory(File directory, List<ScannerTask> events, Set<String> registeredDeployments, Set<String> toRemove) {
        File[] children = directory.listFiles(this.filter);
        if (children == null) {
            return;
        }
        for (File child : children) {
            String deploymentName;
            String fileName = child.getName();
            if (fileName.endsWith(DEPLOYED)) {
                deploymentName = fileName.substring(0, fileName.length() - DEPLOYED.length());
                toRemove.remove(deploymentName);
                if (!this.deployed.containsKey(deploymentName) && !child.delete()) {
                    log.warnf("Cannot removed extraneous deployment marker file %s", (Object)fileName);
                }
                if (this.deployed.get(deploymentName).lastModified == child.lastModified()) continue;
                events.add(new RedeployTask(deploymentName, child.lastModified(), directory));
                continue;
            }
            if (fileName.endsWith(DO_DEPLOY)) {
                deploymentName = fileName.substring(0, fileName.length() - DO_DEPLOY.length());
                File deploymentFile = new File(directory, deploymentName);
                if (!deploymentFile.exists()) {
                    log.warnf("Deployment of '%s' requested, but the deployment is not present", (Object)deploymentFile);
                    if (child.delete()) continue;
                    log.warnf("Cannot removed extraneous deployment marker file %s", (Object)fileName);
                    continue;
                }
                if (registeredDeployments.contains(deploymentName)) {
                    events.add(new ReplaceTask(deploymentName, deploymentFile, child.lastModified()));
                } else {
                    events.add(new DeployTask(deploymentName, deploymentFile, child.lastModified()));
                }
                toRemove.remove(deploymentName);
                continue;
            }
            if (fileName.endsWith(FAILED_DEPLOY)) {
                deploymentName = fileName.substring(0, fileName.length() - FAILED_DEPLOY.length());
                toRemove.remove(deploymentName);
                continue;
            }
            if (!(!this.isEEArchive(fileName) || this.noticeLogged.contains(fileName) || this.deployed.containsKey(fileName) || new File(fileName + DO_DEPLOY).exists() || new File(fileName + FAILED_DEPLOY).exists())) {
                this.noticeLogged.add(fileName);
                log.infof("Found %s in deployment directory. To trigger deployment create a file called %s%s", (Object)fileName, (Object)fileName, (Object)DO_DEPLOY);
                continue;
            }
            if (!child.isDirectory() || this.isEEArchive(fileName)) continue;
            this.scanDirectory(child, events, registeredDeployments, toRemove);
        }
    }

    private boolean isEEArchive(String fileName) {
        return ARCHIVE_PATTERN.matcher(fileName).matches();
    }

    private synchronized void startScan() {
        if (this.scanEnabled) {
            Runnable runnable = new Runnable(){

                @Override
                public void run() {
                    try {
                        FileSystemDeploymentService.this.scan();
                    }
                    catch (Exception e) {
                        log.errorf((Throwable)e, "Scan of %s threw Exception", (Object)FileSystemDeploymentService.this.deploymentDir.getAbsolutePath());
                    }
                }
            };
            this.scanTask = this.scanInterval > 0L ? this.scheduledExecutor.scheduleWithFixedDelay(runnable, 0L, this.scanInterval, TimeUnit.MILLISECONDS) : this.scheduledExecutor.schedule(runnable, this.scanInterval, TimeUnit.MILLISECONDS);
        }
    }

    private void cancelScan() {
        if (this.scanTask != null) {
            this.scanTask.cancel(false);
            this.scanTask = null;
        }
    }

    private Set<String> getDeploymentNames() throws CancellationException {
        ModelNode op = Util.getEmptyOperation((String)"read-children-names", (ModelNode)new ModelNode());
        op.get("child-type").set("deployment");
        ModelNode response = this.serverController.execute(OperationBuilder.Factory.create((ModelNode)op).build());
        ModelNode result = response.get("result");
        HashSet<String> deploymentNames = new HashSet<String>();
        if (result.isDefined()) {
            List deploymentNodes = result.asList();
            for (ModelNode node : deploymentNodes) {
                deploymentNames.add(node.asString());
            }
        }
        return deploymentNames;
    }

    private ModelNode getCompositeUpdate(List<ModelNode> updates) {
        ModelNode op = Util.getEmptyOperation((String)"composite", (ModelNode)new ModelNode());
        ModelNode steps = op.get("steps");
        for (ModelNode update : updates) {
            steps.add(update);
        }
        return op;
    }

    private ModelNode getCompositeUpdate(ModelNode ... updates) {
        ModelNode op = Util.getEmptyOperation((String)"composite", (ModelNode)new ModelNode());
        ModelNode steps = op.get("steps");
        for (ModelNode update : updates) {
            steps.add(update);
        }
        return op;
    }

    private void safeClose(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeFailedMarker(File deploymentFile, ModelNode failureDescription) {
        File deployedMarker;
        File failedMarker = new File(deploymentFile.getParent(), deploymentFile.getName() + FAILED_DEPLOY);
        File deployMarker = new File(deploymentFile.getParent(), deploymentFile.getName() + DO_DEPLOY);
        if (deployMarker.exists() && !deployMarker.delete()) {
            log.warnf("Unable to remove marker file %s", (Object)deployMarker);
        }
        if ((deployedMarker = new File(deploymentFile.getParent(), deploymentFile.getName() + DEPLOYED)).exists() && !deployedMarker.delete()) {
            log.warnf("Unable to remove marker file %s", (Object)deployedMarker);
        }
        FileOutputStream fos = null;
        try {
            failedMarker.createNewFile();
            fos = new FileOutputStream(failedMarker);
            fos.write(failureDescription.asString().getBytes());
            this.safeClose(fos);
        }
        catch (IOException io) {
            try {
                log.errorf((Throwable)io, "Caught exception writing deployment failed marker file %s", (Object)failedMarker.getAbsolutePath());
                this.safeClose(fos);
            }
            catch (Throwable throwable) {
                this.safeClose(fos);
                throw throwable;
            }
        }
    }

    private class DeploymentMarker {
        private long lastModified;

        private DeploymentMarker(long lastModified) {
            this.lastModified = lastModified;
        }
    }

    private final class UndeployTask
    extends ScannerTask {
        private UndeployTask(String deploymentName, File parent) {
            super(deploymentName, parent, FileSystemDeploymentService.UNDEPLOYING);
        }

        @Override
        protected ModelNode getUpdate() {
            ModelNode address = new ModelNode().add("deployment", this.deploymentName);
            ModelNode undeployOp = Util.getEmptyOperation((String)"undeploy", (ModelNode)address);
            ModelNode removeOp = Util.getEmptyOperation((String)"remove", (ModelNode)address);
            return FileSystemDeploymentService.this.getCompositeUpdate(new ModelNode[]{undeployOp, removeOp});
        }

        @Override
        protected void handleSuccessResult() {
            this.removeInProgressMarker();
            File deployedMarker = new File(this.parent, this.deploymentName + FileSystemDeploymentService.UNDEPLOYED);
            this.createMarkerFile(deployedMarker);
            FileSystemDeploymentService.this.deployed.remove(this.deploymentName);
            FileSystemDeploymentService.this.noticeLogged.remove(this.deploymentName);
        }

        @Override
        protected void handleFailureResult(ModelNode result) {
            this.removeInProgressMarker();
        }
    }

    private final class RedeployTask
    extends ScannerTask {
        private final long markerLastModified;

        private RedeployTask(String deploymentName, long markerLastModified, File parent) {
            super(deploymentName, parent, FileSystemDeploymentService.DEPLOYING);
            this.markerLastModified = markerLastModified;
        }

        @Override
        protected ModelNode getUpdate() {
            ModelNode address = new ModelNode().add("deployment", this.deploymentName);
            return Util.getEmptyOperation((String)"redeploy", (ModelNode)address);
        }

        @Override
        protected void handleSuccessResult() {
            this.removeInProgressMarker();
            FileSystemDeploymentService.this.deployed.remove(this.deploymentName);
            FileSystemDeploymentService.this.deployed.put(this.deploymentName, new DeploymentMarker(this.markerLastModified));
        }

        @Override
        protected void handleFailureResult(ModelNode result) {
            this.removeInProgressMarker();
        }
    }

    private final class ReplaceTask
    extends ContentAddingTask {
        private ReplaceTask(String deploymentName, File deploymentFile, long markerTimestamp) {
            super(deploymentName, deploymentFile, markerTimestamp);
        }

        @Override
        protected ModelNode getUpdatesAfterContent(byte[] hash) {
            ModelNode replaceOp = Util.getEmptyOperation((String)"full-replace-deployment", (ModelNode)new ModelNode());
            replaceOp.get("name").set(this.deploymentName);
            replaceOp.get("hash").set(hash);
            return replaceOp;
        }

        @Override
        protected void handleFailureResult(ModelNode result) {
            this.removeInProgressMarker();
            FileSystemDeploymentService.this.writeFailedMarker(this.deploymentFile, result.get("failure-description"));
        }
    }

    private final class DeployTask
    extends ContentAddingTask {
        private DeployTask(String deploymentName, File deploymentFile, long markerTimestamp) {
            super(deploymentName, deploymentFile, markerTimestamp);
        }

        @Override
        protected ModelNode getUpdatesAfterContent(byte[] hash) {
            ModelNode address = new ModelNode().add("deployment", this.deploymentName);
            ModelNode addOp = Util.getEmptyOperation((String)"add", (ModelNode)address);
            addOp.get("hash").set(hash);
            ModelNode deployOp = Util.getEmptyOperation((String)"deploy", (ModelNode)address);
            return FileSystemDeploymentService.this.getCompositeUpdate(new ModelNode[]{addOp, deployOp});
        }

        @Override
        protected void handleFailureResult(ModelNode result) {
            this.removeInProgressMarker();
            FileSystemDeploymentService.this.writeFailedMarker(this.deploymentFile, result.get("failure-description"));
        }
    }

    private abstract class ContentAddingTask
    extends ScannerTask {
        protected final File deploymentFile;
        protected final long doDeployTimestamp;

        protected ContentAddingTask(String deploymentName, File deploymentFile, long markerTimestamp) {
            super(deploymentName, deploymentFile.getParentFile(), FileSystemDeploymentService.DEPLOYING);
            this.deploymentFile = deploymentFile;
            this.doDeployTimestamp = markerTimestamp;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        protected ModelNode getUpdate() {
            byte[] hash = new byte[]{};
            if (this.deploymentFile.isDirectory()) {
                try {
                    hash = FileSystemDeploymentService.this.deploymentRepository.addExternalFileReference(this.deploymentFile);
                }
                catch (IOException e) {
                    log.error((Object)("Failed to add content to deployment repository for [" + this.deploymentName + "]"), (Throwable)e);
                }
                return this.getUpdatesAfterContent(hash);
            }
            FileInputStream inputStream = null;
            try {
                inputStream = new FileInputStream(this.deploymentFile);
                hash = FileSystemDeploymentService.this.deploymentRepository.addDeploymentContent((InputStream)inputStream);
            }
            catch (IOException e) {
                log.error((Object)("Failed to add content to deployment repository for [" + this.deploymentName + "]"), (Throwable)e);
            }
            finally {
                FileSystemDeploymentService.this.safeClose(inputStream);
            }
            return this.getUpdatesAfterContent(hash);
        }

        protected abstract ModelNode getUpdatesAfterContent(byte[] var1);

        @Override
        protected void handleSuccessResult() {
            File failedMarker;
            File doDeployMarker = new File(new File(this.parent), this.deploymentFile.getName() + FileSystemDeploymentService.DO_DEPLOY);
            if (doDeployMarker.lastModified() <= this.doDeployTimestamp && !doDeployMarker.delete()) {
                log.errorf("Failed to delete deployment marker file %s", (Object)doDeployMarker.getAbsolutePath());
            }
            if ((failedMarker = new File(this.deploymentFile.getParent(), this.deploymentFile.getName() + FileSystemDeploymentService.FAILED_DEPLOY)).exists() && !failedMarker.delete()) {
                log.warnf("Unable to remove marker file %s", (Object)failedMarker);
            }
            this.removeInProgressMarker();
            File deployedMarker = new File(this.parent, this.deploymentFile.getName() + FileSystemDeploymentService.DEPLOYED);
            this.createMarkerFile(deployedMarker);
            if (FileSystemDeploymentService.this.deployed.containsKey(this.deploymentName)) {
                FileSystemDeploymentService.this.deployed.remove(this.deploymentName);
            }
            FileSystemDeploymentService.this.deployed.put(this.deploymentName, new DeploymentMarker(deployedMarker.lastModified()));
        }
    }

    private abstract class ScannerTask {
        protected final String deploymentName;
        protected final String parent;
        private final String inProgressMarkerSuffix;

        private ScannerTask(String deploymentName, File parent, String inProgressMarkerSuffix) {
            this.deploymentName = deploymentName;
            this.parent = parent.getAbsolutePath();
            this.inProgressMarkerSuffix = inProgressMarkerSuffix;
            File marker = new File(parent, deploymentName + inProgressMarkerSuffix);
            this.createMarkerFile(marker);
            this.deleteUndeployedMarker();
        }

        protected abstract ModelNode getUpdate();

        protected abstract void handleSuccessResult();

        protected abstract void handleFailureResult(ModelNode var1);

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void createMarkerFile(File marker) {
            FileOutputStream fos = null;
            try {
                marker.createNewFile();
                fos = new FileOutputStream(marker);
                fos.write(this.deploymentName.getBytes());
            }
            catch (IOException io) {
                try {
                    log.errorf((Throwable)io, "Caught exception writing deployment marker file %s", (Object)marker.getAbsolutePath());
                }
                catch (Throwable throwable) {
                    FileSystemDeploymentService.this.safeClose(fos);
                    throw throwable;
                }
                FileSystemDeploymentService.this.safeClose(fos);
            }
            FileSystemDeploymentService.this.safeClose(fos);
        }

        protected void deleteUndeployedMarker() {
            File undeployedMarker = new File(this.parent, this.deploymentName + FileSystemDeploymentService.UNDEPLOYED);
            if (undeployedMarker.exists() && !undeployedMarker.delete()) {
                log.warnf("Unable to remove marker file %s", (Object)undeployedMarker);
            }
        }

        protected void removeInProgressMarker() {
            File marker = new File(new File(this.parent), this.deploymentName + this.inProgressMarkerSuffix);
            if (marker.exists() && !marker.delete()) {
                log.warnf("Cannot delete deployment progress marker file %s", (Object)marker);
            }
        }
    }
}

