/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.osgi.deployment;

import java.util.jar.Manifest;

import org.jboss.as.server.deployment.Attachments;
import org.jboss.as.server.deployment.DeploymentPhaseContext;
import org.jboss.as.server.deployment.DeploymentUnit;
import org.jboss.as.server.deployment.DeploymentUnitProcessingException;
import org.jboss.as.server.deployment.DeploymentUnitProcessor;
import org.jboss.msc.service.ServiceRegistry;
import org.jboss.osgi.deployment.deployer.Deployment;
import org.jboss.osgi.spi.util.BundleInfo;
import org.jboss.osgi.vfs.AbstractVFS;
import org.jboss.vfs.VirtualFile;
import org.osgi.framework.BundleException;

/**
 * Processes deployments that contain a valid OSGi manifest.
 *
 * @author Thomas.Diesler@jboss.com
 * @since 20-Sep-2010
 */
public class OSGiBundleInfoParseProcessor implements DeploymentUnitProcessor {

    @Override
    public void deploy(DeploymentPhaseContext phaseContext) throws DeploymentUnitProcessingException {

        // Check if we already have an OSGi deployment
        final DeploymentUnit deploymentUnit = phaseContext.getDeploymentUnit();
        BundleInfo info = BundleInfoAttachment.getBundleInfo(deploymentUnit);
        if (info != null)
            return;

        // Get the manifest from the deployment's virtual file
        VirtualFile virtualFile = deploymentUnit.getAttachment(Attachments.DEPLOYMENT_ROOT).getRoot();
        Manifest manifest = deploymentUnit.getAttachment(Attachments.OSGI_MANIFEST);
        if (manifest == null)
            return;

        // Construct and attach the {@link BundleInfo}
        try {
            String contextName = deploymentUnit.getName();
            ServiceRegistry serviceRegistry = phaseContext.getServiceRegistry();
            Deployment holderdep = DeploymentHolderService.getDeployment(serviceRegistry, contextName);
            String location = holderdep != null ? holderdep.getLocation() : contextName;
            info = BundleInfo.createBundleInfo(AbstractVFS.adapt(virtualFile), location);
            BundleInfoAttachment.attachBundleInfo(deploymentUnit, info);
        } catch (BundleException ex) {
            throw new DeploymentUnitProcessingException("Cannot create bundle deployment from: " + virtualFile);
        }
    }

    @Override
    public void undeploy(final DeploymentUnit deploymentUnit) {
        BundleInfoAttachment.detachBundleInfo(deploymentUnit);
    }
}
