/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.process;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Executors;
import org.jboss.as.protocol.ProtocolServer;
import org.jboss.logging.MDC;
import org.jboss.logmanager.handlers.ConsoleHandler;

import javax.net.ServerSocketFactory;

import java.util.logging.Handler;
import java.util.logging.Logger;

/**
 * The main entry point for the process manager.
 *
 * @author <a href="mailto:david.lloyd@redhat.com">David M. Lloyd</a>
 */
public final class Main {

    private Main() {
    }

    public static final String SERVER_MANAGER_PROCESS_NAME = "ServerManager";
    public static final String SERVER_MANAGER_MODULE = "org.jboss.as.server-manager";

    public static void main(String[] args) throws IOException {
        MDC.put("process", "process manager");

        String javaHome = System.getProperty("java.home", ".");
        String jvmName = javaHome + "/bin/java";
        String jbossHome = System.getProperty("jboss.home.dir", ".");
        String modulePath = System.getProperty("jboss.module.path", "modules");
        String bootJar = "jboss-modules.jar";
        String logModule = "org.jboss.logmanager";
        String bootModule = SERVER_MANAGER_MODULE;
        String bindAddress = "127.0.0.1";
        int bindPort = 0;

        final List<String> javaOptions = new ArrayList<String>();
        final List<String> smOptions = new ArrayList<String>();

        // logmodule is the same as mine or defaulted
        // target module is always SM
        // -mp is my module path
        // -jar is jboss-modules.jar in jboss-home
        // log config should be fixed loc

        OUT: for (int i = 0; i < args.length; i++) {
            String arg = args[i];
            if ("-jvm".equals(arg)) {
                jvmName = args[++i];
            } else if ("-jboss-home".equals(arg)) {
                jbossHome = args[++i];
            } else if ("-mp".equals(arg)) {
                modulePath = args[++i];
            } else if ("-jar".equals(arg)) {
                bootJar = args[++i];
            } else if ("-logmodule".equals(arg)) {
                logModule = args[++i];
            } else if ("-bind-addr".equals(arg)) {
                bindAddress = args[++i];
            } else if ("-bind-port".equals(arg)) {
                bindPort = Integer.parseInt(args[++i]);
            } else if ("--".equals(arg)) {
                for (i++; i < args.length; i++) {
                    arg = args[i];
                    if ("--".equals(arg)) {
                        for (i++; i < args.length; i++) {
                            arg = args[i];
                            smOptions.add(arg);
                        }
                        break OUT;
                    } else {
                        javaOptions.add(arg);
                    }
                }
                break OUT;
            } else {
                throw new IllegalArgumentException("Bad option: " + arg);
            }
        }

        Handler consoleHandler = null;

        final Logger rootLogger = Logger.getLogger("");
        for (Handler handler : rootLogger.getHandlers()) {
            if (handler instanceof ConsoleHandler) {
                if (consoleHandler != null) {
                    // duplicate handlers
                    rootLogger.removeHandler(handler);
                } else {
                    consoleHandler = handler;
                    ((ConsoleHandler)consoleHandler).setWriter(new SynchronizedWriter(System.out));
                }
            }
        }

        final ProtocolServer.Configuration configuration = new ProtocolServer.Configuration();
        if (bindAddress != null) {
            configuration.setBindAddress(new InetSocketAddress(bindAddress, bindPort));
        } else {
            configuration.setBindAddress(new InetSocketAddress(bindPort));
        }
        // todo better config
        configuration.setBindAddress(new InetSocketAddress(InetAddress.getLocalHost(), 0));
        configuration.setSocketFactory(ServerSocketFactory.getDefault());
        configuration.setThreadFactory(Executors.defaultThreadFactory());
        configuration.setReadExecutor(Executors.newCachedThreadPool());

        final ProcessManager processManager = new ProcessManager(configuration, System.out, System.err);
        final InetSocketAddress boundAddress = processManager.getServer().getBoundAddress();

        final List<String> initialCommand = new ArrayList<String>();
        initialCommand.add(jvmName);
        initialCommand.add("-D" + "jboss.home.dir=" + jbossHome);
        initialCommand.addAll(javaOptions);
        initialCommand.add("-jar");
        initialCommand.add(bootJar);
        initialCommand.add("-mp");
        initialCommand.add(modulePath);
        initialCommand.add("-logmodule");
        initialCommand.add(logModule);
        initialCommand.add(bootModule);
        initialCommand.add(CommandLineConstants.INTERPROCESS_PM_ADDRESS);
        initialCommand.add(boundAddress.getHostName());
        initialCommand.add(CommandLineConstants.INTERPROCESS_PM_PORT);
        initialCommand.add(Integer.toString(boundAddress.getPort()));
        initialCommand.addAll(smOptions);

        processManager.addProcess("Server Manager", initialCommand, Collections.<String, String>emptyMap(), jbossHome, true);
        processManager.startProcess("Server Manager");

        final Thread shutdownThread = new Thread(new Runnable() {
            public void run() {
                processManager.shutdown();
            }
        }, "Shutdown thread");
        shutdownThread.setDaemon(false);
        Runtime.getRuntime().addShutdownHook(shutdownThread);
    }
}
