/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
  *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bootstrap.api.descriptor;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.logging.Logger;

/**
 * {@link BootstrapDescriptor} implementation backed by a {@link URL}
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 */
public class UrlBootstrapDescriptor extends BootstrapDescriptorBase implements BootstrapDescriptor
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Logger
    */
   private static final Logger log = Logger.getLogger(UrlBootstrapDescriptor.class.getName());

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Underlying URL
    */
   private final URL url;

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a new instance with the specified {@link URL} 
    * @param url
    * @throws IllegalArgumentException If either argument is not specified
    */
   public UrlBootstrapDescriptor(final URL url) throws IllegalArgumentException
   {
      // Precondition checks
      if (url == null)
      {
         throw new IllegalArgumentException("url must be specified");
      }

      // Set
      this.url = url;
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Returns the name of the underlying {@link URL}
    * @see org.jboss.bootstrap.api.descriptor.BootstrapDescriptor#getName()
    */
   @Override
   public String getName()
   {
      return url.toExternalForm();
   }

   /**
    * Obtains the contents of the underlying {@link URL}; equivalent to
    * {@link URL#openStream()}
    * @see org.jboss.bootstrap.api.descriptor.BootstrapDescriptor#getContent()
    */
   public InputStream getContent()
   {
      try
      {
         return this.url.openStream();
      }
      catch (final IOException e)
      {
         throw new RuntimeException("Could not obtain content stream from " + url.toExternalForm(), e);
      }
   }
}
