/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.api.config;

import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.jboss.bootstrap.api.descriptor.BootstrapDescriptor;
import org.jboss.bootstrap.api.lifecycle.LifecycleState;

/**
 * ServerConfig
 * 
 * Configuration used in Server creation and initialization.  After the 
 * server has been initialized, configurations become frozen and immutable.
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public interface ServerConfig<T extends ServerConfig<?>>
{

   /**
    * Obtains the ordered {@link BootstrapDescriptor}s used in
    * the server start lifecycle.  If {@link ServerConfig#isFrozen()}
    * is false, this will be a mutable view, allowing the caller to
    * adjust the available bootstraps.  After the configuration is frozen, 
    * this will return an immutable view.
    * @return
    */
   List<BootstrapDescriptor> getBootstrapDescriptors();

   /**
    * Returns an immutable copy of the properties used in configuring the server
    * 
    * @return
    */
   Map<String, String> getProperties();

   /**
    * Returns the configuration property for the specified key, 
    * or null if not present 
    * 
    * @return
    * @param key The key
    * @throws IllegalArgumentException If the key was not specified
    */
   String getProperty(String key) throws IllegalArgumentException;

   /**
    * Sets the property with the specified key to the specified value
    * 
    * @param key
    * @param value
    * @return This configuration
    * @throws IllegalArgumentException If the key was not supplied
    * @throws IllegalStateException If the configuration has been frozen
    */
   T property(String key, String value) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the specified properties upon the configuration
    * 
    * @param properties
    * @return
    * @throws IllegalArgumentException If the properties are null
    * @throws IllegalStateException If the configuration has been frozen
    */
   T properties(Map<String, String> properties) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the specified properties upon the configuration.  For true type safety, 
    * it is recommended to use {@link ServerConfig#properties(Map)} instead.  This
    * is provided for compatibility with {@link System#getProperties()}.
    * 
    * @param properties
    * @return
    * @throws IllegalArgumentException If the properties are null, or if any of the 
    *   properties are not String/String pairs.
    * @throws IllegalStateException If the configuration has been frozen
    */
   T properties(Properties properties) throws IllegalArgumentException, IllegalStateException;

   /**
    * Freezes the configuration, marking it as immutable.  Will typically 
    * be invoked by a server during the {@link LifecycleState#INITIALIZED} lifecycle.  
    * 
    * @throws IllegalStateException If already frozen
    */
   void freeze() throws IllegalStateException;

   /**
    * Returns whether or not this configuration is frozen and immutable
    * 
    * @return
    */
   boolean isFrozen();
}
