/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.base.server;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import org.jboss.bootstrap.spi.Bootstrap;
import org.jboss.bootstrap.spi.config.ConfigurationInitializer;
import org.jboss.bootstrap.spi.config.ConfigurationValidator;
import org.jboss.bootstrap.spi.config.InvalidConfigurationException;
import org.jboss.bootstrap.spi.config.ServerConfig;
import org.jboss.bootstrap.spi.lifecycle.LifecycleState;
import org.jboss.bootstrap.spi.server.Server;
import org.jboss.bootstrap.spi.server.ServerInitializer;
import org.jboss.logging.Logger;

/**
 * AbstractServer
 * 
 * Generic support for 
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public abstract class AbstractServer<K extends Server<K, T>, T extends ServerConfig<T>> implements Server<K, T>
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(AbstractServer.class);

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Current state of the server.  Synchronized on "this".
    */
   private LifecycleState state;

   /** 
    * Underlying configuration.  Must be a Thread-safe implementation
    * as it's exported
    */
   private T configuration;

   /**
    * Validator for the configuration.  Synchronized on "this".  Requires
    * Thread-safe impl (as it's exported)
    */
   private ConfigurationValidator<T> validator;

   /**
    * Initializer for the configuration.  Synchronized on "this".  Requires
    * Thread-safe impl (as it's exported)
    */
   private ConfigurationInitializer<T> configInitializer;

   /**
    * Server initializer.  Synchronized on "this".  Requires
    * Thread-safe impl (as it's exported)
    */
   private ServerInitializer<K, T> serverInitializer;

   /**
    * The list of bootstraps to run upon start, requires Thread-safe impl.
    */
   private final List<Bootstrap> bootstraps = new CopyOnWriteArrayList<Bootstrap>();

   /**
    * The bootstraps that have been started, requires Thread-safe impl.
    */
   private final List<Bootstrap> startedBootstraps = new CopyOnWriteArrayList<Bootstrap>();

   //-------------------------------------------------------------------------------------||
   // Constructors -----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor
    */
   protected AbstractServer()
   {
      this(null);
   }

   /**
    * Constructor
    * 
    * @param configuration The configuration to set
    */
   protected AbstractServer(final T configuration)
   {
      // Check for valid config and set
      this.setConfiguration(configuration);

      // Set properties
      this.setState(LifecycleState.PRE_INIT);
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#getConfiguration()
    */
   public final T getConfiguration()
   {
      return this.configuration;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#setConfiguration(org.jboss.bootstrap.spi.config.ServerConfig)
    */
   public void setConfiguration(final T configuration)
   {
      // Log and set
      if (log.isTraceEnabled())
      {
         log.trace("Set configuration: " + configuration);
      }
      this.configuration = configuration;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#getState()
    */
   public final synchronized LifecycleState getState()
   {
      LifecycleState state = this.state;
      if (state == null)
      {
         throw new IllegalStateException("null state");
      }
      return state;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#shutdown()
    */
   public void shutdown() throws IllegalStateException, Exception
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Received request to shutdown");
      }

      // Ensure running
      LifecycleState required = LifecycleState.STARTED;
      LifecycleState actual = this.getState();
      this.checkState(required, actual);

      // Initiate shutdown sequence
      log.info("Stopping: " + this);
      this.setState(LifecycleState.STOPPING);

      // Shutdown the Bootstraps
      if (log.isTraceEnabled())
      {
         log.trace("Shutting down bootstraps");
      }
      this.shutdownBootstraps();

      // Shutdown
      if (log.isTraceEnabled())
      {
         log.trace("Calling implementation class shutdown...");
      }
      this.doShutdown();

      // Let the initializer clean up
      final ServerInitializer<K, T> serverInitializer = this.getServerInitializer();
      if (serverInitializer != null)
      {
         if (log.isTraceEnabled())
         {
            log.trace("Calling to clean up for shutdown: " + serverInitializer);
         }
         serverInitializer.cleanup(this);
      }

      // Done
      log.info("Stopped: " + this);
      this.setState(LifecycleState.IDLE);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#start()
    */
   public void start() throws IllegalStateException, Exception
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Received request to start");
      }

      // Invoke init() if necessary
      if (this.getState().equals(LifecycleState.PRE_INIT))
      {
         log.debug("Invoking implicit initialization from start()");
         this.initialize();
      }

      // Ensure idle
      final LifecycleState required = LifecycleState.IDLE;
      final LifecycleState actual = this.getState();
      this.checkState(required, actual);

      // Initiate shutdown sequence
      log.info("Starting: " + this);
      this.setState(LifecycleState.STARTING);

      // Start
      if (log.isTraceEnabled())
      {
         log.trace("Entering implementation class start...");
      }
      this.doStart();

      // Run the bootstraps
      if (log.isTraceEnabled())
      {
         log.trace("Starting bootstraps...");
      }
      this.startBootstraps();

      // Done
      log.info("Started: " + this);
      this.setState(LifecycleState.STARTED);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#addBootstrap(org.jboss.bootstrap.spi.Bootstrap)
    */
   public void addBootstrap(final Bootstrap bootstrap) throws IllegalStateException, IllegalArgumentException
   {
      // Precondition check
      if (bootstrap == null)
      {
         throw new IllegalArgumentException("Supplied bootstrap was null");
      }

      // Add
      this.getBootstraps().add(bootstrap);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#removeBootstrap(org.jboss.bootstrap.spi.Bootstrap)
    */
   public void removeBootstrap(final Bootstrap bootstrap) throws IllegalStateException, IllegalArgumentException
   {
      // Remove
      boolean removed = bootstraps.remove(bootstrap);

      // Postcondition check
      if (!removed)
      {
         throw new IllegalArgumentException(
               "Specified bootstrap could not be removed because it is not present in the list");
      }
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#getValidator()
    */
   public ConfigurationValidator<T> getValidator()
   {
      return this.validator;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#initialize()
    */
   public synchronized void initialize() throws IllegalStateException, InvalidConfigurationException
   {
      // Log
      log.debug("Initializing server: " + this);

      /*
       * Precondition checks
       */

      // State must be pre-initialized
      final LifecycleState state = this.getState();
      if (!state.equals(LifecycleState.PRE_INIT))
      {
         throw new IllegalStateException("Cannot initialize an already initialized server, state is: " + state);
      }

      // Config must be in place
      final T config = this.getConfiguration();
      if (config == null)
      {
         throw new IllegalStateException("Configuration must be supplied before server is initialized");
      }

      // If there's a configuration initializer, use it
      final ConfigurationInitializer<T> configInitializer = this.getConfigInitializer();
      if (configInitializer != null)
      {
         if (log.isTraceEnabled())
         {
            log.trace("Performing configuration initialization...");
         }
         configInitializer.initialize(config);
      }
      else
      {
         if (log.isTraceEnabled())
         {
            log.trace("No configuration initializer supplied, skipping");
         }
      }

      // Validate
      if (log.isTraceEnabled())
      {
         log.trace("Validating config...");
      }
      this.validate(config);

      /*
       * If there's an initializer, use it
       */
      final ServerInitializer<K, T> serverInitializer = this.getServerInitializer();
      if (serverInitializer != null)
      {
         serverInitializer.initialize(this);
      }
      else
      {
         log.debug("No initializer defined, skipping initialization of " + this);
      }

      // Freeze config
      config.freeze();

      // Set state
      this.setState(LifecycleState.IDLE);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#getInitializer()
    */
   public synchronized final ServerInitializer<K, T> getServerInitializer()
   {
      return this.serverInitializer;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#setInitializer(org.jboss.bootstrap.spi.server.ServerInitializer)
    */
   public synchronized final void setServerInitializer(final ServerInitializer<K, T> serverInitializer)
         throws IllegalStateException
   {
      // Precondition check
      this.checkState(LifecycleState.PRE_INIT, this.getState());

      this.serverInitializer = serverInitializer;
      log.debug("Set server initializer to " + serverInitializer);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#getConfigInitializer()
    */
   public synchronized final ConfigurationInitializer<T> getConfigInitializer()
   {
      return this.configInitializer;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#setConfigInitializer(org.jboss.bootstrap.spi.config.ConfigurationInitializer)
    */
   public synchronized final void setConfigInitializer(ConfigurationInitializer<T> configInitializer)
   {
      // Precondition check
      this.checkState(LifecycleState.PRE_INIT, this.getState());

      this.configInitializer = configInitializer;
      log.debug("Set config initializer to " + configInitializer);
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.Server#setConfigurationValidator(org.jboss.bootstrap.spi.config.ConfigurationValidator)
    */
   public synchronized final void setValidator(final ConfigurationValidator<T> validator)
   {
      // Precondition check
      this.checkState(LifecycleState.PRE_INIT, this.getState());

      log.debug("Setting validator to: " + validator);
      this.validator = validator;
   }

   //-------------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||
   /**
    * Implementation-specific start hook
    * 
    * @throws Exception
    */
   protected abstract void doStart() throws Exception;

   /**
    * Implementation-specific shutdown hook
    * 
    * @throws Exception
    */
   protected abstract void doShutdown() throws Exception;

   //-------------------------------------------------------------------------------------||
   // Functional Methods -----------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Starts the bootstraps
    * 
    * @throws Exception
    */
   protected void startBootstraps() throws Exception
   {
      // Run the bootstraps
      for (Bootstrap bootstrap : this.getBootstraps())
      {
         // Add in reverse order
         this.getStartedBootstraps().add(0, bootstrap);

         // Start
         bootstrap.start(this);
      }
   }

   /**
    * Shuts down the bootstraps that have been started
    */
   protected void shutdownBootstraps()
   {
      // Initialize
      final List<Bootstrap> startedBootstraps = this.getStartedBootstraps();

      // Signal 
      for (Bootstrap bootstrap : startedBootstraps)
      {
         bootstrap.prepareShutdown(this);
      }

      // Do the bootstraps in reverse order
      for (Bootstrap bootstrap : startedBootstraps)
      {
         try
         {
            bootstrap.shutdown(this);
         }
         catch (Throwable t)
         {
            log.warn("Error shutting down bootstrap: " + bootstrap, t);
         }
      }
   }

   //-------------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Ensures the actual state matches the required, throwing {@link IllegalStateException} 
    * if not
    * 
    * @throws IllegalStateException If the actual state does not match required
    */
   private void checkState(final LifecycleState required, final LifecycleState actual) throws IllegalStateException
   {
      // Check State
      if (required != actual)
      {
         throw new IllegalStateException("Server must be in " + LifecycleState.class.getSimpleName() + " " + required
               + "; is instead: " + actual);
      }
   }

   /**
    * If {@link Server#getValidator()} is non-null, will
    * assert the configuration is valid using the supplied
    * validator
    * 
    * @param configuration
    * @throws InvalidConfigurationException If the configuration is invalid
    * @throws IllegalArgumentException If the confirguation has not been set
    */
   private void validate(T configuration) throws InvalidConfigurationException, IllegalArgumentException
   {
      // Precondition check
      if (configuration == null)
      {
         throw new IllegalArgumentException("Configuration was not specified");
      }

      // Get the validator
      ConfigurationValidator<T> validator = this.getValidator();

      // Is specified, validate
      if (validator != null)
      {
         log.debug("Validating configuration using: " + validator);
         validator.validate(this.getConfiguration());
      }
      else
      {
         if (log.isTraceEnabled())
         {
            log.trace("No validator defined, skipping validation upon configuration");
         }
      }
   }

   //-------------------------------------------------------------------------------------||
   // Accessors / Mutators ---------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * @return the bootstraps
    */
   private List<Bootstrap> getBootstraps()
   {
      return bootstraps;
   }

   /**
    * @return the startedBootstraps
    */
   private List<Bootstrap> getStartedBootstraps()
   {
      return startedBootstraps;
   }

   /**
    * @param state the state to set
    */
   private synchronized final void setState(final LifecycleState state)
   {
      // Log and set
      if (log.isTraceEnabled())
      {
         log.trace("Setting " + LifecycleState.class.getSimpleName() + " to: " + state);
      }
      this.state = state;
   }

}
