/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.base.server;

import java.security.AccessController;
import java.security.PrivilegedAction;

import org.jboss.logging.Logger;


/**
 * SecurityActions
 * 
 * Utility class for security actions, so as to provide
 * some centralization without leaking privileged actions
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
class SecurityActions
{
   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(SecurityActions.class);

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * In place to prevent instanciation 
    */
   private SecurityActions()
   {
   }

   //-------------------------------------------------------------------------------------||
   // Utility Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Sets a system property with specified key and value
    * 
    * @throws IllegalArgumentException If the key was not specified
    */
   static void setSystemProperty(final String key, final String value) throws IllegalArgumentException
   {
      // Precondition check
      ensureSysPropKeySpecified(key);

      // Set
      AccessController.doPrivileged(new PrivilegedAction<Void>()
      {

         public Void run()
         {
            // Set
            System.setProperty(key, value);

            // Log
            if (log.isTraceEnabled())
            {
               log.trace("Set system property \"" + key + "\" > " + value);
            }

            // Return
            return null;
         }

      });
   }

   /**
    * Clears the specified System Property
    * 
    * @throws IllegalArgumentException If the key was not specified
    */
   static void clearSystemProperty(final String key) throws IllegalArgumentException
   {
      // Precondition check
      ensureSysPropKeySpecified(key);

      // Set
      AccessController.doPrivileged(new PrivilegedAction<Void>()
      {

         public Void run()
         {
            // Clear
            String previousValue = System.clearProperty(key);

            // Log
            if (log.isTraceEnabled())
            {
               log.trace("Removed system property \"" + key + "\" which had previous value: " + previousValue);
            }

            // Return
            return null;
         }

      });
   }

   //-------------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Ensures that the specified key is not null
    * 
    * @throws IllegalArgumentException If the specified key was null
    */
   private static void ensureSysPropKeySpecified(String key) throws IllegalArgumentException
   {
      if (key == null)
      {
         throw new IllegalArgumentException("Key for System Property was not specified");
      }
   }
}
