/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.base.server;

import java.net.URL;
import java.util.HashSet;
import java.util.Set;

import org.jboss.bootstrap.spi.config.ServerConfig;
import org.jboss.bootstrap.spi.lifecycle.LifecycleState;
import org.jboss.bootstrap.spi.server.Server;
import org.jboss.bootstrap.spi.server.ServerInitializer;
import org.jboss.logging.Logger;

/**
 * AbstractBasicServerInitializer
 * 
 * Simple server initializer which will default the configuration
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public abstract class AbstractBasicServerInitializer<K extends Server<K, T>, T extends ServerConfig<T>>
      implements
         ServerInitializer<K, T>
{
   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(AbstractBasicServerInitializer.class);

   /**
    * Set of all system properties added during initialization.  Synchronized on "this", 
    * must not be exported.
    */
   private static final Set<String> SYS_PROPS_ADDED = new HashSet<String>();

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.ServerInitializer#initialize(org.jboss.bootstrap.spi.server.Server)
    */
   public synchronized void initialize(final K server) throws IllegalArgumentException, IllegalStateException
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Request to initialize " + server + " using: " + this);
      }

      // Precondition checks
      if (server == null)
      {
         throw new IllegalArgumentException("Server must be specified");
      }

      final LifecycleState state = server.getState();
      final LifecycleState required = LifecycleState.PRE_INIT;
      if (!state.equals(required))
      {
         throw new IllegalStateException(this + " may only be invoked when " + LifecycleState.class.getSimpleName()
               + " is " + required + ", was: " + state);
      }

      // Obtain config
      final T config = server.getConfiguration();

      // Set System Properties
      if (log.isTraceEnabled())
      {
         log.trace("Setting system properties for " + server);
      }
      this.setSystemProperties(config);

      // Log
      log.debug("Initialized " + server + ": " + this);

   }

   /*
    * (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.ServerInitializer#cleanup(org.jboss.bootstrap.spi.server.Server)
    */
   public synchronized void cleanup(final K server) throws IllegalArgumentException, IllegalStateException
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Preparing for shutdown...");
      }

      // Clear all system properties set
      log.debug("Clearing all system properties set in initialization...");
      for (final String sysProp : SYS_PROPS_ADDED)
      {
         SecurityActions.clearSystemProperty(sysProp);
      }

   }

   //-------------------------------------------------------------------------------------||
   // Functional Methods -----------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Sets system properties from the current configuration; may be overridden to set 
    * more properties
    */
   protected void setSystemProperties(final T configuration)
   {
      // Set Properties from Config
      this.setSystemProperty(ServerConfig.PROP_KEY_BOOTSTRAP_HOME_URL, configuration.getBootstrapHome());
      this.setSystemProperty(ServerConfig.PROP_KEY_BOOTSTRAP_URL, configuration.getBootstrapUrl());
      this.setSystemProperty(ServerConfig.PROP_KEY_BOOTSTRAP_NAME, configuration.getBootstrapName());
   }

   /**
    * Sets the specified URL's external form as a system property with the specified key
    * 
    * @param key
    * @param value
    */
   protected final void setSystemProperty(final String key, final URL value)
   {
      final String urlString = value != null ? value.toExternalForm() : "";
      this.setSystemProperty(key, urlString == null ? null : urlString);
   }

   /**
    * Sets the specified Integer as a system property with the specified key
    * 
    * @param key
    * @param value
    */
   protected final void setSystemProperty(final String key, final Integer value)
   {
      final String string = value != null ? value.toString() : "";
      this.setSystemProperty(key, string == null ? null : string);
   }

   /**
    * Sets the specified Boolean as a system property with the specified key
    * 
    * @param key
    * @param value
    */
   protected final void setSystemProperty(final String key, final Boolean value)
   {
      final String string = value != null ? value.toString() : "";
      this.setSystemProperty(key, string == null ? null : string);
   }

   /**
    * Sets the specified system property key to the specified value.  If
    * no value is specified (blank or null) the property will *not* be set.
    * 
    * @param key The non-null key
    * @param value
    */
   protected final void setSystemProperty(final String key, final String value)
   {
      // Precondition check
      assert key != null : "Key for system property was null";

      // Only set non-null values
      if (value == null || "".equals(value))
      {
         if (log.isTraceEnabled())
         {
            log.trace("Not setting system property for key " + key + " as value was blank or null");
         }
         return;
      }

      // Set 
      SecurityActions.setSystemProperty(key, value);

      // Add to Set 
      synchronized (this)
      {
         SYS_PROPS_ADDED.add(key);
      }

      // Log
      if (log.isTraceEnabled())
      {
         log.trace(("Set system property \"" + key + "\" to: ") + (value != null ? "\"" + value + "\"" : "[EMPTY]"));
      }
   }
}
