/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.base.config;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;

import org.jboss.bootstrap.spi.config.ConfigurationInitializer;
import org.jboss.bootstrap.spi.config.InvalidConfigurationException;
import org.jboss.bootstrap.spi.config.ServerConfig;
import org.jboss.logging.Logger;

/**
 * AbstractBasicConfigurationInitializer
 * 
 * Base for initializing a configuration before
 * its validated and used to run a server
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public abstract class AbstractBasicConfigurationInitializer<T extends ServerConfig<T>>
      implements
         ConfigurationInitializer<T>
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(AbstractBasicConfigurationInitializer.class);

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.config.ConfigurationInitializer#initialize(org.jboss.bootstrap.spi.config.ServerConfig)
    */
   public void initialize(T config) throws InvalidConfigurationException, IllegalArgumentException,
         IllegalStateException
   {
      // Internally initialize to prevent recursion leaking into subclass implementations
      this._initialize(config);
   }

   /**
    * Internal implementation of initialize so as to not invoke subclass
    * implementations upon recursion
    * 
    * @param config
    * @throws InvalidConfigurationException
    * @throws IllegalArgumentException
    * @throws IllegalStateException
    * @see org.jboss.bootstrap.spi.config.ConfigurationInitializer#initialize(org.jboss.bootstrap.spi.config.ServerConfig)
    */
   private void _initialize(T config) throws InvalidConfigurationException, IllegalArgumentException,
         IllegalStateException
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Initializing: " + config);
      }

      // Apply overrides
      this.applyPropertyOverrides(config);

      /*
       * Default the config
       */

      final URL bootstrapUrl = config.getBootstrapUrl();
      final URL home = config.getBootstrapHome();
      final String name = config.getBootstrapName();

      /*
       * If bootstrapURL is not directly defined, construct it from
       * home+name (assuming they both *are* defined)
       */

      if (bootstrapUrl == null && (home != null && name != null))
      {
         // Log
         if (log.isTraceEnabled())
         {
            log.trace("No bootstrap URL defined, constructing it from home and name...");
         }

         // Construct
         URL newBootstrapUrl = null;
         try
         {
            newBootstrapUrl = new URL(home, name);
         }
         catch (MalformedURLException e)
         {
            throw new IllegalArgumentException(new InvalidConfigurationException(
                  "Cannot construct bootstrapURL from home and name", e));
         }

         // Log
         log.debug("New bootstrap URL: " + newBootstrapUrl.toExternalForm());

         // Set
         config.bootstrapUrl(newBootstrapUrl);
      }

      /*
       * If bootstrapConf was not initialized, set it from either home or URL base
       */

      // If there's no name, default it, then run this config again to 
      // default other properties if necessary
      if (name == null || name.length() == 0)
      {
         config.bootstrapName(this.getDefaultBootstrapName(config));
         this._initialize(config);
      }

      // If there's no home, default it, then run this config again
      if (home == null)
      {
         final URL newHome = this.getDefaultBootstrapHome(config);
         config.bootstrapHome(newHome);
         this._initialize(config);
      }
   }

   //-------------------------------------------------------------------------------------||
   // Functional Methods -----------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Returns the default bootstrap name
    * @param config A configuration which may be used in determining the default value
    * @return
    */
   protected String getDefaultBootstrapName(final T config)
   {
      return ConfigurationInitializer.DEFAULT_VALUE_BOOTSTRAP_NAME;
   }

   /**
    * Returns the default bootstrap home
    * @param config A configuration which may be used in determining the default value
    * @return
    */
   protected URL getDefaultBootstrapHome(final T config)
   {
      return SecurityActions.getCodeSourceLocation(this.getClass());
   }

   /**
    * Sets the values of contracted system/config properties upon the supplied configuration
    * 
    * @throws IllegalArgumentException If the config was not specified
    */
   protected void applyPropertyOverrides(final T config) throws IllegalArgumentException
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Applying system property overrides...");
      }

      // Precondition check
      if (config == null)
      {
         throw new IllegalArgumentException("Supplied configuration is null");
      }

      // Get Properties
      final Map<String, String> properties = config.getProperties();

      /*
       * Apply overrides if present
       */

      // Home
      final String overrideHome = this.getOverrideValue(ServerConfig.PROP_KEY_BOOTSTRAP_HOME_URL, properties);
      if (overrideHome != null)
      {
         config.bootstrapHome(overrideHome);
      }

      // URL
      final String overrideUrl = this.getOverrideValue(ServerConfig.PROP_KEY_BOOTSTRAP_URL, properties);
      if (overrideUrl != null)
      {
         config.bootstrapUrl(overrideUrl);
      }

      // Name
      final String overrideName = this.getOverrideValue(ServerConfig.PROP_KEY_BOOTSTRAP_NAME, properties);
      if (overrideName != null)
      {
         config.bootstrapName(overrideName);
      }
   }

   //-------------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Returns the override value for the specified property name in the specified 
    * properties, additionally logging if found.  If not found, null will be returned.
    * 
    * @throws IllegalArgumentException If either argument is null
    */
   private String getOverrideValue(final String propertyName, final Map<String, String> properties)
         throws IllegalArgumentException
   {
      // Get the override value
      String override = properties.get(propertyName);

      // Adjust to null if empty
      if (override != null && override.length() == 0)
      {
         override = null;
      }

      // Log if we'll use it
      if (override != null && override.length() > 0)
      {
         log.debug("Got system property " + propertyName + ", so using as override: " + override);
      }

      // Return
      return override;
   }

}
