/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.mc.server;

import java.net.URL;
import java.util.Collections;
import java.util.List;

import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.bootstrap.impl.base.server.AbstractServer;
import org.jboss.bootstrap.impl.base.xml.BootstrapParser;
import org.jboss.bootstrap.impl.mc.deployer.TempBasicXMLDeployer;
import org.jboss.bootstrap.spi.mc.config.MCBasedServerConfig;
import org.jboss.bootstrap.spi.mc.server.MCBasedServer;
import org.jboss.bootstrap.spi.metadata.BootstrapMetaData;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.plugins.bootstrap.basic.BasicBootstrap;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.logging.Logger;

/**
 * AbstractMCServerBase
 * 
 * Microcontainer implementation of a Server.
 *
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 * @author Scott.Stark@jboss.org
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public abstract class AbstractMCServerBase<K extends MCBasedServer<K, T>, T extends MCBasedServerConfig<T>>
      extends
         AbstractServer<K, T> implements MCBasedServer<K, T>
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(AbstractMCServerBase.class);

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /** 
    * The MC Bootstrap 
    */
   private BasicBootstrap bootstrap;

   /**
    * The MC Kernel
    */
   private Kernel kernel;

   /**
    * The XML Deployer
    */
   private TempBasicXMLDeployer kernelDeployer;

   //-------------------------------------------------------------------------------------||
   // Constructors -----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor
    * 
    * Creates the server with no default configuration
    */
   protected AbstractMCServerBase()
   {
      this(null);
   }

   /**
    * Constructor
    * 
    * Creates the server from the specified configuration
    * 
    * @param config
    */
   protected AbstractMCServerBase(final T config)
   {
      // Invoke super
      super(config);
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.mc.server.MCServer#getKernel()
    */
   public Kernel getKernel()
   {
      return this.kernel;
   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.AbstractServer#doShutdown()
    */
   @Override
   protected void doShutdown() throws Exception
   {
      // Shutdown the deployer
      try
      {
         if (kernelDeployer != null)
         {
            kernelDeployer.shutdown();
            kernelDeployer = null;
         }
      }
      catch (Throwable t)
      {
         log.warn("Error stopping xml deployer", t);
      }

      // Shutdown the controller
      try
      {
         Kernel currentKernel = null;
         if (kernel != null)
         {
            currentKernel = kernel;
            kernel = null;
         }
         else if (bootstrap != null)
         {
            currentKernel = bootstrap.getKernel();
         }
         bootstrap = null;

         if (currentKernel != null)
         {
            KernelController controller = currentKernel.getController();
            controller.shutdown();
         }
      }
      catch (Throwable t)
      {
         log.warn("Error stopping xml deployer", t);
      }

   }

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.server.AbstractServer#doStart()
    */
   @Override
   protected void doStart() throws Exception
   {
      // Bootstrap and set the kernel
      final BasicBootstrap bootstrap = new BasicBootstrap();
      bootstrap.run();
      kernel = bootstrap.getKernel();
      this.bootstrap = bootstrap;

      // Register the server implementation
      final String mcServerBindName = "JBossServer";
      final KernelController controller = kernel.getController();
      final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(mcServerBindName, getClass().getName());
      builder.addMethodInstallCallback("addBootstrap");
      builder.addMethodUninstallCallback("removeBootstrap");
      builder.ignoreStart(); // We invoke start manually, don't let MC do it
      try
      {
         log.debug("Installing " + this + " into MC under key: " + mcServerBindName);
         controller.install(builder.getBeanMetaData(), this);
      }
      catch (Throwable t)
      {
         throw new Exception("Error in installing " + mcServerBindName + " during startup", t);
      }

      // Determine the url for the bootstrap
      final T config = this.getConfiguration();
      final URL configURL = config.getBootstrapConfLocation();
      final URL bootstrapUrl = config.getBootstrapUrl();
      log.info("Starting Microcontainer, Main bootstrapURL=" + bootstrapUrl);

      // Parse the bootstrap metadata
      final BootstrapMetaData bootstrapMetaData = BootstrapParser.parse(bootstrapUrl);
      List<String> bootstrapURLs = bootstrapMetaData.getBootstrapURLs();
      if (bootstrapURLs == null)
      {
         bootstrapURLs = Collections.emptyList();
      }
      if (bootstrapURLs.isEmpty())
      {
         log.warn("No bootstrap urls in " + bootstrapUrl);
      }
      log.debug("BootstrapURLs=" + bootstrapURLs);
      log.debug("Bootstrap URLs are relative to: " + configURL);

      // Create an xml deployer
      kernelDeployer = new TempBasicXMLDeployer(kernel);
      try
      {
         // Deploy the bootstrap urls
         for (String bootstrapURL : bootstrapURLs)
         {
            log.debug("BEFORE: " + bootstrapURL);
            final URL suburl = new URL(configURL, bootstrapURL);
            log.debug("Deploying bootstrap xml: " + suburl);
            kernelDeployer.deploy(suburl);
         }

         // Check it is complete
         kernelDeployer.validate();
      }
      catch (Throwable t)
      {
         try
         {
            kernelDeployer.shutdown();
         }
         catch (Throwable ignored)
         {
         }
         throw new Exception("Encountered exception in server startup", t);
      }

   }

   //-------------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

}
