/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.mc.server;

import org.jboss.bootstrap.api.mc.config.MCServerConfig;
import org.jboss.bootstrap.api.mc.server.MCServer;
import org.jboss.bootstrap.impl.mc.config.BasicMCServerConfig;
import org.jboss.bootstrap.impl.mc.config.MCConfigurationInitializer;
import org.jboss.bootstrap.impl.mc.config.MCConfigurationValidator;
import org.jboss.bootstrap.spi.mc.server.MCServerProvider;
import org.jboss.logging.Logger;

/**
 * MCServerImpl
 * 
 * Concrete implementation of a basic Microcontainer-baked
 * Server
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class MCServerImpl extends AbstractMCServerBase<MCServer, MCServerConfig> implements MCServerProvider
{
   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(MCServerImpl.class);

   /**
    * Config initializer instance
    */
   private static final MCConfigurationInitializer CONFIG_INITIALIZER = new MCConfigurationInitializer();

   /**
    * Config validator instance
    */
   private static final MCConfigurationValidator VALIDATOR = new MCConfigurationValidator();

   /**
    * Server Initializer instance
    */
   private static final MCServerInitializer SERVER_INITIALIZER = new MCServerInitializer();

   //-------------------------------------------------------------------------------------||
   // Constructors -----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor
    * 
    * Creates the server with no initial configuration
    */
   public MCServerImpl()
   {
      this(null);
   }

   /**
    * Creates the server with the specified initial configuration, 
    * defaulting the config initializer, server initializer, and 
    * config validator if not already set
    * 
    * @param config
    */
   public MCServerImpl(final MCServerConfig config)
   {
      this(config, true);
   }

   /**
    * Creates the server with the specified initial configuration, 
    * optionally defaulting the config initializer, server initializer, and 
    * config validator if not already set
    * 
    * @param config
    * @param defaultInit Whether to default the config initializer, 
    * server initializer, and config validator if not already set
    */
   public MCServerImpl(final MCServerConfig config, boolean defaultInit)
   {
      super(MCServer.class, config);

      if (defaultInit)
      {
         synchronized (this)
         {
            // Set properties if not already set
            if (this.getConfigInitializer() == null)
            {
               this.setConfigInitializer(CONFIG_INITIALIZER);
               log.debug("Defaulted: " + CONFIG_INITIALIZER);
            }
            if (this.getValidator() == null)
            {
               this.setValidator(VALIDATOR);
               log.debug("Defaulted: " + VALIDATOR);
            }
            if (this.getServerInitializer() == null)
            {
               this.setServerInitializer(SERVER_INITIALIZER);
               log.debug("Defaulted: " + SERVER_INITIALIZER);
            }
         }
      }
      else
      {
         if (log.isTraceEnabled())
         {
            log
                  .trace("defaultInit is not set, skipping defaulting of server initializer, config initializer, server validator");
         }
      }
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.impl.base.server.AbstractServer#getDefaultServerConfigClass()
    */
   @Override
   protected Class<? extends MCServerConfig> getDefaultServerConfigClass()
   {
      return BasicMCServerConfig.class;
   }
}
