/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.spi.as.config;

import org.jboss.bootstrap.spi.config.ServerConfig;
import org.jboss.bootstrap.spi.factory.ServerConfigFactory;

/**
 * JBossASServerConfigFactory
 * 
 * Factory of {@link JBossASServerConfig} implementations
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class JBossASServerConfigFactory
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * The FQN of the default implementation class to create
    */
   public static final String DEFAULT_AS_SERVER_CONFIG_IMPL_CLASS_NAME = "org.jboss.bootstrap.impl.as.config.BasicJBossASServerConfig";

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor, should not be called
    */
   private JBossASServerConfigFactory()
   {
      // No external access
   }

   //-------------------------------------------------------------------------------------||
   // Factory Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a JBossASServerConfig from the default implementation class name
    * {@link JBossASServerConfigFactory#DEFAULT_AS_SERVER_CONFIG_IMPL_CLASS_NAME}, using the 
    * Thread Context ClassLoader.  The returned config will additionally be created using
    * the TCCL. 
    * 
    * @return The newly-created config
    */
   public static JBossASServerConfig createServerConfig()
   {
      return createServerConfig(SecurityActions.getTccl());
   }

   /**
    * Creates a JBossASServerConfig from the default implementation class name
    * {@link JBossASServerConfigFactory#DEFAULT_AS_SERVER_CONFIG_IMPL_CLASS_NAME}, using the
    * specified ClassLoader.  The returned config will additionally be created using
    * the ClassLoader denoted. 
    * 
    * @throws IllegalArgumentException If the ClassLoader is null
    * @return The newly-created config
    */
   public static JBossASServerConfig createServerConfig(final ClassLoader cl) throws IllegalArgumentException
   {
      // Precondition check
      if (cl == null)
      {
         throw new IllegalArgumentException(ClassLoader.class.getSimpleName() + " is required.");
      }

      // Get the config
      ServerConfig<?> config = null;
      try
      {
         config = ServerConfigFactory.createServerConfig(DEFAULT_AS_SERVER_CONFIG_IMPL_CLASS_NAME, cl);
      }
      catch (Exception e)
      {
         // An exception here is likely our fault, so throw a RuntimeException
         throw new RuntimeException("Error in creating the configuration", e);
      }

      // Cast
      return JBossASServerConfig.class.cast(config);
   }
}
