/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.spi.mc.server;

import org.jboss.bootstrap.spi.factory.ServerFactory;
import org.jboss.bootstrap.spi.mc.config.MCServerConfig;
import org.jboss.bootstrap.spi.mc.config.MCServerConfigFactory;
import org.jboss.bootstrap.spi.server.Server;

/**
 * MCServerFactory
 * 
 * Factory of {@link MCServer} implementations
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class MCServerFactory
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * The FQN of the default server implementation class to create
    */
   public static final String DEFAULT_MC_SERVER_IMPL_CLASS_NAME = "org.jboss.bootstrap.impl.mc.server.MCServerImpl";

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private MCServerFactory()
   {
      // No external access
   }

   //-------------------------------------------------------------------------------------||
   // Factory Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates an MCServer from the default implementation class name
    * {@link MCServerFactory#DEFAULT_MC_SERVER_IMPL_CLASS_NAME}, using the 
    * Thread Context ClassLoader.  The returned server will additionally be created using
    * the TCCL.
    * 
    * @return The newly-created Server
    */
   public static MCServer createServer()
   {
      return createServer(SecurityActions.getTccl());
   }

   /**
    * Creates an MCServer from the default implementation class name
    * {@link MCServerFactory#DEFAULT_MC_SERVER_IMPL_CLASS_NAME}, using the 
    * Thread Context ClassLoader.  The returned server will additionally be created using
    * the TCCL.  As a convenience, the server will contain a default configuration.
    * 
    * @return The newly-created Server populated with a default configuration
    */
   public static MCServer createServerWithDefaultConfiguration()
   {
      return createServerWithDefaultConfiguration(SecurityActions.getTccl());
   }

   /**
    * Creates an MCServer from the default implementation class name
    * {@link MCServerFactory#DEFAULT_MC_SERVER_IMPL_CLASS_NAME}, using the
    * specified ClassLoader.  The returned server will additionally be created using
    * the ClassLoader denoted.
    * 
    * @throws IllegalArgumentException If the ClassLoader is null
    * @return The newly-created Server 
    */
   public static MCServer createServer(final ClassLoader cl) throws IllegalArgumentException
   {
      try
      {
         return createServer(DEFAULT_MC_SERVER_IMPL_CLASS_NAME, cl);
      }
      catch (Exception e)
      {
         // An exception here is likely our fault, so throw a RuntimeException
         throw new RuntimeException("Error in creating the Server", e);
      }
   }

   /**
    * Creates an MCServer from the default implementation class name
    * {@link MCServerFactory#DEFAULT_MC_SERVER_IMPL_CLASS_NAME}, using the
    * specified ClassLoader.  The returned server will additionally be created using
    * the ClassLoader denoted. As a convenience, the server will contain a 
    * default configuration.
    * 
    * @throws IllegalArgumentException If the ClassLoader is null
    * @return The newly-created Server populated with a default configuration
    */
   public static MCServer createServerWithDefaultConfiguration(final ClassLoader cl) throws IllegalArgumentException
   {
      try
      {
         return createServerWithDefaultConfiguration(DEFAULT_MC_SERVER_IMPL_CLASS_NAME, cl);
      }
      catch (Exception e)
      {
         // An exception here is likely our fault, so throw a RuntimeException
         throw new RuntimeException("Error in creating the Server", e);
      }
   }

   /**
    * Creates an MCServer from the specified implementation class name
    * using the specified ClassLoader.  The returned server will additionally 
    * be created using the ClassLoader denoted. 
    * 
    * @throws IllegalArgumentException If the ClassLoader or server implementation class is null
    * @throws Exception If there was an error in creating the Server
    * @return The newly-created Server
    */
   public static MCServer createServer(final String implClassName, final ClassLoader cl)
         throws IllegalArgumentException, Exception
   {
      // Precondition check
      if (cl == null)
      {
         throw new IllegalArgumentException(ClassLoader.class.getSimpleName() + " is required.");
      }
      if (implClassName == null || implClassName.length() == 0)
      {
         throw new IllegalArgumentException("Implementation class name must be specified");
      }

      // Get the server
      final Server<?, ?> server = ServerFactory.createServer(implClassName, cl);

      // Cast
      MCServer mcServer = null;
      try
      {
         mcServer = MCServer.class.cast(server);
      }
      catch (ClassCastException cce)
      {
         // Catch, explain giving some context, throw new
         throw new ClassCastException("Specified server implementation class, " + implClassName
               + " must be assignable to " + MCServer.class.getName());
      }

      // Set a default config
      final MCServerConfig config = MCServerConfigFactory.createServerConfig(cl);
      mcServer.setConfiguration(config);

      // Return
      return mcServer;
   }

   /**
    * Creates an MCServer from the specified implementation class name
    * using the specified ClassLoader.  The returned server will additionally 
    * be created using the ClassLoader denoted. As a convenience, the server 
    * will contain a default configuration.
    * 
    * @throws IllegalArgumentException If the ClassLoader or server implementation class is null
    * @throws Exception If there was an error in creating the Server
    * @return The newly-created Server populated with a default configuration
    */
   public static MCServer createServerWithDefaultConfiguration(final String implClassName, final ClassLoader cl)
         throws IllegalArgumentException, Exception
   {
      // Make the server, applying a default config
      return applyDefaultConfiguration(createServer(implClassName, cl), cl);
   }

   //-------------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Sets a new, default configuration upon the specified server
    * 
    * @param server The server upon which we'll set a new config
    * @param cl The ClassLoader used to create the new config
    * @return The same server instance passed in, with a set config
    */
   private static MCServer applyDefaultConfiguration(final MCServer server, final ClassLoader cl)
   {
      // Set a default config
      final MCServerConfig config = MCServerConfigFactory.createServerConfig(cl);
      server.setConfiguration(config);
      return server;
   }
}
