/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.spi.config;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;

/**
 * ServerConfig
 * 
 * Configuration used in Server creation and initialization
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public interface ServerConfig<T extends ServerConfig<?>>
{
   //-------------------------------------------------------------------------------------||
   // Constants --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Property denoting the URL pointing to the home of 
    * the bootstrap (from which other properties may be defaulted)
    */
   String PROP_KEY_BOOTSTRAP_HOME_URL = "jboss.bootstrap.home.url";

   /**
    * Property denoting the URL of the configuration home
    */
   String PROP_KEY_BOOTSTRAP_CONF_URL = "jboss.bootstrap.conf.url";

   /**
    * Property denoting the name of the bootstrap configuration
    */
   String PROP_KEY_BOOTSTRAP_NAME = "jboss.bootstrap.name";

   /**
    * Property denoting the URL of the bootstrap configuration 
    */
   String PROP_KEY_BOOTSTRAP_URL = "jboss.bootstrap.url";

   //-------------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Obtains the location of the bootstrap directory.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_HOME_URL}.
    * 
    * @return
    */
   URL getBootstrapHome();

   /**
    * Sets the location of the bootstrap directory.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_HOME_URL}.
    * 
    * @param bootstrapHome
    * @return This configuration
    * @throws IllegalArgumentException If the bootstrap home location was not specified
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapHome(URL bootstrapHome) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the location of the bootstrap directory.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_HOME_URL}.
    * 
    * @param bootstrapHome
    * @return This configuration
    * @throws IllegalArgumentException If the bootstrap home location was not specified, 
    *       or could not be assigned to a URL (ie. {@link MalformedURLException}
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapHome(String bootstrapHome) throws IllegalArgumentException, IllegalStateException;

   /**
    * Obtains the location of the bootstrap file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_URL}.
    * 
    * @return
    */
   URL getBootstrapUrl();

   /**
    * Sets the location of the bootstrap file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_URL}.
    * 
    * @param bootstrapLocation
    * @return This configuration
    * @throws IllegalArgumentException If the location was not specified
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapUrl(URL bootstrapLocation) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the location of the bootstrap file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_URL}.
    * 
    * @param bootstrapUrl
    * @return This configuration
    * @throws IllegalArgumentException If the bootstrap URL location was not specified, 
    *       or could not be assigned to a URL (ie. {@link MalformedURLException}
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapUrl(String bootstrapUrl) throws IllegalArgumentException, IllegalStateException;

   /**
    * Obtains the name of the bootstrap configuration.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_NAME}.
    * 
    * @return
    */
   String getBootstrapName();

   /**
    * Sets the name of the bootstrap configuration file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_NAME}.
    * 
    * @param name
    * @return This configuration
    * @throws IllegalArgumentException If the name was not specified
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapName(String name) throws IllegalArgumentException, IllegalStateException;

   /**
    * Obtains the location of the bootstrap configuration location.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_CONF_URL}.
    * 
    * @return
    */
   URL getBootstrapConfLocation();

   /**
    * Sets the name of the bootstrap configuration location.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_CONF_URL}.
    * 
    * @param confLocation
    * @return This configuration
    * @throws IllegalArgumentException If the location was not specified
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapConfLocation(URL confLocation) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the location of the bootstrap file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_CONF_URL}.
    * 
    * @param confLocation
    * @return This configuration
    * @throws IllegalArgumentException If the location was not specified, 
    *       or could not be assigned to a URL (ie. {@link MalformedURLException}
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapConfLocation(String confLocation) throws IllegalArgumentException, IllegalStateException;

   /**
    * Returns an immutable copy of the properties used in configuring the server
    * 
    * @return
    */
   Map<String, String> getProperties();

   /**
    * Sets the property with the specified key to the specified value
    * 
    * @param key
    * @param value
    * @return This configuration
    * @throws IllegalArgumentException If the key was not supplied
    * @throws IllegalStateException If the configuration has been frozen
    */
   T property(String key, String value) throws IllegalArgumentException, IllegalStateException;

   /**
    * Freezes the configuration, marking it as immutable.  Will typically 
    * be invoked by a server during the start lifecycle.  
    * 
    * @throws IllegalStateException
    */
   void freeze() throws IllegalStateException;

   /**
    * Returns whether or not this configuration is frozen and immutable
    * 
    * @return
    */
   boolean isFrozen();
}
