/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.spi.config;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.Properties;

/**
 * ServerConfig
 * 
 * Configuration used in Server creation and initialization
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public interface ServerConfig<T extends ServerConfig<?>>
{
   //-------------------------------------------------------------------------------------||
   // Constants --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Property denoting the URL pointing to the home of 
    * the bootstrap (from which other properties may be defaulted)
    */
   String PROP_KEY_BOOTSTRAP_HOME_URL = "jboss.bootstrap.home.url";

   /**
    * Property denoting the name of the bootstrap configuration
    */
   String PROP_KEY_BOOTSTRAP_NAME = "jboss.bootstrap.name";

   /**
    * Property denoting the URL of the bootstrap configuration 
    */
   String PROP_KEY_BOOTSTRAP_URL = "jboss.bootstrap.url";

   //-------------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Obtains the location of the bootstrap directory.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_HOME_URL}.
    * 
    * @return
    */
   URL getBootstrapHome();

   /**
    * Sets the location of the bootstrap directory.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_HOME_URL}.
    * 
    * @param bootstrapHome
    * @return This configuration
    * @throws IllegalArgumentException If the bootstrap home location was not specified
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapHome(URL bootstrapHome) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the location of the bootstrap directory.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_HOME_URL}.
    * 
    * @param bootstrapHome
    * @return This configuration
    * @throws IllegalArgumentException If the bootstrap home location was not specified, 
    *       or could not be assigned to a URL (ie. {@link MalformedURLException}
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapHome(String bootstrapHome) throws IllegalArgumentException, IllegalStateException;

   /**
    * Obtains the location of the bootstrap file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_URL}.
    * 
    * @return
    */
   URL getBootstrapUrl();

   /**
    * Sets the location of the bootstrap file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_URL}.
    * 
    * @param bootstrapLocation
    * @return This configuration
    * @throws IllegalArgumentException If the location was not specified
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapUrl(URL bootstrapLocation) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the location of the bootstrap file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_URL}.
    * 
    * @param bootstrapUrl
    * @return This configuration
    * @throws IllegalArgumentException If the bootstrap URL location was not specified, 
    *       or could not be assigned to a URL (ie. {@link MalformedURLException}
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapUrl(String bootstrapUrl) throws IllegalArgumentException, IllegalStateException;

   /**
    * Obtains the name of the bootstrap configuration.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_NAME}.
    * 
    * @return
    */
   String getBootstrapName();

   /**
    * Sets the name of the bootstrap configuration file.  Corresponds to 
    * {@link ServerConfig#PROP_KEY_BOOTSTRAP_NAME}.
    * 
    * @param name
    * @return This configuration
    * @throws IllegalArgumentException If the name was not specified
    * @throws IllegalStateException If the configuration has been frozen
    */
   T bootstrapName(String name) throws IllegalArgumentException, IllegalStateException;

   /**
    * Returns an immutable copy of the properties used in configuring the server
    * 
    * @return
    */
   Map<String, String> getProperties();

   /**
    * Returns the configuration property for the specified key, 
    * or null if not present 
    * 
    * @return
    * @param key The key
    * @throws IllegalArgumentException If the key was not specified
    */
   String getProperty(String key) throws IllegalArgumentException;

   /**
    * Sets the property with the specified key to the specified value
    * 
    * @param key
    * @param value
    * @return This configuration
    * @throws IllegalArgumentException If the key was not supplied
    * @throws IllegalStateException If the configuration has been frozen
    */
   T property(String key, String value) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the specified properties upon the configuration
    * 
    * @param properties
    * @return
    * @throws IllegalArgumentException If the properties are null
    * @throws IllegalStateException If the configuration has been frozen
    */
   T properties(Map<String, String> properties) throws IllegalArgumentException, IllegalStateException;

   /**
    * Sets the specified properties upon the configuration.  For true type safety, 
    * it is recommended to use {@link ServerConfig#properties(Map)} instead.  This
    * is provided for compatibility with {@link System#getProperties()}.
    * 
    * @param properties
    * @return
    * @throws IllegalArgumentException If the properties are null, or if any of the 
    *   properties are not String/String pairs.
    * @throws IllegalStateException If the configuration has been frozen
    */
   T properties(Properties properties) throws IllegalArgumentException, IllegalStateException;

   /**
    * Freezes the configuration, marking it as immutable.  Will typically 
    * be invoked by a server during the start lifecycle.  
    * 
    * @throws IllegalStateException
    */
   void freeze() throws IllegalStateException;

   /**
    * Returns whether or not this configuration is frozen and immutable
    * 
    * @return
    */
   boolean isFrozen();
}
