/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.api.runtime;

// $Id: Token.java 3449 2008-12-19 17:39:28Z thomas.diesler@jboss.com $

import java.util.Set;

import javax.management.ObjectName;

import org.jboss.bpm.api.model.Node;
import org.jboss.bpm.api.model.ProcessInstance;
import org.jboss.bpm.api.service.ProcessEngine;

/**
 * A Token is a descriptive construct used to describe how the flow of a Process will proceed at runtime.
 * 
 * By tracking how the Token traverses the Flow Objects, gets diverted through alternative paths, and gets split into
 * parallel paths, the normal Sequence Flow should be completely definable.
 * 
 * A Token will have a unique identity that can be used to separate multiple Tokens that may exist because of concurrent
 * process instances or the splitting of the Token for parallel processing within a single process instance.
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 20-Apr-2007
 */
public interface Token
{
  public enum TokenStatus
  {
    Created, Started, Stoped, Destroyed, Suspended
  }

  /**
   * Get the associated ProcessEngine
   */
  ProcessEngine getProcessEngine();
  
  /**
   * Get the associated {@link ProcessInstance}
   */
  ProcessInstance getProcess();
  
  /**
   * Get the unique token identity
   */
  ObjectName getKey();

  /**
   * Get the token status
   */
  TokenStatus getTokenStatus();
  
  /**
   * Get the associated {@link Attachments}
   */
  Attachments getAttachments();

  /**
   * Get the current {@link Node}
   */
  Node getNode();
  
  /**
   * Get the root token that his Token is associated with 
   */
  Token getRootToken();
  
  /**
   * Get the parent token that his Token is associated with 
   * @return null if this is the root token 
   */
  Token getParentToken();
  
  /**
   * Get the set of child tokens
   * @return An empty set if there are none 
   */
  Set<Token> getChildTokens();
  
  /**
   * Signal the Token, which takes it to the next wait state
   */
  void signal();
  
  /**
   * Signal the Token, which takes it to the next wait state
   * @param name A transition or target name
   */
  void signal(String name);
}