/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.api.test;

import java.net.URL;
import java.util.Hashtable;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jboss.bpm.api.model.ObjectNameFactory;

/**
 * An integration test helper that deals with test deployment/undeployment, etc.
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 14-Oct-2004
 */
public class IntegrationTestHelper extends APITestHelper
{
  private static MBeanServerConnection server;
  private String integrationTarget;

  public void deploy(String archive) throws Exception
  {
    URL url = getTestArchiveFile(archive).toURI().toURL();
    deploy(url);
  }

  public void deploy(URL archive) throws Exception
  {
    getDeployer().deploy(archive);
  }

  public void undeploy(String archive) throws Exception
  {
    URL url = getTestArchiveFile(archive).toURI().toURL();
    undeploy(url);
  }

  public void undeploy(URL archive) throws Exception
  {
    getDeployer().undeploy(archive);
  }

  public boolean isTargetJBoss500()
  {
    String target = getIntegrationTarget();
    return target.startsWith("jboss500");
  }

  public boolean isTargetJBoss423()
  {
    String target = getIntegrationTarget();
    return target.startsWith("jboss423");
  }

  public boolean isTargetJBoss422()
  {
    String target = getIntegrationTarget();
    return target.startsWith("jboss422");
  }

  private String getIntegrationTarget()
  {
    if (integrationTarget == null)
    {
      String jbossVersion;
      try
      {
        ObjectName oname = ObjectNameFactory.create("jboss.system:type=ServerConfig");
        jbossVersion = (String)getServer().getAttribute(oname, "SpecificationVersion");
      }
      catch (Exception ex)
      {
        throw new IllegalStateException("Cannot obtain jboss version", ex);
      }

      if (jbossVersion.startsWith("5.0.0"))
        integrationTarget = "jboss500";
      else if (jbossVersion.startsWith("4.2.3"))
        integrationTarget = "jboss423";
      else if (jbossVersion.startsWith("4.2.2"))
        integrationTarget = "jboss422";
      else
        throw new IllegalStateException("Unsupported jboss version: " + jbossVersion);
    }
    return integrationTarget;
  }

  @SuppressWarnings("unchecked")
  public MBeanServerConnection getServer()
  {
    if (server == null)
    {
      Hashtable jndiEnv = null;
      try
      {
        InitialContext iniCtx = new InitialContext();
        jndiEnv = iniCtx.getEnvironment();
        server = (MBeanServerConnection)iniCtx.lookup("jmx/invoker/RMIAdaptor");
      }
      catch (NamingException ex)
      {
        throw new RuntimeException("Cannot obtain MBeanServerConnection using jndi props: " + jndiEnv, ex);
      }
    }
    return server;
  }

  private ArchiveDeployer getDeployer()
  {
    return new JBossArchiveDeployer(getServer());
  }
}
