/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.incubator.client;

//$Id: UserTaskCallback.java 3450 2008-12-19 17:59:45Z thomas.diesler@jboss.com $

import javax.management.ObjectName;

import org.jboss.bpm.api.model.Message;
import org.jboss.bpm.api.model.MessageListener;
import org.jboss.bpm.api.model.ObjectNameFactory;
import org.jboss.bpm.api.model.builder.MessageBuilder;
import org.jboss.bpm.api.runtime.Attachments;
import org.jboss.bpm.api.runtime.BasicAttachments;
import org.jboss.bpm.api.service.MessageBuilderService;
import org.jboss.bpm.api.service.MessageService;
import org.jboss.bpm.api.service.ProcessEngine;
import org.jboss.bpm.incubator.model.UserTask;

/**
 * A callback that can be attached to a {@link UserTask} to facilitate message handling; 
 * 
 * The callback registers a {@link MessageListener}, extracts the data from the received message 
 * and calls the user provided 'callback' method. The response message is then constructed from 
 * the user provided data and automatically sent back to the {@link UserTask}.
 *  
 * @author thomas.diesler@jboss.com
 * @since 08-Oct-2008
 */
public abstract class UserTaskCallback
{
  private MessageListener messageListener;

  /**
   * Get the associated MessageListener
   */
  public MessageListener getMessageListener()
  {
    return messageListener;
  }

  /**
   * Attached this callback to the user task
   */
  public void attach(UserTask userTask)
  {
    userTask.setUserTaskCallback(this);
    
    messageListener = new CallbackMessageListener(userTask);
    
    ProcessEngine engine = userTask.getProcessDefinition().getProcessEngine();
    MessageService msgService = engine.getService(MessageService.class);
    msgService.addMessageListener(messageListener);
  }

  /**
   * Detach this callback from the user task
   */
  public void detach(UserTask userTask)
  {
    ProcessEngine engine = userTask.getProcessDefinition().getProcessEngine();
    MessageService msgService = engine.getService(MessageService.class);
    msgService.removeMessageListener(messageListener.getKey());
  }
  
  public abstract void callback(Attachments att);

  class CallbackMessageListener implements MessageListener
  {
    private UserTask userTask;
    
    public CallbackMessageListener(UserTask userTask)
    {
      this.userTask = userTask;
    }

    @Override
    public void catchMessage(Message msg)
    {
      // Get the message data
      Attachments att = new BasicAttachments();
      for (String propName : msg.getPropertyNames())
      {
        String value = msg.getProperty(propName).getValue();
        att.addAttachment(propName, value);
      }
        
      // Call the user callback
      callback(att);

      // Build the response message
      Message msgRef = userTask.getInMessageRef();
      MessageBuilder msgBuilder = MessageBuilderService.locateMessageBuilder();
      msgBuilder.newMessage(msgRef.getName());
      for (String propName : msgRef.getPropertyNames())
      {
        Object value = att.getAttachment(propName);
        if (value == null)
          throw new IllegalStateException("Cannot obtain required property: " + propName);
        msgBuilder.addProperty(propName, value);
      }
      Message resMessage = msgBuilder.getMessage();
      
      MessageService msgService = MessageService.locateMessageService();
      
      ObjectName procID = userTask.getProcessInstance().getKey();
      msgService.sendMessage(procID, userTask.getName(), resMessage);
    }

    @Override
    public ObjectName getKey()
    {
      String oname = userTask.getKey().getCanonicalName();
      return ObjectNameFactory.create(oname + ",msgListener=UserTaskCallback");
    }
  }
}