/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client;

import com.google.gwt.http.client.*;
import com.google.gwt.core.client.GWT;
import org.jboss.bpm.console.client.util.ConsoleLog;
import org.jboss.bpm.console.client.model.DTOParser;

import java.util.List;
import java.util.ArrayList;

/**
 * Does HTTP Basic Auth and keeps assigned roles
 * (piggy backed on response) and the username that was used.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class Authentication
{
  private String loginUrl;
  private AuthCallback callback;
  private List<String> rolesAssigned = new ArrayList<String>();
  private String username;

  public Authentication(String loginUrl)
  {
    this.loginUrl = loginUrl;
  }

  public Authentication(String loginUrl, AuthCallback callback)
  {
    this.loginUrl = loginUrl;
    this.callback = callback;
  }

  /**
   * Login, but leave credentials to GWT.
   * This will pop up a username/password dialog netive to the browser.
   */
  public void doLogin()
  {
    doLogin(null, null);
  }

  /**
   * Login using specific credentials.
   * This delegates to {@link com.google.gwt.http.client.RequestBuilder#setUser(String)}
   * and {@link com.google.gwt.http.client.RequestBuilder#setPassword(String)}
   */
  public void doLogin(String user, String pass)
  {

    RequestBuilder rb = new RequestBuilder(RequestBuilder.GET, loginUrl);
    this.username = user;
    // discard password

    ConsoleLog.debug("Authentication against: " + rb.getUrl());

    if (user != null && pass != null)
    {
      rb.setUser(user);
      rb.setPassword(pass);

      if (!GWT.isScript()) // hosted mode only
      {
        rb.setHeader("xtest-user", user);
        rb.setHeader("xtest-pass", pass); // NOTE: This is plaintext, use for testing only
      }
    }

    try
    {
      rb.sendRequest(null,
          new RequestCallback()
          {

            public void onResponseReceived(Request request, Response response)
            {
              // parse roles
              if (200 == response.getStatusCode())
              {
                rolesAssigned = DTOParser.parseRolesAssigned(response.getText());
                if (callback != null) callback.onLoginSuccess(request, response);
              }
              else
              {
                onError(request, new Exception(response.getText()));
              }
            }

            public void onError(Request request, Throwable t)
            {
              // auth failed
              // Couldn't connect to server (could be timeout, SOP violation, etc.)
              if (callback != null)
                callback.onLoginFailed(request, t);
              else
                throw new RuntimeException("Unknown exception upon login attempt", t);
            }
          });
    }

    catch (RequestException e1)
    {
      // Couldn't connect to server
      throw new RuntimeException("Unknown error upon login attempt", e1);
    }
  }


  public void setCallback(AuthCallback callback)
  {
    this.callback = callback;
  }

  public native void doLogout() /*-{
       $wnd.location.reload();
     }-*/;


  public interface AuthCallback
  {
    void onLoginSuccess(Request request, Response response);

    void onLoginFailed(Request request, Throwable t);
  }


  public List<String> getRolesAssigned()
  {
    return rolesAssigned;
  }

  public String getUsername()
  {
    return username;
  }
}
